#pragma once

#include <drive/backend/abstract/base.h>
#include <drive/backend/common/attributed.h>
#include <drive/backend/common/host_filter.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/expected.h>

#include <util/datetime/base.h>

class IRTBackgroundProcessState;
class TRTBackgroundProcessContainer;

using TExpectedState = TExpected<TAtomicSharedPtr<IRTBackgroundProcessState>, TString>;

class IRTBackgroundProcess : public TAttributedEntity<TAttributedEntityDefaultFieldNames> {
public:
    R_FIELD(TString, Description);
    R_FIELD(TString, RTProcessName);
    R_FIELD(bool, Enabled, false);
    R_READONLY(TTimeRestrictionsPool<class TTimeRestriction>, TimeRestrictions);
    R_FIELD(THostFilter, HostFilter);
    R_READONLY(TDuration, Freshness, TDuration::Minutes(1));
    R_READONLY(TSet<TString>, Owners, {});

protected:
    mutable TInstant DataActuality;
    mutable TInstant StartInstant;

public:
    class TExecutionContext {
    private:
        const IServerBase& BaseServer;

    public:
        TExecutionContext(const TExecutionContext& parent)
            : BaseServer(parent.BaseServer)
        {
        }

        virtual ~TExecutionContext() = default;

        TExecutionContext(const IServerBase& server)
            : BaseServer(server)
        {
        }

        const IServerBase& GetServer() const {
            return BaseServer;
        }

        template <class T>
        const T& GetServerAs() const {
            return dynamic_cast<const T&>(BaseServer);
        }
    };

protected:
    virtual NDrive::TScheme DoGetScheme(const IServerBase& /*server*/) const {
        return {};
    }
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const = 0;
    virtual NJson::TJsonValue DoSerializeToJson() const {
        return NJson::JSON_MAP;
    }
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& /*jsonInfo*/) {
        return true;
    }

    virtual bool DoStart(const TRTBackgroundProcessContainer& container);

public:
    virtual bool IsSimultaneousProcess() const {
        return false;
    }

    bool IsHostAvailable(const IServerBase& server) const {
        return HostFilter.CheckHost(server);
    }

    virtual bool Start(const TRTBackgroundProcessContainer& container) final {
        return DoStart(container);
    }

    virtual TString GetType() const = 0;

    IRTBackgroundProcess() = default;
    virtual ~IRTBackgroundProcess() = default;

    using TFactory = NObjectFactory::TParametrizedObjectFactory<IRTBackgroundProcess, TString>;
    using TPtr = TAtomicSharedPtr<IRTBackgroundProcess>;

    NJson::TJsonValue GetReport(bool compact = false) const;
    NDrive::TScheme GetScheme(const IServerBase& server) const;

    bool DeserializeFromJson(const NJson::TJsonValue& jsonSettings);
    NJson::TJsonValue SerializeToJson() const;

    virtual TInstant GetNextStartInstant(const TInstant lastExecution) const = 0;
    virtual TExpectedState Execute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const;
};

class TRTBackgroundProcessContainer {
public:
    using TId = TString;

private:
    R_FIELD(IRTBackgroundProcess::TPtr, ProcessSettings);

    R_FIELD(TString, Name);
    R_READONLY(TString, Id);
    R_READONLY(ui64, Revision, Max<ui64>());

public:
    bool HasRevision() const {
        return Revision != Max<ui64>();
    }

    TMaybe<ui64> OptionalRevision() const {
        return HasRevision() ? Revision : TMaybe<ui64>();
    }

    class TDecoder: public TBaseDecoder {
    public:
        R_FIELD(i32, Id, -1);
        R_FIELD(i32, Revision, -1);
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, Type, -1);
        R_FIELD(i32, Meta, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase, const bool strict = true)
            : TBaseDecoder(strict)
        {
            Id = GetFieldDecodeIndex("bp_id", decoderBase);
            Revision = GetFieldDecodeIndex("bp_revision", decoderBase);
            Name = GetFieldDecodeIndex("bp_name", decoderBase);
            Meta = GetFieldDecodeIndex("bp_settings", decoderBase);
            Type = GetFieldDecodeIndex("bp_type", decoderBase);
        }
    };

    explicit operator bool() const {
        return ProcessSettings != nullptr;
    }

    const IRTBackgroundProcess* operator->() const {
        return ProcessSettings.Get();
    }

    const IRTBackgroundProcess* Get() const {
        return ProcessSettings.Get();
    }

    IRTBackgroundProcess* operator->() {
        return ProcessSettings.Get();
    }

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    TRTBackgroundProcessContainer Clone() const;

    TRTBackgroundProcessContainer() = default;
    TRTBackgroundProcessContainer(IRTBackgroundProcess::TPtr settings)
        : ProcessSettings(settings)
    {
    }

    bool CheckOwner(const TString& userId) const;

    NJson::TJsonValue GetReport(bool serializeSettings = true) const;
    NDrive::TScheme GetScheme(const IServerBase& server) const;
    NStorage::TTableRecord SerializeToTableRecord() const;

    NJson::TJsonValue BuildJsonReport() const {
        return GetReport(/* serializeSettings = */ false);
    }

    static TString GetTableName() {
        return "rt_background_settings";
    }
};

class IRTRegularBackgroundProcess: public IRTBackgroundProcess {
private:
    using TBase = IRTBackgroundProcess;

private:
    R_FIELD(TDuration, Period, TDuration::Max());
    R_FIELD(TSet<ui16>, Timetable);
    R_OPTIONAL(TString, RobotUserId);

protected:
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual NJson::TJsonValue DoSerializeToJson() const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;

public:
    virtual TInstant GetNextStartInstant(const TInstant lastCallInstant) const override;

    virtual TExpectedState Execute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    TString GetRobotId() const;
    TString GetRobotUserId() const;

private:
    mutable TMaybe<TString> CachedRobotUserId;
};
