#include "state.h"

#include <rtline/util/json_processing.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/string_utils/base64/base64.h>

IRTBackgroundProcessState::TFactory::TRegistrator<IRTBackgroundProcessState> IRTBackgroundProcessState::Registrator(IRTBackgroundProcessState::GetTypeName());

NJson::TJsonValue TJsonRTBackgroundProcessState::GetReport() const {
    return SerializeToJson();
}

TBlob TJsonRTBackgroundProcessState::SerializeToBlob() const {
    return TBlob::FromString(SerializeToJson().GetStringRobust());
}

bool TJsonRTBackgroundProcessState::DeserializeFromBlob(const TBlob& data) {
    TStringBuf s(data.AsCharPtr(), data.Size());
    NJson::TJsonValue value;
    if (!NJson::ReadJsonFastTree(s, &value)) {
        ERROR_LOG << GetType() << ": cannot read Json from " << s << Endl;
        return false;
    }
    if (!DeserializeFromJson(value)) {
        ERROR_LOG << GetType() << ": cannot DeserializeFromJson " << value.GetStringRobust() << Endl;
        return false;
    }
    return true;
}

NJson::TJsonValue TRTBackgroundProcessStateContainer::GetReport() const {
    Y_ENSURE_BT(ProcessState);
    NJson::TJsonValue result = ProcessState->GetReport();
    NJson::InsertField(result, "bp_name", ProcessName);
    NJson::InsertField(result, "type", ProcessState->GetType());
    JWRITE_INSTANT(result, "last_execution", LastExecution);
    JWRITE_INSTANT(result, "last_flush", LastFlush);
    NJson::InsertField(result, "status", Status);
    NJson::InsertField(result, "host", HostName);
    NJson::InsertNonNull(result, "message", Message);
    return result;
}

bool TRTBackgroundProcessStateContainer::DeserializeFromTableRecord(const NStorage::TTableRecord& record) {
    ProcessName = record.Get("bp_name");
    if (!ProcessName) {
        ERROR_LOG << "Incorrect background process deserialization for " << record.ToCSV(",") << Endl;
        return false;
    }
    HostName = record.Get("host");
    if (!record.TryGet("bp_last_execution", LastExecution)) {
        ERROR_LOG << "Incorrect bp_last_execution for " << record.ToCSV(",") << Endl;
        return false;
    }

    TString statusData;
    if (!record.TryGet("bp_status", statusData)) {
        ERROR_LOG << "Incorrect bp_status for " << record.ToCSV(",") << Endl;
        return false;
    }

    NJson::TJsonValue statusJson;
    if (!NJson::ReadJsonFastTree(statusData, &statusJson)) {
        ERROR_LOG << "Incorrect status json for " << record.ToCSV(",") << Endl;
        return false;
    }
    if (statusJson.IsString()) {
        Status = statusJson.GetString();
    } else {
        JREAD_INSTANT_OPT(statusJson, "last_flush", LastFlush);
        JREAD_STRING_OPT(statusJson, "status", Status);
        JREAD_STRING_OPT(statusJson, "host", HostName);
        if (!NJson::ParseField(statusJson["message"], Message)) {
            return false;
        }
    }

    ProcessState = IRTBackgroundProcessState::TFactory::Construct(record.Get("bp_type"), "default");
    if (!ProcessState) {
        return false;
    }
    return ProcessState->DeserializeFromBlob(TBlob::FromString(Base64Decode(record.Get("bp_state"))));
}

NStorage::TTableRecord TRTBackgroundProcessStateContainer::SerializeToTableRecord() const {
    CHECK_WITH_LOG(!!ProcessState);
    NStorage::TTableRecord result;
    result.Set("bp_name", ProcessName);
    result.Set("bp_type", ProcessState->GetType());
    result.Set("bp_last_execution", LastExecution);
    {
        NJson::TJsonValue statusData;
        statusData.InsertValue("message", Message);
        statusData.InsertValue("status", Status);
        statusData.InsertValue("host", HostName);
        statusData.InsertValue("last_flush", LastFlush.Seconds());
        result.Set("bp_status", statusData.GetStringRobust());
    }
    const TBlob data = ProcessState->SerializeToBlob();
    result.Set("bp_state", Base64Encode(TStringBuf(data.AsCharPtr(), data.Size())));
    return result;
}
