#pragma once

#include <drive/library/cpp/scheme/scheme.h>

#include <library/cpp/json/writer/json_value.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/object_factory/object_factory.h>

#include <rtline/library/storage/structured.h>
#include <rtline/util/types/accessor.h>

#include <util/generic/ptr.h>
#include <util/memory/blob.h>

class IRTBackgroundProcessState {
public:
    using TPtr = TAtomicSharedPtr<IRTBackgroundProcessState>;
    using TFactory = NObjectFactory::TObjectFactory<IRTBackgroundProcessState, TString>;
private:
    static TFactory::TRegistrator<IRTBackgroundProcessState> Registrator;
protected:
    virtual NDrive::TScheme DoGetScheme() const {
        return NDrive::TScheme();
    }
public:
    IRTBackgroundProcessState() = default;
    virtual ~IRTBackgroundProcessState() = default;

    virtual NDrive::TScheme GetScheme() const final {
        return DoGetScheme();
    }

    virtual NJson::TJsonValue GetReport() const {
        return NJson::JSON_MAP;
    }

    virtual TString GetType() const {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "default";
    }

    virtual TBlob SerializeToBlob() const {
        return TBlob();
    }

    virtual bool DeserializeFromBlob(const TBlob& /*data*/) {
        return true;
    }
};

class TJsonRTBackgroundProcessState: public IRTBackgroundProcessState {
protected:
    virtual NJson::TJsonValue SerializeToJson() const = 0;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& value) = 0;

public:
    virtual NJson::TJsonValue GetReport() const override;
    virtual TBlob SerializeToBlob() const override;
    virtual bool DeserializeFromBlob(const TBlob& data) override;
};

template <class TProto, class TBaseClass = IRTBackgroundProcessState>
class IProtoStateSerializable: public TBaseClass {
protected:
    virtual void SerializeToProto(TProto& proto) const = 0;
    virtual bool DeserializeFromProto(const TProto& proto) = 0;

public:
    using TBaseClass::TBaseClass;

    virtual TBlob SerializeToBlob() const override final {
        TProto proto;
        SerializeToProto(proto);
        return TBlob::FromString(proto.SerializeAsString());
    }

    virtual bool DeserializeFromBlob(const TBlob& data) override final {
        TProto proto;
        if (!proto.ParseFromArray(data.Data(), data.Size())) {
            ERROR_LOG << "Cannot parse data from proto in regular process" << Endl;
            return false;
        }
        return DeserializeFromProto(proto);
    }
};

class TRTBackgroundProcessStateContainer {
private:
    IRTBackgroundProcessState::TPtr ProcessState;
    R_FIELD(TString, ProcessName);
    R_FIELD(TString, HostName);
    R_FIELD(TInstant, LastExecution, Now());
    R_FIELD(TInstant, LastFlush, TInstant::Zero());
    R_FIELD(TString, Status);
    R_FIELD(TString, Message);

public:
    TRTBackgroundProcessStateContainer& SetProcessState(IRTBackgroundProcessState::TPtr state) {
        ProcessState = state;
        return *this;
    }

    IRTBackgroundProcessState::TPtr GetState() const {
        return ProcessState;
    }

    explicit operator bool() const {
        return ProcessState != nullptr;
    }

    const IRTBackgroundProcessState* operator->() const {
        return ProcessState.Get();
    }

    TRTBackgroundProcessStateContainer() = default;
    TRTBackgroundProcessStateContainer(IRTBackgroundProcessState::TPtr state)
        : ProcessState(state)
    {
    }

    NJson::TJsonValue GetReport() const;

    NDrive::TScheme GetScheme() const {
        NDrive::TScheme result;
        if (!!ProcessState) {
            result = ProcessState->GetScheme();
        }
        result.Add<TFSString>("host");
        result.Add<TFSString>("message");
        result.Add<TFSString>("status");
        result.Add<TFSString>("type");
        result.Add<TFSNumeric>("last_execution").SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
        result.Add<TFSNumeric>("last_flush").SetVisual(NDrive::TFSNumeric::EVisualType::DateTime);
        return result;
    }

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& record);
    NStorage::TTableRecord SerializeToTableRecord() const;
};
