#include "process.h"

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/leasing/company.h>
#include <drive/backend/data/leasing/leasing.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/offers/offers/standart.h>

namespace {
    using namespace NDrivematics;
}

NThreading::TFuture<TTaxiDriverStatusClient::TRecognizeStatuses> TBeginEndTaxiSessionsProcess::DoRequests(
      TTaxiFleetVehiclesClient::TRequestParams&& params, auto& eventLogState
    , auto& TaxiFleetVehiclesClient
    , auto& TaxiDriverProfilesClient
    , auto& TaxiDriverStatusClient
    , auto& parkIdCarIdToCarNumber
    , auto& parkIdDriverIdToCarNumber
    , auto& driverIdToDriverInfo) const {
    return TaxiFleetVehiclesClient->GetCarsByNumbers(params, eventLogState)
        .Apply([&parkIdCarIdToCarNumber = parkIdCarIdToCarNumber
                , eventLogState
                , &TaxiDriverProfilesClient]
            (const NThreading::TFuture<TTaxiFleetVehiclesClient::TRecognizedVehicles>& response) {
            TTaxiDriverProfilesClient::TRequestParams driverProfilesRequestParams;
            for (const auto& vehicle : response.GetValue().GetRecognizedVehicle()) {
                driverProfilesRequestParams.MutableParkIdCarIdInSet().insert(vehicle.GetParkIdCarId());
                parkIdCarIdToCarNumber[vehicle.GetParkIdCarId()] = vehicle.GetData().GetNumber();
            }
            driverProfilesRequestParams.MutableProjection().insert({"park_driver_profile_id", "data.full_name", "data.fire_date", "data.hire_date", "data.work_status"});
            return TaxiDriverProfilesClient->GetParkDriverProfileIds(driverProfilesRequestParams, eventLogState);
        })
        .Apply([&ParkIdDriverIdToCarNumber = parkIdDriverIdToCarNumber
                , &parkIdCarIdToCarNumber = parkIdCarIdToCarNumber
                , &driverIdToDriverInfo = driverIdToDriverInfo
                , &TaxiDriverStatusClient
                , eventLogState]
            (const NThreading::TFuture<TTaxiDriverProfilesClient::TRecognizedProfilesByParkIdCarIds>& response) {
            TTaxiDriverStatusClient::TRequestParams driverStatusRequestParams;
            for (const auto& profilesByParkIdCarId : response.GetValue().GetRecognizedProfilesByParkIdCarId()) {
                for (const auto& profile : profilesByParkIdCarId.GetProfiles()) {
                    TTaxiDriverStatusClient::TParkIdDriverId parkIdDriverId({profile.GetParkDriverProfileId().parkId, profile.GetParkDriverProfileId().driverProfileId});
                    driverIdToDriverInfo[profile.GetParkDriverProfileId().driverProfileId] = profile.GetData();
                    driverStatusRequestParams.MutableParkIdDrivedIds().insert(parkIdDriverId);
                    ParkIdDriverIdToCarNumber[parkIdDriverId.ParkId + '_' + parkIdDriverId.DriverId] = parkIdCarIdToCarNumber[profilesByParkIdCarId.GetParkIdCarId()];
                }
            }
            return TaxiDriverStatusClient->GetDriversStatuses(driverStatusRequestParams, eventLogState);
        });
}

bool TBeginEndTaxiSessionsProcess::CloseSession( auto& session, const auto& userId, const auto& server, const auto& carId) const {
    try {
        const auto api = Yensured(server.GetDriveAPI());
        const auto& tagManager = api->GetTagsManager().GetDeviceTags();

        auto permissions = api->GetUserPermissions(userId);
        R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << userId);
        TString sessionId = session->GetSessionId();
        auto tx = tagManager.BuildSession(/*readOnly=*/false, /*repeatableRead=*/true);
        tx.SetOriginatorId(userId);

        R_ENSURE(
            TChargableTag::DirectEvolve(session, TChargableTag::Reservation, *permissions, server, tx, nullptr),
            {},
            "cannot evolve to " << TChargableTag::Reservation << " on carId: " << carId,
            tx
        );
        if (DryRun) {
            Y_UNUSED(tx.Rollback());
            NOTICE_LOG << GetRobotId() << ": dry run 'CloseSession' for sessionId: " << sessionId << " with carId: " << carId << " and userId: " << userId << Endl;
            return true;
        }
        tx.Committed().Subscribe([sessionId, carId, userId](const NThreading::TFuture<void>& c) {
            if (c.HasValue()) {
                NDrive::TEventLog::Log("FinishTaxiSession", NJson::TMapBuilder
                    ("session_id", sessionId)
                    ("car_id", carId)
                    ("user_id", userId)

                );
            }
        });
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit for " << sessionId << ": " << tx.GetStringReport() << Endl;
            return false;
        }
        INFO_LOG << GetRobotId() << ": evolved to 'Reservation' for carId: " << carId << " with session: "  << sessionId << Endl;
        return true;
    } catch (...) {
        ERROR_LOG << GetRobotId() << ": an exception occurred while 'CloseSession' processing: " << CurrentExceptionInfo().GetStringRobust() << Endl;
        return false;
    }
}

bool TBeginEndTaxiSessionsProcess::ActualizeFullName(const TTaxiDriverProfilesClient::TRecognizedData& driverInfo, const auto& userId,  const auto& server) const {
    if (!UpdateTaxiFullName) {
        return true;
    }
    const auto api = Yensured(server.GetDriveAPI());

    auto tx = api->GetUsersData()->template BuildTx<NSQL::Writable>();
    auto userFetchResult = api->GetUsersData()->FetchInfo(userId);
    if (!userFetchResult || userFetchResult.empty()) {
        ERROR_LOG << GetRobotId() << ": no such user: " << userId << Endl;
        return false;
    }
    auto p = userFetchResult.MutableResult().begin();
    auto userData = std::move(p->second);
    const auto& fullName = driverInfo.GetFullName();
    if (userData.GetFirstName() != fullName.FirstName
        || userData.GetLastName() != fullName.LastName
        || userData.GetPName() != fullName.MiddleName
        || userData.GetJoinedAt() != driverInfo.GetHireDate()) {
        userData.SetFirstName(fullName.FirstName);
        userData.SetLastName(fullName.LastName);
        userData.SetPName(fullName.MiddleName);
        userData.SetJoinedAt(driverInfo.GetHireDate());
        if (!api->GetUsersData()->UpdateUser(userData, userId, tx)) {
            ERROR_LOG << GetRobotId() << ": cannot UpdateUser: " << tx.GetStringReport() << Endl;
            return false;
        }
        if (DryRun) {
            bool rolledBack = tx.Rollback();
            Y_UNUSED(rolledBack);
            NOTICE_LOG << GetRobotId() << ": dry run 'ActualizeFullName' for userId: " << userId << Endl;
            return true;
        }
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit UpdateUser, error: " << tx.GetStringReport() << Endl;
            return false;
        }
    }
    return true;
}

bool TBeginEndTaxiSessionsProcess::CheckAndLinkRole(const auto& userId, const auto& server) const {
    const auto api = Yensured(server.GetDriveAPI());
    const auto& tagManager = api->GetTagsManager().GetDeviceTags();

    auto tx = tagManager.BuildSession(/*readOnly=*/false, /*repeatableRead=*/true);
    tx.SetOriginatorId(GetRobotId());

    auto userRoles = api->GetUsersData()->GetRoles().RestoreUserRoles(userId, tx);
    if (!userRoles) {
        NDrive::TEventLog::Log("CannotRestoreUserRoles", NJson::TMapBuilder
            ("session_error", tx.GetReport())
            ("error", "Fail to get taxi user roles")
            ("user_id", userId)
        );
        ERROR_LOG << GetRobotId() << ": error when try to get roles for taxi user: " << userId << Endl;
        return false;
    }
    bool hasNeededRole = false;
    if (!hasNeededRole && userRoles) {
        const auto& roles = userRoles->GetRoles();
        hasNeededRole = std::find_if(roles.begin(), roles.end(), [&NeededUserRole = NeededUserRole](const TUserRole& role) {
            return role.GetRoleId() == NeededUserRole;
        }) != roles.end();
    }
    if (!hasNeededRole) {
        TUserRole role;
        role.SetRoleId(NeededUserRole);
        role.SetUserId(userId);
        role.SetForced(true);
        auto rolesManager = api->GetRolesManager();
        if (!rolesManager->UpsertRoleForUser(role, userId, tx)) {
            ERROR_LOG << GetRobotId() << ": cannot UpsertRoleForUser taxi user: " << userId << Endl;
            return false;
        }
        if (DryRun) {
            Y_UNUSED(tx.Rollback());
            NOTICE_LOG << GetRobotId() << ": dry run 'CheckAndLinkRole' for userId: " << userId << Endl;
            return true;
        }
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit " << tx.GetStringReport() << Endl;
            return false;
        }
    }
    return true;
}

bool TBeginEndTaxiSessionsProcess::CheckDescriptionCompanyTag(const auto& userId, const auto& owningCarTagName, const auto& taxiCompanyDispayName, const auto& server) const {
    const auto api = Yensured(server.GetDriveAPI());
    const auto& tagsManager = api->GetTagsManager();
    auto tx = api->template BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
    const TString tagName = UserAffiliationOrganizationTagNamePrefix + owningCarTagName;

    auto optionalTaggedUser = tagsManager.GetUserTags().RestoreObject(userId, tx);
    if (!optionalTaggedUser) {
        NDrive::TEventLog::Log("CannotRestoreUserTags", NJson::TMapBuilder
            ("session_error", tx.GetReport())
            ("error", "Fail to get user tags")
            ("user_id", userId)
        );
        return false;
    }

    auto userOrganizationAffiliationTag = optionalTaggedUser->template GetFirstTagByClass<TUserOrganizationAffiliationTag>();
    if (!userOrganizationAffiliationTag) {
        auto tagDescriptionsPtr = tagsManager.GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User, { TUserOrganizationAffiliationTag::TypeName });
        auto tagDescIterator = std::find_if(tagDescriptionsPtr.begin(), tagDescriptionsPtr.end(), [owningCarTagName](const auto& element) {
            auto tagDesc = std::dynamic_pointer_cast<const TUserOrganizationAffiliationTag::TDescription>(element.second);
            R_ENSURE(tagDesc, HTTP_INTERNAL_SERVER_ERROR, "cannot cast tag description");
            return tagDesc->GetOwningCarTagName() == owningCarTagName;
        });
        if (tagDescIterator == tagDescriptionsPtr.end()) {
            auto tagDescription = MakeAtomicShared<TUserOrganizationAffiliationTag::TDescription>();
            tagDescription->SetName(tagName);
            tagDescription->SetType(TUserOrganizationAffiliationTag::TypeName);
            tagDescription->SetDisplayName("Тег организации " + taxiCompanyDispayName);
            tagDescription->SetCompanyName(owningCarTagName);
            tagDescription->SetOwningCarTagName(owningCarTagName);
            R_ENSURE(tagsManager.GetTagsMeta().RegisterTag(tagDescription, GetRobotId(), tx), HTTP_INTERNAL_SERVER_ERROR, "cannot register tag ", tx);
            NDrive::TEventLog::Log("CheckDescriptionCompanyTag", NJson::TMapBuilder
                ("event", "AutonomousTagCreate")
                ("tag_name", owningCarTagName)
            );
        }
        if (DryRun) {
            Y_UNUSED(tx.Rollback());
            NOTICE_LOG << GetRobotId() << ": dry run 'CheckDescriptionCompanyTag' for userId: " << userId << Endl;
            return true;
        }
        return tx.Commit();
    }
    return true;
}

bool TBeginEndTaxiSessionsProcess::CheckAndAddOrganizationTag(const auto& userId, const auto& carId, const auto& server) const {
    if (AffiliationOrganizationTagPolicy == EAffiliationOrganizationTagPolicy::Nothing) {
        return true;
    }
    const auto api = Yensured(server.GetDriveAPI());
    const auto& tagsManager = api->GetTagsManager();
    auto permissions = server.GetDriveAPI()->GetUserPermissions(userId);

    auto tx = api->template BuildTx<NSQL::Writable | NSQL::RepeatableRead>();

    auto optionalTaggedCar = tagsManager.GetDeviceTags().RestoreObject(carId, tx);
    if (!optionalTaggedCar) {
        NDrive::TEventLog::Log("CannotRestoreCarTags", NJson::TMapBuilder
            ("session_error", tx.GetReport())
            ("error", "Fail to get car tags")
            ("car_id", carId)
        );
        return false;
    }
    auto taxiCompanyDBTag = optionalTaggedCar->template GetFirstTagByClass<TTaxiCompanyTag>();
    if (!taxiCompanyDBTag) {
        return (!(AffiliationOrganizationTagPolicy == EAffiliationOrganizationTagPolicy::Update || AffiliationOrganizationTagPolicy == EAffiliationOrganizationTagPolicy::All)
            || tagsManager.GetUserTags().RemoveTag(taxiCompanyDBTag, userId, &server, tx))
            && (!DryRun || tx.Commit());
    }
    auto taxiCompanytagDescriptionPtr = tagsManager.GetTagsMeta().GetDescriptionByName(taxiCompanyDBTag->GetName());
    auto taxiCompanytagDescription = dynamic_cast<const TTaxiCompanyTag::TDescription*>(taxiCompanytagDescriptionPtr.Get());
    if (!taxiCompanytagDescription) {
        return false;
    }

    const TString tagName = UserAffiliationOrganizationTagNamePrefix + taxiCompanyDBTag->GetName();

    if (!CheckDescriptionCompanyTag(userId, taxiCompanyDBTag->GetName(), taxiCompanytagDescription->GetName(), server)) {
        return false;
    }

    auto optionalTaggedUser = tagsManager.GetUserTags().RestoreObject(userId, tx);
    if (!optionalTaggedUser) {
        NDrive::TEventLog::Log("CannotRestoreUserTags", NJson::TMapBuilder
            ("session_error", tx.GetReport())
            ("error", "Fail to get user tags")
            ("user_id", userId)
        );
        return false;
    }

    auto userOrganizationAffiliationTagPtr = optionalTaggedUser->template GetFirstTagByClass<TUserOrganizationAffiliationTag>();
    auto affiliatedCompanyTag = userOrganizationAffiliationTagPtr.template GetTagAs<TUserOrganizationAffiliationTag>();
    auto tagDescription = tagsManager.GetTagsMeta().GetDescriptionByName(tagName);
    auto affiliatedCompanyTagDesc = dynamic_cast<const TUserOrganizationAffiliationTag::TDescription*>(tagDescription.Get());
    if (!affiliatedCompanyTagDesc) {
        return false;
    }

    if ((AffiliationOrganizationTagPolicy == EAffiliationOrganizationTagPolicy::All && (!affiliatedCompanyTag || affiliatedCompanyTagDesc->GetOwningCarTagName() != taxiCompanyDBTag->GetName()))
        || (AffiliationOrganizationTagPolicy == EAffiliationOrganizationTagPolicy::Update && affiliatedCompanyTagDesc->GetOwningCarTagName() != taxiCompanyDBTag->GetName())) {
        ITag::TPtr tagData = api->GetTagsManager().GetTagsMeta().CreateTag(tagName, "", Now());
        {
            auto* tag = dynamic_cast<TUserOrganizationAffiliationTag*>(tagData.Get());
            R_ENSURE(tag, HTTP_INTERNAL_SERVER_ERROR, "can't cast tag as user organization affiliation tag");
            tag->SetRoles({TUserOrganizationAffiliationTag::DriverRoleName});
        }
        R_ENSURE(api->GetTagsManager().GetUserTags().AddTag(tagData, GetRobotId(), userId, &server, tx), {}, "can't add tag", tx);
        NDrive::TEventLog::Log("CheckAndAddOrganizationTag", NJson::TMapBuilder
            ("event", "DriverSwitchTaxiCompany")
            ("user_id", userId)
            ("old_taxi_tag_name", affiliatedCompanyTagDesc->GetOwningCarTagName())
            ("new_taxi_tag_name", taxiCompanyDBTag->GetName())
        );
    }
    if (DryRun) {
        Y_UNUSED(tx.Rollback());
        NOTICE_LOG << GetRobotId() << ": dry run 'CheckAndAddOrganizationTag' for userId: " << userId << Endl;
        return true;
    }
    return tx.Commit();
}

bool TBeginEndTaxiSessionsProcess::StartSession(const TDriverCar& driverCar, const auto& server, const auto& carId, const auto& driverIdToDriverInfo) const {
    try {
        const auto api = Yensured(server.GetDriveAPI());
        const auto& tagManager = api->GetTagsManager().GetDeviceTags();

        auto userId = driverCar.GetOrCreateUserId(server);
        R_ENSURE(!userId.empty(), HTTP_INTERNAL_SERVER_ERROR, "cannot GetOrCreateUserId for: " << driverCar.GetTaxiUserId());
        auto permissions = api->GetUserPermissions(userId);
        R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << userId);

        if (driverIdToDriverInfo.contains(driverCar.DriverId)) {
            if (!ActualizeFullName(driverIdToDriverInfo.at(driverCar.DriverId), userId, server)) {
                ERROR_LOG << GetRobotId() << ": cannot ActualizeFullName taxi user: " << userId << Endl;
            }
        }
        if (!CheckAndLinkRole(userId, server)) {
            ERROR_LOG << GetRobotId() << ": cannot get roles for taxi user: " << userId << Endl;
            return false;
        }
        if (!CheckAndAddOrganizationTag(userId, carId, server)) {
            ERROR_LOG << GetRobotId() << ": cannot update UserOrganizationAffiliationTag user: " << userId << Endl;
        }

        auto tx = tagManager.BuildSession(/*readOnly=*/false, /*repeatableRead=*/true);
        tx.SetOriginatorId(GetRobotId());
        IOffer::TPtr offer;
        {
            TFullPricesContext ctx;
            auto standardOffer = MakeAtomicShared<TStandartOffer>(ctx);
            standardOffer->SetChargableAccounts({ "card" });
            standardOffer->SetName(server.GetSettings().template GetValue<TString>("taxi.taxi_session.offer_name").GetOrElse("taxi_session"));
            standardOffer->SetObjectId(carId);
            standardOffer->SetUserId(userId);
            offer = standardOffer;
        }
        TChargableTag::TBookOptions bookOptions;
        bookOptions.MultiRent = false;
        auto booked = TChargableTag::Book(offer, *permissions, server, tx, bookOptions);
        R_ENSURE(booked, {}, "cannot start session", tx);
        R_ENSURE(
            TChargableTag::DirectEvolve(booked, TChargableTag::Acceptance, *permissions, server, tx, nullptr),
            {},
            "cannot evolve to " << TChargableTag::Acceptance,
            tx
        );
        R_ENSURE(
            TChargableTag::DirectEvolve(booked.GetTagId(), TChargableTag::Riding, *permissions, server, tx, nullptr),
            {},
            "cannot evolve to " << TChargableTag::Riding,
            tx
        );
        if (DryRun) {
            bool rolledBack = tx.Rollback();
            Y_UNUSED(rolledBack);
            NOTICE_LOG << GetRobotId() << ": dry run 'StartSession' for user_id: " << userId << " and carId: " << carId << Endl;
            return true;
        }
        tx.Committed().Subscribe([carId, offer](const NThreading::TFuture<void>& c) {
            if (c.HasValue()) {
                NDrive::TEventLog::Log("StartTaxiSession", NJson::TMapBuilder
                    ("id", carId)
                    ("offer", NJson::ToJson(offer))
                );
            }
        });
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit in StartSession: " << tx.GetStringReport() << Endl;
            return false;
        }
        INFO_LOG << GetRobotId() << ": started session with carId: " << carId <<  " and userId: " << userId << Endl;
        return true;
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": an exception occurred while StartSession processing: " << CurrentExceptionInfo().GetStringRobust() << Endl;
        return false;
    }
}

TExpectedState TBeginEndTaxiSessionsProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& tagManager = api->GetTagsManager().GetDeviceTags();

    try {
        auto taxiSessionTagName = server.GetSettings().template GetValue<TString>("taxi.taxi_session.tag");
        Y_ENSURE(taxiSessionTagName && *taxiSessionTagName, "skipping when trying to get value taxi.taxi_session.tag");
        auto filter = TTagsFilter::BuildFromString(*taxiSessionTagName);
        auto fetchedCarIds = tagManager.PrefilterObjects(filter, nullptr, TInstant::Now());
        Y_ENSURE(fetchedCarIds, "skipping when trying to get cars with TTagsFilter: " << filter.ToString());
        auto fetchedCarsInfo = api->GetCarsData()->FetchInfo(*fetchedCarIds, TInstant::Now());
        if (fetchedCarsInfo.empty()) {
            INFO_LOG << GetRobotId() << "skipping fetchedCarsInfo is empty" << Endl;
            return MakeAtomicShared<IRTBackgroundProcessState>();
        }
        auto TaxiFleetVehiclesClient = server.GetTaxiFleetVehiclesClient();
        Y_ENSURE(TaxiFleetVehiclesClient, "skipping when trying to get TTaxiFleetVehiclesClient");
        auto TaxiDriverProfilesClient = server.GetTaxiDriverProfilesClient();
        Y_ENSURE(TaxiDriverProfilesClient, "skipping when trying to get TTaxiDriverProfilesClient");
        auto TaxiDriverStatusClient = server.GetTaxiDriverStatusClient();
        Y_ENSURE(TaxiDriverStatusClient, "skipping when trying to get TTaxiDriverStatusClient");

        THashMap<TString, TVector<TDriverCar>> numberToDriverCar;
        THashMap<TString, TString> parkIdCarIdToCarNumber;
        THashMap<TString, TString> parkIdDriverIdToCarNumber;
        THashMap<TString, TString> numberToIdCar;
        THashMap<TString, TTaxiDriverProfilesClient::TRecognizedData> driverIdToDriverInfo;

        TVector<NThreading::TFuture<TTaxiDriverStatusClient::TRecognizeStatuses>> resultRecognizeStatusesFuture;
        BatchSize = BatchSize > fetchedCarsInfo.size() ? fetchedCarsInfo.size() : BatchSize;
        auto eventLogState =  MakeAtomicShared<NDrive::TEventLog::TState>(NDrive::TEventLog::CaptureState());
        {
            ui32 batch = 0;
            TInstant checkpointTime = TInstant::Now();
            RPSLimit = RPSLimit ? RPSLimit : 1;
            const double secondsOnRequest = 1.0 / RPSLimit;
            TTaxiFleetVehiclesClient::TRequestParams fleetVehiclesRequestParams;
            for (auto it = fetchedCarsInfo.begin(); it != fetchedCarsInfo.end(); it++) {
                fleetVehiclesRequestParams.MutableNumbers().insert(it->second.GetNumber());
                numberToIdCar[ToUpperUTF8(it->second.GetNumber())] = it->first;
                if (batch == BatchSize || it == prev(fetchedCarsInfo.end())) {
                    fleetVehiclesRequestParams.MutableProjection().insert({"park_id_car_id", "data.number"});
                    resultRecognizeStatusesFuture.push_back(
                        DoRequests(std::move(fleetVehiclesRequestParams)
                        , eventLogState
                        , TaxiFleetVehiclesClient
                        , TaxiDriverProfilesClient
                        , TaxiDriverStatusClient
                        , parkIdCarIdToCarNumber
                        , parkIdDriverIdToCarNumber
                        , driverIdToDriverInfo)
                    );
                    batch = 0;
                    auto duration = TInstant::Now().MicroSeconds() - checkpointTime.MicroSeconds();
                    if (duration < (secondsOnRequest * 1000000)) {
                        Sleep(TDuration::MicroSeconds(secondsOnRequest * 1000000 - duration));
                    }
                    checkpointTime = TInstant::Now();
                }
                batch++;
            }
        }

        for (auto& statusFuture : resultRecognizeStatusesFuture) {
            statusFuture.Wait(WaitForTaxiApi);
            if (statusFuture.HasException() || !statusFuture.HasValue()) {
                ERROR_LOG << GetRobotId() << " skipping when requesting taxi api, error: " << NThreading::GetExceptionMessage(statusFuture) << Endl;
                continue;
            }
            for (const auto& status :  statusFuture.GetValue().GetRecognizedStatuses()) {
                if (parkIdDriverIdToCarNumber.contains(status.GetParkId() + '_' + status.GetDriverId())) {
                    numberToDriverCar[
                        parkIdDriverIdToCarNumber[status.GetParkId() + '_' + status.GetDriverId()]
                    ].push_back(TDriverCar({status.GetDriverId(), status.GetStatus()}));
                }
            }
        }

        auto sessionBuilder = tagManager.GetHistoryManager().GetSessionsBuilder("billing", Now());
        Y_ENSURE(sessionBuilder, "cannot get sessionbuilder");
        for (auto& [number, driverCars] : numberToDriverCar) {
            INFO_LOG << GetRobotId() << ": start process for carId: " << numberToIdCar[number] << " with carNumber: " << number << Endl;
            auto session = sessionBuilder->GetLastObjectSession(numberToIdCar[number]);
            if (!session) {
                INFO_LOG << GetRobotId() << ": not found last session for carId: " << numberToIdCar[number] << " with carNumber: " << number << Endl;
                for (const auto& carDriver : driverCars) {
                    if (carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Online) {
                        INFO_LOG << GetRobotId() << ": userId: " << carDriver.GetOrCreateUserId(server) << " with carId: " << numberToIdCar[number] << " and carNumber: " << number << " has taxi status 'Online', trying to start session" << Endl;
                        if (StartSession(carDriver, server, numberToIdCar[number], driverIdToDriverInfo)) {
                            INFO_LOG << GetRobotId() << ": created first session for carId: " << numberToIdCar[number] << " with carNumber: " << number << Endl;
                            break;
                        }
                    }
                }
            } else if (session->GetCurrentState() == ISession::ECurrentState::Started) {
                INFO_LOG << GetRobotId() << ": last session for carId: " << numberToIdCar[number] << " and carNumber: " << number << " is 'Started', sessionId: " << session->GetSessionId() << Endl;
                auto carUserId = session->GetUserId();
                for (const auto& carDriver : driverCars) {
                    if (carDriver.MatchTaxiUser(carUserId, server)) {
                        INFO_LOG << GetRobotId() << ": find taxi driver with userId: " << carUserId << "in 'Started' session, sessionId: " << session->GetSessionId() << " carId: " << numberToIdCar[number] << " carNumber: " << number << Endl;
                        if (carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Offline
                            || carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Busy ) {
                            INFO_LOG << GetRobotId() << ": taxi userId: " << carUserId << " in sessionId: "<< session->GetSessionId() << " have taxi status 'Offline/Busy', session will be closed" << Endl;
                            if (!CloseSession(session, carUserId, server, numberToIdCar[number])) {
                                continue;
                            }
                            continue;
                        }
                    } else if (carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Online) {
                        INFO_LOG << GetRobotId() << ": trying to reload session witn new taxi user on carId: " << numberToIdCar[number]
                            << " current UserId: " << carUserId << " new driver taxi userId: " << carDriver.GetOrCreateUserId(server) << Endl;
                        if (!CloseSession(session, carUserId, server, numberToIdCar[number])) {
                            ERROR_LOG << GetRobotId() << ": error when reload session with 'CloseSession' session method, carId: " << numberToIdCar[number] << Endl;
                            continue;
                        }
                        if (DryRun) {
                            INFO_LOG << GetRobotId() << ": dry run when reload session trying only close session, carId: " << numberToIdCar[number] << Endl;
                        }
                        if (!StartSession(carDriver, server, numberToIdCar[number], driverIdToDriverInfo)) {
                            ERROR_LOG << GetRobotId() << ": error when reload session with 'StartSession' session method, carId: " << numberToIdCar[number] << Endl;
                            continue;
                        }
                        NDrive::TEventLog::Log("ReloadTaxiSessionWithNewUserSuccess", NJson::TMapBuilder
                            ("car_number", number)
                            ("car_id", numberToIdCar[number])
                            ("current_driver_id", carUserId)
                            ("taxi_user_id", carDriver.GetTaxiUserId())
                        );
                    }
                }
            } else if (session->GetCurrentState() == ISession::ECurrentState::Closed) {
                INFO_LOG << GetRobotId() << ": last session for carId: " << numberToIdCar[number] << " is 'Closed'" << Endl;
                for (const auto& carDriver : driverCars) {
                    if (carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Online) {
                        if (StartSession(carDriver, server, numberToIdCar[number], driverIdToDriverInfo)) {
                            break;
                        }
                    } else if (carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Offline
                        || carDriver.DriverStatus == TTaxiDriverStatusClient::EStatus::Busy) {
                        continue;
                    }
                }
            } else {
                ERROR_LOG << GetRobotId() << ": unrecognized session state: " << session->GetCurrentState() << Endl;
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": an exception occurred while processing, error: " << CurrentExceptionInfo().GetStringRobust() << Endl;
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TBeginEndTaxiSessionsProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSBoolean>("update_taxi_full_name").SetDefault(UpdateTaxiFullName);
    scheme.Add<TFSString>("name_taxi_session_tag").SetDefault(server.GetSettings().template GetValue<TString>("taxi.taxi_session.tag").GetOrElse("Empty GVars taxi.taxi_session.tag")).SetReadOnly(true);
    scheme.Add<TFSString>("offer_name_taxi_session").SetDefault(server.GetSettings().template GetValue<TString>("taxi.taxi_session.offer_name").GetOrElse("taxi_session")).SetReadOnly(true);
    scheme.Add<TFSString>("role_id_for_evolve_taxi_session_tags").SetDefault(NeededUserRole).SetRequired(true);
    scheme.Add<TFSVariants>("user_affiliation_organization_tag_policy").InitVariants<EAffiliationOrganizationTagPolicy>().SetDefault(ToString(AffiliationOrganizationTagPolicy));
    scheme.Add<TFSNumeric>("batch_size").SetDefault(BatchSize);
    scheme.Add<TFSNumeric>("rps_limit").SetDefault(RPSLimit);
    scheme.Add<TFSDuration>("api_taxi_timeout").SetDefault(WaitForTaxiApi);
    return scheme;
}

bool TBeginEndTaxiSessionsProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["update_taxi_full_name"], UpdateTaxiFullName) &&
        NJson::ParseField(value["role_id_for_evolve_taxi_session_tags"], NeededUserRole) &&
        NJson::ParseField(value["user_affiliation_organization_tag_policy"], NJson::Stringify(AffiliationOrganizationTagPolicy)) &&
        NJson::ParseField(value["batch_size"], BatchSize) &&
        NJson::ParseField(value["rps_limit"], RPSLimit) &&
        NJson::ParseField(value["api_taxi_timeout"], WaitForTaxiApi);
}

NJson::TJsonValue TBeginEndTaxiSessionsProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["update_taxi_full_name"] = UpdateTaxiFullName;
    result["role_id_for_evolve_taxi_session_tags"] = NeededUserRole;
    result["user_affiliation_organization_tag_policy"] = ToString(AffiliationOrganizationTagPolicy);
    result["batch_size"] = BatchSize;
    result["rps_limit"] = RPSLimit;
    result["api_taxi_timeout"] = NJson::ToJson(NJson::Hr(WaitForTaxiApi));
    return result;
}

TBeginEndTaxiSessionsProcess::TFactory::TRegistrator<TBeginEndTaxiSessionsProcess> TBeginEndTaxiSessionsProcess::Registrator(TBeginEndTaxiSessionsProcess::GetTypeName());
