#include "process.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/library/json/adapters.h>

TExpectedState TObjectsSelectorWatcherProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const auto api = server.GetDriveAPI();
    const auto actions = Yensured(api)->GetRolesManager()->GetActionsDB().GetActionsPtrWithType<TObjectSelectorUsageAction>();
    if (actions.empty()) {
        return MakeUnexpected<TString>("no actions registered");
    }

    const auto& deviceTagManager = Yensured(api)->GetTagsManager().GetDeviceTags();
    const auto objects = deviceTagManager.GetCachedObjectsVector();

    auto timestamp = Now();
    auto access = TMap<TString, TSet<TString>>();
    for (auto&& action : actions) {
        auto objectsSelector = action;
        if (!objectsSelector) {
            continue;
        }
        auto name = objectsSelector->GetName();
        if (!objectsSelector->IsEnabled()) {
            INFO_LOG << GetRobotId() << ": skip disabled action " << name << Endl;
            continue;
        }
        auto& objectIds = access[name];
        for (auto&& object : objects) {
            auto matched = objectsSelector->IsMatching(object.GetTags());
            if (!matched) {
                continue;
            }

            auto checked = objectsSelector->Check(object.GetId(), timestamp);
            if (!checked) {
                continue;
            }

            objectIds.insert(object.GetId());
        }
    }

    NDrive::TEventLog::Log("ObjectsSelectorVisibility", NJson::TMapBuilder
        ("actions", NJson::ToJson(NJson::Dictionary(access)))
        ("timestamp", timestamp.Seconds())
    );
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TObjectsSelectorWatcherProcess::TFactory::TRegistrator<TObjectsSelectorWatcherProcess> TObjectsSelectorWatcherProcess::Registrator(TObjectsSelectorWatcherProcess::GetTypeName());
