#include "processor.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/sessions/manager/billing.h>

#include <drive/library/cpp/mds/client.h>
#include <drive/library/cpp/yt/common/writer.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/yson/node/node_io.h>


namespace {
    constexpr TStringBuf LinksPrefix = "https://carsharing-acceptances.s3.yandex.net/";

    class TCollector {
    private:
        R_FIELD(TString, Path);
        R_FIELD(TVector<TString>, Result);
        R_FIELD(bool, HasMore, false);
        R_FIELD(ui32, Tries, 0);

    public:
        TCollector(const TString& path, bool hasMore)
            : Path(path)
            , HasMore(hasMore)
        {
        }
    };

    NJson::TJsonValue GetLinks(const TVector<TString>& paths, const NS3::TBucket& bucket, ui32 triesLimit) {
        TVector<TCollector> workers;
        Transform(paths.begin(), paths.end(), std::back_inserter(workers), [](const TString& path) { return TCollector(path, true); });
        bool needMore;
        do {
            needMore = false;
            TVector<NThreading::TFuture<NS3::TBucketElementList>> futures;
            TVector<ui32> workerIds;
            for (ui32 i(0); i < workers.size(); i++) {
                auto& worker = workers[i];
                if (!worker.GetHasMore()) {
                    continue;
                }
                TString marker = worker.GetResult() ? worker.GetResult().back() : "";
                futures.emplace_back(bucket.GetKeys(worker.GetPath(), marker));
                workerIds.push_back(i);
            }
            if (auto waiter = NThreading::WaitAll(futures); !waiter.Wait(bucket.GetConfig().GetRequestTimeout())) {
                NDrive::TEventLog::Log("GetOrdersExportLinksWarning", NJson::TMapBuilder
                    ("warning", "requests timeouted")
                );
            }
            for (ui32 i(0); i < Min(futures.size(), workerIds.size()); i++) {
                auto& future = futures[i];
                auto& worker = workers[workerIds[i]];
                if (future.HasValue()) {
                    auto subResult = future.GetValue();
                    worker.SetHasMore(subResult.HasMore && !subResult.List.empty());
                    Transform(subResult.List.begin(), subResult.List.end(), std::back_inserter(worker.MutableResult()), [](const auto& image) { return image.Key; });
                    needMore |= worker.GetHasMore();
                } else {
                    ui32 tryId = worker.GetTries();
                    if (tryId >= triesLimit) {
                        NDrive::TEventLog::Log("OrdersExportLinksError", NJson::TMapBuilder
                            ("path", worker.GetPath())
                            ("error", "stop by limit")
                            ("mds_error", NThreading::GetExceptionMessage(future))
                            ("try_id", tryId)
                        );
                        return NJson::JSON_NULL;
                    } else {
                        NDrive::TEventLog::Log("GetOrdersExportLinksWarning", NJson::TMapBuilder
                            ("warning", "request failed")
                            ("path", worker.GetPath())
                            ("mds_error", NThreading::GetExceptionMessage(future))
                            ("try_id", tryId)
                        );
                        worker.SetTries(tryId + 1);
                    }
                }
            }
        } while (needMore);
        NJson::TJsonValue result(NJson::JSON_ARRAY);
        for (auto&& worker : workers) {
            for (auto&& image : worker.GetResult()) {
                result.AppendValue(TStringBuilder() << LinksPrefix << image);
            }
        }
        return result;
    }
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTOrdersExport> TRTOrdersExport::Registrator(TRTOrdersExport::GetTypeName());
TRTOrdersExportState::TFactory::TRegistrator<TRTOrdersExportState> TRTOrdersExportState::Registrator(TRTOrdersExport::GetTypeName());

TString TRTOrdersExportState::GetType() const {
    return TRTOrdersExport::GetTypeName();
}


TExpectedState TRTOrdersExport::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    THolder<TRTHistoryWatcherState> result = MakeHolder<TRTOrdersExportState>();
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    const TRTHistoryWatcherState* state = dynamic_cast<const TRTHistoryWatcherState*>(stateExt.Get());
    ui64 lastEventId = state ? state->GetLastEventId() : StartFromId;

    const ui64 historyIdCursor = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetLockedMaxEventId();

    if (!server.GetDriveAPI()->HasBillingManager()) {
        return MakeUnexpected<TString>("no billing manager configured");
    }

    if (!server.GetDriveAPI()->HasMDSClient()) {
        return MakeUnexpected<TString>("no MDS configured");
    }

    auto bucket = server.GetDriveAPI()->GetMDSClient().GetBucket("carsharing-acceptances");
    if (!bucket) {
        return MakeUnexpected<TString>("bucket carsharing-acceptances does not exist");
    }

    TInstant startInstant = Now();
    TVector<std::pair<TCommonTagSessionManager::TSessionConstPtr, TCompiledBill>> closedSessions;
    TMap<TString, TPaymentsData> payments;
    {
        auto tx = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto optionalSessions = server.GetDriveDatabase().GetSessionManager().GetSessionsActualSinceId({lastEventId, historyIdCursor}, tx);
        if (!optionalSessions) {
            return MakeUnexpected("cannot fetch actual sessions " + tx.GetStringReport());
        }
        auto sessions = std::move(*optionalSessions);
        std::sort(sessions.begin(), sessions.end(), [](TCommonTagSessionManager::TSessionConstPtr a, TCommonTagSessionManager::TSessionConstPtr b) -> bool {
            return a->GetLastEventId() < b->GetLastEventId();
        });
        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "read_records", sessions.size());
        const TBillingManager& billingManager = server.GetDriveAPI()->GetBillingManager();

        TVector<TString> sessionIds;
        for (auto&& bSession : sessions) {
            if (closedSessions.size() >= MaxCount) {
                break;
            }
            if (bSession->GetLastEventId() > historyIdCursor) {
                break;
            }
            if (bSession->GetClosed() && bSession->GetLastTS() + TimeDelay > startInstant) {
                break;
            }

            if (bSession->GetClosed()) {
                auto compiledBill = billingManager.GetCompiledBills().GetFullBillFromDB(bSession->GetSessionId(), tx);
                if (!compiledBill) {
                    TUnistatSignalsCache::SignalAdd("billing-report-undefined-" + GetRTProcessName(), "count", 1);
                    ERROR_LOG << "Undefined billing report for " << bSession->GetSessionId() << " " << tx.GetStringReport() << Endl;
                    return MakeUnexpected("cannot acquire compiled_bills for " + bSession->GetSessionId() + ": " + tx.GetStringReport());
                }
                if (!compiledBill->GetSessionId()) {
                    TUnistatSignalsCache::SignalAdd("billing-report-empty-" + GetRTProcessName(), "count", 1);
                    ERROR_LOG << "Undefined billing report for " << bSession->GetSessionId() << " with empty compilled bill" << Endl;
                    return MakeUnexpected("cannot acquire compiled_bills for " + bSession->GetSessionId() + ": empty compilled bill");
                }
                if (!compiledBill->GetFinal()) {
                    TUnistatSignalsCache::SignalAdd("billing-report-nonfinal-" + GetRTProcessName(), "count", 1);
                    ERROR_LOG << "Undefined billing report for " << bSession->GetSessionId() << " with nonfinal compilled bill" << Endl;
                    return MakeUnexpected("cannot acquire compiled_bills for " + bSession->GetSessionId() + ": nonfinal compilled bill");
                }
                closedSessions.emplace_back(bSession, std::move(*compiledBill));
                sessionIds.emplace_back(bSession->GetSessionId());
            }
        }

        auto finishTasks = billingManager.GetPaymentsManager().GetFinishedPayments(sessionIds, tx);
        if (!finishTasks) {
            return MakeUnexpected("cannot fetch bills " + tx.GetStringReport());
        }
        payments = std::move(*finishTasks);
    }

    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "closed_sessions", closedSessions.size());

    TAtomicSharedPtr<TBillingReportCustomization> billingCustomization = new TBillingReportCustomization;
    billingCustomization->SetNeedBill(false).SetNeedPriceByTags(true);

    ui64 nextReal = lastEventId;
    ui32 sent = 0;
    try {
        auto locale = DefaultLocale;
        auto ytClient = NYT::CreateClient(GetYTCluster());
        TYTWritersSet<TTableSelectorDay> writers(ytClient, GetYTDir());
        for (auto&& [bSession, compiledBill] : closedSessions) {
            TBillingSession::TBillingEventsCompilation evCompilation;
            TBillingSession::TBillingCompilation bCompilation;
            if (bSession->FillCompilation(evCompilation) && bSession->FillCompilation(bCompilation)) {
                TVector<TString> paths;
                {
                    TFsPath photoPath(bSession->GetUserId());
                    photoPath = photoPath / bCompilation.GetSessionId();
                    paths.emplace_back(photoPath.GetPath());
                }
                {
                    TFsPath photoPath(bCompilation.GetSessionId());
                    paths.emplace_back(photoPath.GetPath());
                }
                NJson::TJsonValue links = GetLinks(paths, *bucket, MdsRetries);
                if (links.IsNull()) {
                    break;
                }

                NYT::TNode recordNode;
                recordNode["session_id"] = bCompilation.GetSessionId();
                recordNode["start_ts"] = bSession->GetStartTS().Seconds();
                recordNode["last_ts"] = bSession->GetLastTS().Seconds();
                recordNode["original_price"] = bCompilation.GetReportSumOriginalPrice();
                recordNode["user_id"] = bSession->GetUserId();
                recordNode["object_id"] = bSession->GetObjectId();
                recordNode["mileage"] = evCompilation.GetMileageMax() - evCompilation.GetMileageOnStart();
                auto billIt = payments.find(bCompilation.GetSessionId());
                billingCustomization->OptionalPaymentsData() = (billIt == payments.end()) ? TMaybe<TPaymentsData>() : billIt->second;
                recordNode["billing_details"] = NYT::NodeFromJsonValue(bCompilation.GetReport(locale, &server, billingCustomization));
                if (!!bCompilation.GetCurrentOffer()) {
                    NDrive::NProto::TOffer protoOffer = bCompilation.GetCurrentOffer()->SerializeToProto();
                    recordNode["offer_details"] = NYT::NodeFromJsonValue(NProtobufJson::Proto2Json(protoOffer));
                } else {
                    recordNode["offer_details"] = NYT::NodeFromJsonValue(NJson::JSON_MAP);
                }
                recordNode["events_details"] = NYT::NodeFromJsonValue(evCompilation.GetReport(locale, &server, nullptr));

                NJson::TJsonValue paymentDetails;
                paymentDetails["wallets"] = compiledBill.GetReport();
                recordNode["payment_details"] = NYT::NodeFromJsonValue(paymentDetails);
                recordNode["photos"] = NYT::NodeFromJsonValue(links);
                ++sent;

                writers.GetWriter(bSession->GetLastTS())->AddRow(recordNode);
                nextReal = Max(nextReal, bSession->GetLastEventId());
            }
        }
        writers.Finish();
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        return MakeUnexpected<TString>({});
    }

    TUnistatSignalsCache::SignalAdd("orders_exports-sent-" + GetRTProcessName(), "records", sent);
    result->SetLastEventId(nextReal);
    return result.Release();
}

NDrive::TScheme TRTOrdersExport::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("start_from", "Начать с id").SetDefault(0);
    scheme.Add<TFSString>("yt_cluster", "Кластер YT");
    scheme.Add<TFSString>("yt_dir", "Директория для экспорта");
    scheme.Add<TFSNumeric>("max_count", "Максимальное количество отгружаемых сессий").SetDefault(1000);
    scheme.Add<TFSDuration>("delay", "Отставание в экспорте").SetDefault(TDuration::Minutes(1));
    scheme.Add<TFSNumeric>("mds_retries", "Количество попыток чтения из S3").SetDefault(MdsRetries);
    return scheme;
}

bool TRTOrdersExport::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_UINT_OPT(jsonInfo, "start_from", StartFromId)
    JREAD_STRING(jsonInfo, "yt_cluster", YTCluster);
    JREAD_STRING(jsonInfo, "yt_dir", YTDir);
    JREAD_UINT_OPT(jsonInfo, "max_count", MaxCount);
    JREAD_DURATION_OPT(jsonInfo, "delay", TimeDelay);
    JREAD_UINT_OPT(jsonInfo, "mds_retries", MdsRetries);
    return true;
}

NJson::TJsonValue TRTOrdersExport::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["start_from"] = StartFromId;
    result["yt_dir"] = YTDir;
    result["yt_cluster"] = YTCluster;
    result["max_count"] = MaxCount;
    TJsonProcessor::WriteDurationString(result, "delay", TimeDelay);
    result["mds_retries"] = MdsRetries;
    return result;
}
