#include "config.h"

#include <drive/backend/major/client.h>


TRTInsuranceDateWatcherProcess::TFactory::TRegistrator<TRTInsuranceDateWatcherProcess> TRTInsuranceDateWatcherProcess::Registrator(TRTInsuranceDateWatcherProcess::GetTypeName());

NDrive::TScheme TRTInsuranceDateWatcherProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSDuration>("period_check", "Период до истечения срока страховки").SetDefault(TDuration::Days(14)).SetRequired(true);
    scheme.Add<TFSVariants>("provider", "Идентификатор страховой").InitVariants<TNeedInsuranceUpdateTag::EProvider>().SetRequired(true);
    scheme.Add<TFSBoolean>("auto_approve", "Не ждать подтверждения для обновления");
    TSet<TString> tagNames;
    if (auto impl = server.GetAs<NDrive::IServer>()) {
        tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TNeedInsuranceUpdateTag::TypeName});
    }
    scheme.Add<TFSVariants>("insurance_tag", "Имя тега для обозначения, что страховка скоро закончится").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSVariants>("tags_filter", "Фильтр тегов").SetVariants(tagNames).SetMultiSelect(true);
    return scheme;
}

bool TRTInsuranceDateWatcherProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo["notifier"], NotifierName)
        && NJson::ParseField(jsonInfo["provider"], NJson::Stringify(Provider), true)
        && NJson::ParseField(jsonInfo["period_check"], PeriodCheck, true)
        && NJson::ParseField(jsonInfo["auto_approve"], AutoApprove, true)
        && NJson::ParseField(jsonInfo["insurance_tag"], InsuranceTag, true)
        && NJson::ParseField(jsonInfo["tags_filter"], TagsFilter)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTInsuranceDateWatcherProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "period_check", ToString(PeriodCheck.Seconds()));
    NJson::InsertField(result, "auto_approve", AutoApprove);
    NJson::InsertField(result, "insurance_tag", InsuranceTag);
    NJson::InsertField(result, "tags_filter", TagsFilter);
    NJson::InsertField(result, "provider", NJson::Stringify(Provider));
    NJson::InsertNonNull(result, "notifier", NotifierName);
    return result;
}

TExpectedState TRTInsuranceDateWatcherProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    const auto startProcess = Now();
    TVector<TString> carErrors;
    TVector<TString> carMessages;
    const auto& manager = server.GetDriveAPI()->GetTagsManager();
    ITag::TPtr tag = manager.GetTagsMeta().CreateTag(InsuranceTag);
    if (auto updateTag = std::dynamic_pointer_cast<TNeedInsuranceUpdateTag>(tag)) {
        updateTag->SetProvider(Provider);
        updateTag->SetApproved(AutoApprove);
    } else {
        ERROR_LOG << GetRobotId() << " Тег некорректного типа '" << InsuranceTag << "'" << Endl;
        return nullptr;
    }
    TSet<TString> taggedCars;
    {
        auto session = manager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> tags;
        auto tagNames = MakeVector(TagsFilter);
        tagNames.push_back(InsuranceTag);
        if (!manager.GetDeviceTags().RestoreTags({}, tagNames, tags, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": failed get car tags " + session.GetStringReport());
        }
        Transform(tags.begin(), tags.end(), std::inserter(taggedCars, taggedCars.begin()), [](const TDBTag& dbTag) { return dbTag.GetObjectId(); });
    }
    TVector<TString> carsToTag;
    for (auto&& [id, carData] : context.GetFetchedCarsData()) {
        if (taggedCars.contains(id)) {
            continue;
        }
        {
            TCarGenericAttachment currentRegistryDocument;
            if (!server.GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(id, EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess)) {
                carErrors.emplace_back(carData.GetVin() + ": Нет базовых данных");
                continue;
            }
            auto baseDocument = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
            if (!baseDocument) {
                carErrors.emplace_back(carData.GetVin() + ": Нет базовых данных");
                continue;
            }
            if (baseDocument->GetOsagoDateTo() == TInstant::Zero()) {
                carErrors.emplace_back(carData.GetVin() + ": Нет даты истечения страховки");
                continue;
            }
            if (baseDocument->GetOsagoDateTo() - PeriodCheck > Now()) {
                continue;
            }
        }
        carsToTag.push_back(id);
        carMessages.emplace_back(carData.GetVin() + ": Добавлен тег " + InsuranceTag);
    }
    auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
    for (const auto& id : carsToTag) {
        if (!manager.GetDeviceTags().AddTag(tag, GetRobotUserId(), id, &server, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": failed to add car tag " + session.GetStringReport());
        }
    }
    if (!session.Commit()) {
        return MakeUnexpected<TString>(GetRobotId() + ": failed to commit session " + session.GetStringReport());
    } else if (!carMessages.empty()) {
        if (NotifierName) {
            NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Изменены теги у машин " + ToString(carMessages.size()), carMessages);
        } else {
            NDrive::TEventLog::Log("InsuranceWatcherProcessChanges", NJson::TMapBuilder("Изменены теги у машин " + ToString(carMessages.size()), NJson::ToJson(carMessages)));
        }
    }
    if (!carErrors.empty()) {
        if (NotifierName) {
            NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Ошибки при проверке страховки " + ToString(carErrors.size()), carErrors);
        } else {
            NDrive::TEventLog::Log("InsuranceWatcherProcessErrors", NJson::TMapBuilder("Ошибки при проверке страховки " + ToString(carErrors.size()), NJson::ToJson(carErrors)));
        }
    }
    return new IRTBackgroundProcessState;
}


TRTInsuranceOrderStarterProcess::TFactory::TRegistrator<TRTInsuranceOrderStarterProcess> TRTInsuranceOrderStarterProcess::Registrator(TRTInsuranceOrderStarterProcess::GetTypeName());

NDrive::TScheme TRTInsuranceOrderStarterProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    TSet<TString> tagNames;
    if (auto impl = server.GetAs<NDrive::IServer>()) {
        tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TNeedInsuranceUpdateTag::TypeName});
    }
    scheme.Add<TFSVariants>("insurance_tag", "Имя тега обозначающий необходимость обновления").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSVariants>("waiting_order_tag", "Имя тега для ожидания заказа новой страховки").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSVariants>("waiting_update_tag", "Имя тега для ожидания обновления страховки").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSVariants>("insurance_order_error_tag", "Имя тега для обозначения ошибки обновления").SetVariants(tagNames).SetRequired(true);
    return scheme;
}

bool TRTInsuranceOrderStarterProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo["notifier"], NotifierName)
        && NJson::ParseField(jsonInfo["insurance_tag"], InsuranceTag, true)
        && NJson::ParseField(jsonInfo["waiting_order_tag"], WaitingOrderTag, true)
        && NJson::ParseField(jsonInfo["waiting_update_tag"], WaitingUpdateTag, true)
        && NJson::ParseField(jsonInfo["insurance_order_error_tag"], InsuranceOrderErrorTag, false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTInsuranceOrderStarterProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "notifier", NotifierName);
    NJson::InsertField(result, "insurance_tag", InsuranceTag);
    NJson::InsertField(result, "waiting_order_tag", WaitingOrderTag);
    NJson::InsertField(result, "waiting_update_tag", WaitingUpdateTag);
    NJson::InsertField(result, "insurance_order_error_tag", InsuranceOrderErrorTag);
    return result;
}

template <>
void Out<TInsuranceProcessError>(IOutputStream& out, const TInsuranceProcessError& value) {
    out
        << value.Message << ' '
        << value.RequestError << ' ';
}

TExpected<TOrderInfo, TInsuranceProcessError> TRTInsuranceOrderStarterProcess::RequestForOrder(const NDrive::IServer& server, const TDBTag& dbTag, const TInstant startProcess) const {
    TCarGenericAttachment currentRegistryDocument;
    if (!server.GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(dbTag.GetObjectId(), EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess)) {
        return MakeUnexpected<TInsuranceProcessError>({"Нет базовых данных"});
    }
    auto baseDocument = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
    if (!baseDocument || baseDocument->GetOsagoDateTo() == TInstant::Zero()) {
        return MakeUnexpected<TInsuranceProcessError>({"Нет базовых данных"});
    }
    auto tag = dbTag.GetTagAs<TNeedInsuranceUpdateTag>();
    if (!tag || !tag->HasProvider()) {
        return MakeUnexpected<TInsuranceProcessError>({"Ошибка при получении данных тега " + dbTag->GetName()});
    }
    switch (tag->GetProviderUnsafe()) {
    case TNeedInsuranceUpdateTag::EProvider::Major:
        {
            if (!server.GetDriveAPI()->HasMajorClient()) {
                return MakeUnexpected<TInsuranceProcessError>({"Major не определён"});
            }
            TMessagesCollector errors;
            NMajorClient::TOSAGORequest::TOSAGODocument majorDocument;
            if (!server.GetDriveAPI()->GetMajorClient().GetOSAGO(baseDocument->GetVin(), false, majorDocument, errors)) {
                return MakeUnexpected<TInsuranceProcessError>({"Ошибка при заказе страховки", false, errors.GetStringReport()});
            }
            if (baseDocument->GetOsagoMDSKey() && majorDocument.GetNumber() == baseDocument->GetOsagoNumber()) {
                return MakeUnexpected<TInsuranceProcessError>({"Ошибка при заказе страховки", false, "Страховка на обновлена"});
            }
            return {true, ""};
        }
    default:
        break;
    }
    return MakeUnexpected<TInsuranceProcessError>({"Неизвестна страховая"});
}

TExpectedState TRTInsuranceOrderStarterProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    const auto startProcess = Now();
    TVector<TString> carErrors;
    auto robotUserPermissions = server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), {});
    const auto& manager = server.GetDriveAPI()->GetTagsManager();
    TMap<TString, TDBTag> tagsByCarId;
    {
        auto session = manager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> tags;
        if (!manager.GetDeviceTags().RestoreTags({}, { InsuranceTag, WaitingOrderTag, InsuranceOrderErrorTag }, tags, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": failed get car tags " + session.GetStringReport());
        }
        Transform(tags.begin(), tags.end(), std::inserter(tagsByCarId, tagsByCarId.begin()), [](const TDBTag& tag) { return std::pair<TString, TDBTag>{tag.GetObjectId(), tag}; });
    }
    for (auto&& [id, carData] : context.GetFetchedCarsData()) {
        const TString& vin = carData.GetVin();
        auto currentTag = tagsByCarId.find(id);
        if (currentTag == tagsByCarId.end()) {
            continue;
        }
        auto& dbTag = currentTag->second;
        auto iTag = dbTag.MutableTagAs<TNeedInsuranceUpdateTag>();
        if (!iTag) {
            carErrors.emplace_back(vin + ": Тег некорректного типа '" + dbTag->GetName() + "'");
            continue;
        } else if (!iTag->GetApproved()) {
            continue;
        }
        auto orderData = RequestForOrder(server, dbTag, startProcess);
        if (orderData) {
            if (orderData.GetValue().Completed) {
                iTag->SetUpdateInfo(orderData.GetValue().Info);
                ITag::TPtr tag = manager.GetTagsMeta().CreateTag(WaitingUpdateTag);
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().EvolveTag(dbTag, tag, *robotUserPermissions, &server, session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            } else if (dbTag->GetName() != WaitingOrderTag) {
                iTag->SetUpdateInfo(orderData.GetValue().Info);
                ITag::TPtr tag = manager.GetTagsMeta().CreateTag(WaitingOrderTag);
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().EvolveTag(dbTag, tag, *robotUserPermissions, &server, session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            } else if (orderData.GetValue().Info && orderData.GetValue().Info != iTag->GetUpdateInfo()) {
                iTag->SetUpdateInfo(orderData.GetValue().Info);
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().UpdateTagData(dbTag, GetRobotUserId(), session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            }
        } else if (orderData.GetError().Critical) {
            carErrors.emplace_back(vin + ": " + orderData.GetError().Message);
        } else if (orderData.GetError().RequestError) {
            if (dbTag->GetName() != InsuranceOrderErrorTag) {
                ITag::TPtr tag = manager.GetTagsMeta().CreateTag(InsuranceOrderErrorTag);
                if (auto newTag = std::dynamic_pointer_cast<TNeedInsuranceUpdateTag>(tag)) {
                    newTag->SetErrorInfo(orderData.GetError().RequestError);
                } else {
                    carErrors.emplace_back(vin + ": Тег некорректного типа '" + InsuranceOrderErrorTag + "'");
                    continue;
                }
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().EvolveTag(dbTag, tag, *robotUserPermissions, &server, session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            } else if (orderData.GetError().RequestError && orderData.GetError().RequestError != iTag->GetErrorInfo()) {
                iTag->SetErrorInfo(orderData.GetError().RequestError);
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().UpdateTagData(dbTag, GetRobotUserId(), session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            }
        } else {
            carErrors.emplace_back(vin + ": " + orderData.GetError().Message);
        }
    }
    if (!carErrors.empty()) {
        if (NotifierName) {
            NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Ошибки при заказе страховки " + ToString(carErrors.size()), carErrors);
        } else {
            return MakeUnexpected<TString>(GetRobotId() + "Ошибки при заказе страховки " + ToString(carErrors.size()) + "\n" + JoinStrings(carErrors, "\n"));
        }
    }
    return new IRTBackgroundProcessState;
}

TRTInsuranceUploaderProcess::TFactory::TRegistrator<TRTInsuranceUploaderProcess> TRTInsuranceUploaderProcess::Registrator(TRTInsuranceUploaderProcess::GetTypeName());

NDrive::TScheme TRTInsuranceUploaderProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSDuration>("update_period", "Период до истечения срока страховки").SetDefault(TDuration::Hours(1)).SetRequired(true);
    TSet<TString> tagNames;
    if (auto impl = server.GetAs<NDrive::IServer>()) {
        tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TNeedInsuranceUpdateTag::TypeName});
    }
    scheme.Add<TFSVariants>("waiting_update_tag", "Имя тега для ожидания обновления страховки").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSVariants>("insurance_update_error_tag", "Имя тега для обозначения ошибки обновления").SetVariants(tagNames).SetRequired(true);
    return scheme;
}

bool TRTInsuranceUploaderProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo["notifier"], NotifierName)
        && NJson::ParseField(jsonInfo["update_period"], UpdatePeriod, true)
        && NJson::ParseField(jsonInfo["waiting_update_tag"], WaitingUpdateTag, true)
        && NJson::ParseField(jsonInfo["insurance_update_error_tag"], InsuranceUpdateErrorTag, false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTInsuranceUploaderProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "notifier", NotifierName);
    NJson::InsertField(result, "update_period", ToString(UpdatePeriod.Seconds()));
    NJson::InsertField(result, "waiting_update_tag", WaitingUpdateTag);
    NJson::InsertField(result, "insurance_update_error_tag", InsuranceUpdateErrorTag);
    return result;
}

TMaybe<TInsuranceProcessError> TRTInsuranceUploaderProcess::RequestForUpdate(const NDrive::IServer& server, const TS3Client::TBucket* mdsBucket, const TDBTag& dbTag, const TInstant startProcess, const TString& /*updateData*/) const {
    TCarGenericAttachment currentRegistryDocument;
    if (!server.GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(dbTag.GetObjectId(), EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess)) {
        return TInsuranceProcessError("Нет базовых данных");
    }
    auto baseDocument = std::dynamic_pointer_cast<TCarRegistryDocument>(currentRegistryDocument.GetImpl());
    if (!baseDocument || baseDocument->GetOsagoDateTo() == TInstant::Zero()) {
        return TInsuranceProcessError("Нет базовых данных");
    }
    if (baseDocument->GetOsagoDateTo() - UpdatePeriod > Now()) {
        return {};
    }
    auto tag = dbTag.GetTagAs<TNeedInsuranceUpdateTag>();
    if (!tag || !tag->HasProvider()) {
        return TInsuranceProcessError("Ошибка при получении данных тега " + dbTag->GetName());
    }
    switch (tag->GetProviderUnsafe()) {
    case TNeedInsuranceUpdateTag::EProvider::Major:
        {
            bool isUpdate = false;
            TMessagesCollector errors;
            if (!baseDocument->UpdateOSAGO(isUpdate, *mdsBucket, server, errors)) {
                return TInsuranceProcessError("Ошибка при обновлении страховки", false, errors.GetStringReport());
            }
            if (!isUpdate) {
                return TInsuranceProcessError("Ошибка при обновлении страховки", false, "Страховка на обновлена");
            }
        }
        break;
    default:
        return TInsuranceProcessError("Неизвестна страховая");
    }
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    TCarGenericAttachment attachmentNew(baseDocument);
    if (!server.GetDriveAPI()->GetCarAttachmentAssignments().Attach(attachmentNew, dbTag.GetObjectId(), GetRobotUserId(), session, &server)) {
        return TInsuranceProcessError("Страховка не обновлена" + session.GetStringReport(), true);
    }
    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTag(dbTag, GetRobotUserId(), &server, session)) {
        return TInsuranceProcessError("Ошибка при удалении тега" + session.GetStringReport(), true);
    }
    if (!session.Commit()) {
        return TInsuranceProcessError("Ошибка при обновлении данных о страховке" + session.GetStringReport(), true);
    }
    return {};
}

TExpectedState TRTInsuranceUploaderProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    const auto startProcess = Now();
    TVector<TString> carErrors;
    auto robotUserPermissions = server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), {});
    const auto& manager = server.GetDriveAPI()->GetTagsManager();
    const TS3Client::TBucket* mdsBucket = server.GetDriveAPI()->GetMDSClient().GetBucket(MdsBucket);
    if (!mdsBucket) {
        ERROR_LOG << GetRobotId() << ": mdsBucket " << MdsBucket << " undefined" << Endl;
        return nullptr;
    }
    TMap<TString, TDBTag> tagsByCarId;
    {
        auto session = manager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        TVector<TDBTag> tags;
        if (!manager.GetDeviceTags().RestoreTags({}, { WaitingUpdateTag, InsuranceUpdateErrorTag }, tags, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": failed get car tags " + session.GetStringReport());
        }
        Transform(tags.begin(), tags.end(), std::inserter(tagsByCarId, tagsByCarId.begin()), [](const TDBTag& tag) { return std::pair<TString, TDBTag>{tag.GetObjectId(), tag}; });
    }
    for (auto&& [id, carData] : context.GetFetchedCarsData()) {
        const TString& vin = carData.GetVin();
        auto currentTag = tagsByCarId.find(id);
        if (currentTag == tagsByCarId.end()) {
            continue;
        }
        auto& dbTag = currentTag->second;
        auto iTag = dbTag.MutableTagAs<TNeedInsuranceUpdateTag>();
        if (!iTag || !iTag->HasProvider()) {
            carErrors.emplace_back(vin + ": Тег некорректного типа '" + dbTag->GetName() + "'");
            continue;
        } else if (!iTag->GetApproved()) {
            continue;
        }
        auto updateError = RequestForUpdate(server, mdsBucket, dbTag, startProcess, iTag->GetUpdateInfo());
        if (!updateError) {
            continue;
        }
        if (updateError->Critical) {
            carErrors.emplace_back(vin + ": " + updateError->Message);
        } else if (updateError->RequestError) {
            if (iTag->GetName() != InsuranceUpdateErrorTag) {
                ITag::TPtr tag = manager.GetTagsMeta().CreateTag(InsuranceUpdateErrorTag);
                if (auto newTag = std::dynamic_pointer_cast<TNeedInsuranceUpdateTag>(tag)) {
                    newTag->SetErrorInfo(updateError->RequestError);
                } else {
                    carErrors.emplace_back(vin + ": Тег некорректного типа '" + InsuranceUpdateErrorTag + "'");
                    continue;
                }
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().EvolveTag(dbTag, tag, *robotUserPermissions, &server, session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            } else if (updateError->RequestError != iTag->GetErrorInfo()) {
                iTag->SetErrorInfo(updateError->RequestError);
                auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                if (!manager.GetDeviceTags().UpdateTagData(dbTag, GetRobotUserId(), session) || !session.Commit()) {
                    carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                }
            }
            carErrors.emplace_back(vin + ": " + updateError->Message + " " + updateError->RequestError);
        } else {
            carErrors.emplace_back(vin + ": " + updateError->Message);
        }
    }
    if (!carErrors.empty()) {
        if (NotifierName) {
            NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Ошибки при загрузке страховки " + ToString(carErrors.size()), carErrors);
        } else {
            return MakeUnexpected<TString>(GetRobotId() + "Ошибки при загрузке страховки " + ToString(carErrors.size()) + "\n" + JoinStrings(carErrors, "\n"));
        }
    }
    return new IRTBackgroundProcessState;
}

TRTInsuranceOrderWatcherProcess::TFactory::TRegistrator<TRTInsuranceOrderWatcherProcess> TRTInsuranceOrderWatcherProcess::Registrator(TRTInsuranceOrderWatcherProcess::GetTypeName());

NDrive::TScheme TRTInsuranceOrderWatcherProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    TSet<TString> tagNames;
    if (auto impl = server.GetAs<NDrive::IServer>()) {
        tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TNeedInsuranceUpdateTag::TypeName});
    }
    scheme.Add<TFSVariants>("waiting_update_tag", "Имя тега для ожидания обновления страховки").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSVariants>("insurance_update_error_tag", "Имя тега для обозначения ошибки обновления").SetVariants(tagNames).SetRequired(true);
    return scheme;
}

bool TRTInsuranceOrderWatcherProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo["notifier"], NotifierName)
        && NJson::ParseField(jsonInfo["waiting_update_tag"], WaitingUpdateTag, true)
        && NJson::ParseField(jsonInfo["insurance_update_error_tag"], InsuranceUpdateErrorTag, false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTInsuranceOrderWatcherProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "notifier", NotifierName);
    NJson::InsertField(result, "waiting_update_tag", WaitingUpdateTag);
    NJson::InsertField(result, "insurance_update_error_tag", InsuranceUpdateErrorTag);
    return result;
}

TMaybe<TInsuranceProcessError> TRTInsuranceOrderWatcherProcess::RequestToCheckUpdate(const NDrive::IServer& server, const TDBTag& dbTag, const TInstant startProcess) const {
    TCarGenericAttachment currentRegistryDocument;
    if (!server.GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(dbTag.GetObjectId(), EDocumentAttachmentType::CarRegistryDocument, currentRegistryDocument, startProcess)) {
        return TInsuranceProcessError("Нет базовых данных");
    }
    auto baseDocument = dynamic_cast<const TCarRegistryDocument*>(currentRegistryDocument.Get());
    if (!baseDocument || baseDocument->GetOsagoDateTo() == TInstant::Zero()) {
        return TInsuranceProcessError("Нет базовых данных");
    }
    auto tag = dbTag.GetTagAs<TNeedInsuranceUpdateTag>();
    if (!tag || !tag->HasProvider()) {
        return TInsuranceProcessError("Ошибка при получении данных тега " + dbTag->GetName());
    }
    switch (tag->GetProviderUnsafe()) {
    case TNeedInsuranceUpdateTag::EProvider::Major:
        {
            if (!server.GetDriveAPI()->HasMajorClient()) {
                return TInsuranceProcessError("Major не определён");
            }
            TMessagesCollector errors;
            NMajorClient::TOSAGORequest::TOSAGODocument majorDocument;
            if (!server.GetDriveAPI()->GetMajorClient().GetOSAGO(baseDocument->GetVin(), false, majorDocument, errors)) {
                return TInsuranceProcessError("Ошибка при скачивании страховки", false, errors.GetStringReport());
            }
            if (baseDocument->GetOsagoMDSKey() && majorDocument.GetNumber() == baseDocument->GetOsagoNumber()) {
                return TInsuranceProcessError("Страховка не обновлена", false, "Страховка не обновлена у Major");
            }
            return {};
        }
    default:
        break;
    }
    return TInsuranceProcessError("Неизвестна страховая");
}

TExpectedState TRTInsuranceOrderWatcherProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    const auto startProcess = Now();
    TVector<TString> carErrors;
    auto robotUserPermissions = server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), {});
    const auto& manager = server.GetDriveAPI()->GetTagsManager();
    TMap<TString, TDBTag> tagsByCarId;
    {
        auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
        TVector<TDBTag> tags;
        if (!manager.GetDeviceTags().RestoreTags({}, { WaitingUpdateTag, InsuranceUpdateErrorTag }, tags, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": failed get car tags " + session.GetStringReport());
        }
        Transform(tags.begin(), tags.end(), std::inserter(tagsByCarId, tagsByCarId.begin()), [](const TDBTag& tag) { return std::pair<TString, TDBTag>{tag.GetObjectId(), tag}; });
    }
    for (auto&& [id, carData] : context.GetFetchedCarsData()) {
        const TString& vin = carData.GetVin();
        auto currentTag = tagsByCarId.find(id);
        if (currentTag == tagsByCarId.end()) {
            continue;
        }
        auto& dbTag = currentTag->second;
        auto iTag = dbTag.MutableTagAs<TNeedInsuranceUpdateTag>();
        if (!iTag || !iTag->HasProvider()) {
            carErrors.emplace_back(vin + ": Тег некорректного типа '" + dbTag->GetName() + "'");
            continue;
        } else if (!iTag->GetApproved()) {
            continue;
        }
        auto checkError = RequestToCheckUpdate(server, dbTag, startProcess);
        if (checkError) {
            if (checkError->Critical) {
                carErrors.emplace_back(vin + ": " + checkError->Message);
                break;
            } else if (checkError->RequestError) {
                if (iTag->GetName() != InsuranceUpdateErrorTag) {
                    ITag::TPtr tag = manager.GetTagsMeta().CreateTag(InsuranceUpdateErrorTag);
                    if (auto newTag = std::dynamic_pointer_cast<TNeedInsuranceUpdateTag>(tag)) {
                        newTag->SetErrorInfo(checkError->RequestError);
                    } else {
                        carErrors.emplace_back(vin + ": Тег некорректного типа '" + InsuranceUpdateErrorTag + "'");
                        continue;
                    }
                    auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                    if (!manager.GetDeviceTags().EvolveTag(dbTag, tag, *robotUserPermissions, &server, session) || !session.Commit()) {
                        carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                    }
                } else if (checkError->RequestError != iTag->GetErrorInfo()) {
                    iTag->SetErrorInfo(checkError->RequestError);
                    auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
                    if (!manager.GetDeviceTags().UpdateTagData(dbTag, GetRobotUserId(), session) || !session.Commit()) {
                        carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
                    }
                }
            }
            carErrors.emplace_back(vin + ": " + checkError->Message + " " + checkError->RequestError);
        } else if (iTag->GetName() != WaitingUpdateTag) {
            ITag::TPtr tag = manager.GetTagsMeta().CreateTag(WaitingUpdateTag);
            auto session = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
            if (!manager.GetDeviceTags().EvolveTag(dbTag, tag, *robotUserPermissions, &server, session) || !session.Commit()) {
                carErrors.emplace_back(vin + ": Ошибка при изменении тега " + session.GetStringReport());
            }
        }
    }
    if (!carErrors.empty()) {
        if (NotifierName) {
            NDrive::INotifier::MultiLinesNotify(server.GetNotifier(GetNotifierName()), "Ошибки при загрузке страховки " + ToString(carErrors.size()), carErrors);
        } else {
            return MakeUnexpected<TString>(GetRobotId() + "Ошибки при загрузке страховки " + ToString(carErrors.size()) + "\n" + JoinStrings(carErrors, "\n"));
        }
    }
    return new IRTBackgroundProcessState;
}
