#include "process.h"

#include <drive/backend/data/parking_payment.h>
#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/threading/future.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/container.h>

TExpectedState TParkingPaymentMarkerWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    Y_UNUSED(state);
    const TDriveAPI* api = server.GetDriveAPI();
    if (!api) {
        ERROR_LOG << GetRobotId() << ": DriveApi is missing" << Endl;
        return MakeUnexpected<TString>({});
    }

    const IDriveTagsManager& tagsManager = api->GetTagsManager();
    TVector<TTaggedDevice> objects;
    {
        auto tags = { TParkingInfoTag::TagName() };
        if (!tagsManager.GetDeviceTags().GetObjectsFromCache(context.GetFilteredCarIds(), tags, objects, TInstant::Zero())) {
            ERROR_LOG << GetRobotId() << ": cannot get tagged objects" << Endl;
            return MakeUnexpected<TString>({});
        }
    }

    for (auto&& object : objects) {
        auto tag = object.GetTag(TParkingInfoTag::TagName());
        if (!tag) {
            ERROR_LOG << GetRobotId() << ": cannot acquire " << TParkingInfoTag::TagName() << " tag from " << object.GetId() << Endl;
            continue;
        }
        auto parkingPaymentTag = tag->GetTagAs<TParkingInfoTag>();
        if (!parkingPaymentTag) {
            ERROR_LOG << GetRobotId() << ": cannot cast " << tag->GetTagId() << " to ParkingInfoTag" << Endl;
            continue;
        }
        double cost = parkingPaymentTag->GetCost(Now());
        auto session = api->template BuildTx<NSQL::Writable>();
        auto existingMarkers = tagsManager.GetDeviceTags().RestoreEntityTags(object.GetId(), { TagName }, session);
        if (!existingMarkers) {
            ERROR_LOG << GetRobotId() << ": cannot restore tags for " << object.GetId() << ": " << session.GetStringReport() << Endl;
            continue;
        }
        auto existingMarker = !existingMarkers->empty() ? existingMarkers->front() : TDBTag();
        if (cost >= CostThreshold && !existingMarker) {
            auto marker = tagsManager.GetTagsMeta().CreateTag(TagName, ToString(cost));
            if (!marker) {
                ERROR_LOG << GetRobotId() << ": cannot create tag " << TagName << Endl;
                return MakeUnexpected<TString>({});
            }
            if (!tagsManager.GetDeviceTags().AddTag(marker, GetRobotUserId(), object.GetId(), &server, session)) {
                ERROR_LOG << GetRobotId() << ": cannot add tag for " << object.GetId() << ": " << session.GetStringReport() << Endl;
                continue;
            }
        } else if (cost < CostThreshold && existingMarker) {
            if (!tagsManager.GetDeviceTags().RemoveTag(existingMarker, GetRobotUserId(), &server, session)) {
                ERROR_LOG << GetRobotId() << ": cannot remove tag " << existingMarker.GetTagId() << ": " << session.GetStringReport() << Endl;
                continue;
            }
        }
        if (!session.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot commit transaction: " << session.GetStringReport() << Endl;
            continue;
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TParkingPaymentMarkerWatcher::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto registeredDeviceTags = api ? api->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car) : ITagsMeta::TTagDescriptionsByName();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("cost_threshold", "Cost threshold").SetDefault(CostThreshold).SetRequired(true);
    scheme.Add<TFSVariants>("tag_name", "Name of the tag to add").SetVariants(registeredDeviceTags).SetRequired(true);
    return scheme;
}

bool TParkingPaymentMarkerWatcher::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["cost_threshold"], CostThreshold, true) &&
        NJson::ParseField(value["tag_name"], TagName, true);
}

NJson::TJsonValue TParkingPaymentMarkerWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["cost_threshold"] = CostThreshold;
    result["tag_name"] = TagName;
    return result;
}

TParkingPaymentMarkerWatcher::TFactory::TRegistrator<TParkingPaymentMarkerWatcher> TParkingPaymentMarkerWatcher::Registrator(TParkingPaymentMarkerWatcher::GetTypeName());
