#include "config.h"

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/history_iterator/history_iterator.h>


TRTLeadsDataDistributeState::TFactory::TRegistrator<TRTLeadsDataDistributeState> TRTLeadsDataDistributeState::Registrator(TRTLeadsDataDistributer::GetTypeName());

TString TRTLeadsDataDistributeState::GetType() const {
    return TRTLeadsDataDistributer::GetTypeName();
}


TRTLeadsDataDistributer::TFactory::TRegistrator<TRTLeadsDataDistributer> TRTLeadsDataDistributer::Registrator(TRTLeadsDataDistributer::GetTypeName());

TString TRTLeadsDataDistributer::GetType() const {
    return GetTypeName();
}

NDrive::TScheme TRTLeadsDataDistributer::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("since", "Скакого момента обрабатывать поездки.").SetVisual(TFSNumeric::DateTime);
    scheme.Add<TFSVariants>("receiver", "Идентификатор партнёра").InitVariants<NDrive::NPartners::EAggregator>().SetRequired(true);
    TSet<TString> tagNames;
    if (auto impl = server.GetAs<NDrive::IServer>()) {
        tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TUniqueUserTag::TypeName });
    }
    scheme.Add<TFSVariants>("allow_tag", "Имя тега, обозначающий пользователей, чьи данные можно собирать").SetVariants(tagNames).SetRequired(true);
    scheme.Add<TFSNumeric>("sessions_limit", "Количество поездок в обработке за раз").SetDefault(SessionsLimit);
    scheme.Add<TFSText>("client_config", "Конфиг агрегатора");
    return scheme;
}

bool TRTLeadsDataDistributer::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!NJson::ParseField(jsonInfo["receiver"], NJson::Stringify(Receiver), /* required = */ true)) {
        return false;
    }
    Client = NDrive::NPartners::ILeadsSender::GetPartnerClient(Receiver);
    if (!Client) {
        return false;
    }
    if (!NJson::ParseField(jsonInfo["client_config"], ClientConfig, /* required = */ true) || !Client->Init(ClientConfig)) {
        return false;
    }
    return NJson::ParseField(jsonInfo["since"], Since, /* required = */ false)
        && NJson::ParseField(jsonInfo["allow_tag"], AllowTag, /* required = */ true)
        && NJson::ParseField(jsonInfo["sessions_limit"], SessionsLimit, /* required = */ true)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTLeadsDataDistributer::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "since", Since.Seconds());
    NJson::InsertField(result, "receiver", NJson::Stringify(Receiver));
    NJson::InsertField(result, "allow_tag", AllowTag);
    NJson::InsertField(result, "sessions_limit", SessionsLimit);
    NJson::InsertField(result, "client_config", ClientConfig);
    return result;
}

TExpectedState TRTLeadsDataDistributer::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    ui64 lastEventId = 0;
    if (auto currentState = std::dynamic_pointer_cast<TRTLeadsDataDistributeState>(state)) {
        lastEventId = currentState->GetLastEventId();
    }

    const auto& ridesManager = server.GetDriveAPI()->GetMinimalCompiledRides();
    auto tx = ridesManager.BuildTx<NSQL::ReadOnly>();
    auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("rtleads_data_distributer", &server);
    auto rides = ridesManager.GetObjects<TMinimalCompiledRiding>(MakeVector(context.GetFilteredCarIds()), tx, ydbTx, { Since }, { lastEventId + 1 }, SessionsLimit);
    if (!rides) {
        return MakeUnexpected<TString>(GetRobotId() + ": fail to get history rides: " + tx.GetStringReport());
    }

    TVector<TDBTag> agreements;
    {
        TSet<TString> userIds;
        Transform(rides->begin(), rides->end(), std::inserter(userIds, userIds.begin()), [](const auto& item) { return item.GetHistoryUserId(); });
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(userIds, { AllowTag }, agreements, tx)) {
            return MakeUnexpected<TString>(GetRobotId() + ": fail to get user tags: " + tx.GetStringReport());
        }
    }

    TVector<TString> userIds;
    Transform(agreements.begin(), agreements.end(), std::back_inserter(userIds), [](const auto& item) { return item.GetObjectId(); });
    auto usersFetchResult = server.GetDriveAPI()->GetUsersData()->FetchInfo(userIds, tx);
    if (!usersFetchResult) {
        return MakeUnexpected<TString>(GetRobotId() + ": fail to get users data: " + tx.GetStringReport());
    }

    for (auto&& ride : *rides) {
        if (auto lead = usersFetchResult.GetResultPtr(ride.GetHistoryUserId()); lead && (lead->GetPhone() || lead->GetEmail())) {
            auto future = Client->SendLead(*lead, ride);
            if (!future.Wait(Client->GetRequestTimeout())) {
                NDrive::TEventLog::Log("LeadsDataDistributerError", NJson::TMapBuilder
                    ("error", "Fail to send lead data with timeout")
                );
                break;
            }
            if (!future.HasValue()) {
                NDrive::TEventLog::Log("LeadsDataDistributerError", NJson::TMapBuilder
                    ("error", "Fail to send leads data with error: " + NThreading::GetExceptionMessage(future))
                );
                break;
            }
            lastEventId = ride.GetHistoryEventId();
        }
    }

    auto result = MakeAtomicShared<TRTLeadsDataDistributeState>();
    result->SetLastEventId(lastEventId);
    return result;
}
