#include "processor.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/abstract/settings.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TUserPassiveStatusMaintainer::TFactory::TRegistrator<TUserPassiveStatusMaintainer> TUserPassiveStatusMaintainer::Registrator("user_passive_status_maintainer");

TExpectedState TUserPassiveStatusMaintainer::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    TSet<TString> userIds;
    if (!server->GetDriveAPI()->GetUsersData()->GetRoles().GetUsersWithAtLeastOneRoles(Roles, userIds, true)) {
        return new IRTBackgroundProcessState();
    }

    auto usersFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(userIds);
    for (auto&& userIt : usersFR) {
        auto user = userIt.second;
        if (user.GetStatus() == NDrive::UserStatusOnboarding) {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            user.SetStatus(NDrive::UserStatusPassive);
            bool isOK = server->GetDriveAPI()->GetUsersData()->UpdateUser(user, GetRobotUserId(), session) && session.Commit();
            if (isOK) {
                NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "Пользователь " + user.GetHRReport() + " переходит в статус passive, так как это сервисный работник");
            } else {
                NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "Не могу перевести пользователя " + user.GetHRReport() + " в статус passive");
            }
        }
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TUserPassiveStatusMaintainer::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("notifier_name", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSArray>("roles", "Сервисные роли").SetElement<TFSString>();

    return scheme;
}

NJson::TJsonValue TUserPassiveStatusMaintainer::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    NJson::TJsonValue rolesJson = NJson::JSON_ARRAY;
    for (auto&& role : Roles) {
        rolesJson.AppendValue(role);
    }
    result["roles"] = std::move(rolesJson);

    TJsonProcessor::Write(result, "notifier_name", NotifierName);

    return result;
}

bool TUserPassiveStatusMaintainer::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING_OPT(jsonInfo, "notifier_name", NotifierName);

    Roles.clear();
    if (jsonInfo.Has("roles") && jsonInfo["roles"].IsArray()) {
        for (auto&& roleRaw : jsonInfo["roles"].GetArray()) {
            if (!roleRaw.IsString()) {
                return false;
            }
            Roles.insert(roleRaw.GetString());
        }
    } else {
        return false;
    }

    return true;
}
