#include "processor.h"

#include <drive/backend/data/radar.h>
#include <drive/backend/device_snapshot/manager.h>

#include <rtline/library/geometry/polyline.h>

TExpectedState TRadarUserTagProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& tagsManager = api->GetTagsManager();
    auto session = api->template BuildTx<NSQL::Writable | NSQL::RepeatableRead | NSQL::Deferred>();
    TDBTags radarTags;
    if (!tagsManager.GetUserTags().RestoreTags({}, {TRadarUserTag::TypeName}, radarTags, session)) {
        return MakeUnexpected<TString>(GetRobotId() + ": Cannot get user tags " + session.GetStringReport());
    }
    TSet<TString> removedTags;
    TRadarUserTag::Sort(radarTags);
    {
        auto elsg = NDrive::TEventLog::Guard("RadarProcessOrderTags");
        for (auto&& radarTag : radarTags) {
            const TRadarUserTag* tag = Yensured(radarTag.GetTagAs<TRadarUserTag>());
            if (tag->GetAction() != TRadarUserTag::EAction::Order) {
                continue;
            }
            if (removedTags.contains(radarTag.GetTagId())) {
                continue;
            }
            const TString& userId = radarTag.GetObjectId();
            if (!TRadarUserTag::UserCanRentCar(server, userId)) {
                continue;
            }
            auto permissions = api->GetUserPermissions(userId, {}, TInstant::Zero(), nullptr, false, false);
            if (!permissions) {
                NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                    ("stage", "RadarProcessOrderTags")
                    ("user_id", userId)
                    ("error", "can not retrieve user permissions")
                );
                continue;
            }
            auto result = ApplyRadarTag(server, permissions, radarTag, session);
            if (!result) {
                NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                    ("stage", "RadarProcessOrderTags")
                    ("user_id", userId)
                    ("error", session.GetStringReport())
                );
                return MakeUnexpected<TString>(GetRobotId() + ": Error while applying radar tag: " + session.GetStringReport());
            }
            session.Check();
            if (*result) {
                removedTags.insert(radarTag.GetTagId());
            }
        }
    }
    {
        auto elsg = NDrive::TEventLog::Guard("RadarProcessSearchTags");
        for (auto&& radarTag : radarTags) {
            const TRadarUserTag* tag = Yensured(radarTag.GetTagAs<TRadarUserTag>());
            if (tag->GetAction() != TRadarUserTag::EAction::Search) {
                continue;
            }
            if (removedTags.contains(radarTag.GetTagId())) {
                continue;
            }
            const TString& userId = radarTag.GetObjectId();
            if (!TRadarUserTag::UserCanRentCar(server, userId)) {
                continue;
            }
            auto permissions = api->GetUserPermissions(userId, {}, TInstant::Zero(), nullptr, false, false);
            if (!permissions) {
                NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                    ("stage", "RadarProcessSearchTags")
                    ("user_id", userId)
                    ("error", "can not retrieve user permissions")
                );
                continue;
            }
            auto result = ApplyRadarTag(server, permissions, radarTag, session);
            if (!result) {
                NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                    ("stage", "RadarProcessOrderTags")
                    ("user_id", userId)
                    ("error", session.GetStringReport())
                );
                return MakeUnexpected<TString>(GetRobotId() + ": Error while applying radar tag: " + session.GetStringReport());
            }
            session.Check();
            if (*result) {
                removedTags.insert(radarTag.GetTagId());
            }
        }
    }
    {
        auto elsg = NDrive::TEventLog::Guard("RadarProcessEmptyAreaTags");
        TDBTags filteredRadarTags;
        for (auto&& radarTag : radarTags) {
            const TRadarUserTag* tag = Yensured(radarTag.GetTagAs<TRadarUserTag>());
            if (!tag->GetSearchArea().empty()) {
                continue;
            }
            if (removedTags.contains(radarTag.GetTagId())) {
                continue;
            }
            if (!TRadarUserTag::UserCanRentCar(server, radarTag.GetObjectId())) {
                continue;
            }
            filteredRadarTags.push_back(radarTag);
        }
        TVector<TCarInfo> result;
        auto snapshots = server.GetSnapshotsManager().GetSnapshots();
        for (auto&& [objectId, _] : snapshots.GetSnapshots()) {
            bool carBooked = false;
            for (auto&& radarTag : filteredRadarTags) {
                const TRadarUserTag* tag = Yensured(radarTag.GetTagAs<TRadarUserTag>());
                if (tag->GetAction() != TRadarUserTag::EAction::Order) {
                    continue;
                }
                if (removedTags.contains(radarTag.GetTagId())) {
                    continue;
                }
                const TString& userId = radarTag.GetObjectId();
                auto permissions = api->GetUserPermissions(userId, {}, TInstant::Zero(), nullptr, false, false);
                if (!permissions) {
                    NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                        ("stage", "RadarProcessEmptyAreaTags")
                        ("user_id", userId)
                        ("error", "can not retrieve user permissions")
                    );
                    continue;
                }
                auto result = ApplyRadarTagByCar(server, permissions, radarTag, objectId, session);
                if (!result) {
                    NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                        ("stage", "RadarProcessEmptyAreaTags")
                        ("user_id", userId)
                        ("error", session.GetStringReport())
                    );
                    return MakeUnexpected<TString>(GetRobotId() + ": Error while applying radar tag by car: " + session.GetStringReport());
                }
                session.Check();
                // We do not check radar search area.
                if (*result) {
                    removedTags.insert(radarTag.GetTagId());
                    carBooked = true;
                    break;
                }
            }
            if (carBooked) {
                continue;
            }
            for (auto&& radarTag : filteredRadarTags) {
                const TRadarUserTag* tag = Yensured(radarTag.GetTagAs<TRadarUserTag>());
                if (tag->GetAction() != TRadarUserTag::EAction::Search) {
                    continue;
                }
                if (removedTags.contains(radarTag.GetTagId())) {
                    continue;
                }
                const TString& userId = radarTag.GetObjectId();
                auto permissions = api->GetUserPermissions(userId, {}, TInstant::Zero(), nullptr, false, false);
                if (!permissions) {
                    NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                        ("stage", "RadarProcessEmptyAreaTags")
                        ("user_id", userId)
                        ("error", "can not retrieve user permissions")
                    );
                    continue;
                }
                auto result = ApplyRadarTagByCar(server, permissions, radarTag, objectId, session);
                if (!result) {
                    NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                        ("stage", "RadarProcessEmptyAreaTags")
                        ("user_id", userId)
                        ("error", session.GetStringReport())
                    );
                    return MakeUnexpected<TString>(GetRobotId() + ": Error while applying radar tag by car: " + session.GetStringReport());
                }
                session.Check();
                // We do not check radar search area.
                if (*result) {
                    removedTags.insert(radarTag.GetTagId());
                }
            }
        }
    }
    if (!session.Commit()) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TVector<TRadarUserTagProcess::TCarInfo> TRadarUserTagProcess::FindBBoxCars(const NDrive::IServer& server, const TDBTag& radarTag) {
    auto elsg = NDrive::TEventLog::Guard("RadarProcessFindBBoxCars");
    const TRadarUserTag* tag = Yensured(radarTag.GetTagAs<TRadarUserTag>());
    if (tag->GetSearchArea().empty()) {
        if (tag->GetFilterIdentifiers().empty()) {
            NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
                ("stage", "FindBBoxCars")
                ("error", "empty search area")
                ("tag_id", radarTag.GetTagId())
            );
        }
        return {};
    }
    auto rect = TPolyLine<TGeoCoord>(tag->GetSearchArea()).GetRectSafe();
    auto geoHash = Yensured(server.GetSnapshotsManager().GetGeoHash());
    TVector<TCarInfo> result;
    auto actor = [&result, &radarTag](const TDevicesSnapshotManager::THashedDevice& d) {
        TCarInfo info = {d.CarId, d.Location.GetCoord()};
        if (!TRadarUserTag::FilterByCoords(radarTag, info.Coord)) {
            return true;
        }
        result.push_back(info);
        return true;
    };
    geoHash->FindObjects(rect, actor);
    return result;
}

TMaybe<bool> TRadarUserTagProcess::ApplyRadarTagByCar(const NDrive::IServer& server, const TUserPermissions::TConstPtr permissions, TDBTag& dbTag, const TString& carId, NDrive::TEntitySession& session) {
    auto elsg = NDrive::TEventLog::Guard("RadarProcessApplyRadarTagByCar");
    const TString& userId = dbTag.GetObjectId();
    auto device = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObject(carId);
    if (!device) {
        NDrive::TEventLog::Log("RadarProcessError", NJson::TMapBuilder
            ("stage", "ApplyRadarTagByCar")
            ("car_id", carId)
            ("user_id", userId)
            ("error", "can not retrieve car info")
        );
        return false;
    }
    if (!TRadarUserTag::FilterByFilterSet(server, dbTag, *device)) {
        return false;
    }
    if (permissions->GetVisibility(*device, NEntityTagsManager::EEntityType::Car) != TUserPermissions::EVisibility::Visible) {
        return false;
    }
    NDrive::TEventLog::TUserIdGuard userIdGuard(userId);
    const TRadarUserTag* tag = Yensured(dbTag.GetTagAs<TRadarUserTag>());
    if (tag->GetAction() == TRadarUserTag::EAction::Order) {
        return TRadarUserTag::ApplyCarOrder(server, permissions, dbTag, carId, session);
    } else if (tag->GetAction() == TRadarUserTag::EAction::Search) {
        return TRadarUserTag::ApplyCarSearch(server, permissions, dbTag, carId, session);
    }
    return false;
}

TMaybe<bool> TRadarUserTagProcess::ApplyRadarTag(const NDrive::IServer& server, const TUserPermissions::TConstPtr permissions, TDBTag& dbTag, NDrive::TEntitySession& tx) {
    TVector<TCarInfo> carInfos = FindBBoxCars(server, dbTag);
    if (carInfos.empty()) {
        return false;
    }
    for (auto&& carInfo : carInfos) {
        auto result = ApplyRadarTagByCar(server, permissions, dbTag, carInfo.Id, tx);
        if (!result) {
            return {};
        }
        if (*result) {
            return true;
        }
        tx.Check();
    }
    return false;
}

NDrive::TScheme TRadarUserTagProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    return scheme;
}

bool TRadarUserTagProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeFromJson(value);
}

NJson::TJsonValue TRadarUserTagProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    return result;
}

TRadarUserTagProcess::TFactory::TRegistrator<TRadarUserTagProcess> TRadarUserTagProcess::Registrator(TRadarUserTagProcess::GetTypeName());
