#include "processor.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/abstract/settings.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/database/drive/private_data.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TRegistratorMaintenance::TFactory::TRegistrator<TRegistratorMaintenance> TRegistratorMaintenance::Registrator(TRegistratorMaintenance::GetTypeName());
TRegistratorRegular::TFactory::TRegistrator<TRegistratorRegular> TRegistratorRegular::Registrator(TRegistratorRegular::GetTypeName());

TExpectedState TRegistratorMaintenance::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const TDriveAPI& api = *Yensured(server.GetDriveAPI());
    const TUsersDB& userDb = *Yensured(api.GetUsersData());
    const IUserRegistrationManager& userRegistrationManager = *Yensured(server.GetUserRegistrationManager());
    const TUserTagsManager& userTagManager = api.GetTagsManager().GetUserTags();
    {
        TUsersDB::TFetchResult userFetchResult;
        {
            auto session = userTagManager.BuildSession(true);
            userFetchResult = userDb.FetchWithCustomQuery(R"(
                SELECT
                    *
                FROM
                    "user"
                WHERE
                    passport_ds_revision IS NOT NULL AND passport_ds_revision <> ''
                AND
                    uid IS NOT NULL
                AND
                    (passport_names_hash IS NULL OR passport_names_hash = '')
                LIMIT
                    42000
            )", session.GetTransaction());
        }
        INFO_LOG << GetRobotId() << ": fetched " << userFetchResult.size() << " candidates" << Endl;
        TMap<TString, NThreading::TFuture<TUserPassportData>> userPassportData;
        for (auto&& [id, user] : userFetchResult) {
            if (!user.GetPassportDatasyncRevision()) {
                DEBUG_LOG << GetRobotId() << ": skipping " << id << Endl;
                continue;
            }
            INFO_LOG << GetRobotId() << ": fetching data for " << id << Endl;
            auto passportData = api.GetPrivateDataClient().GetPassport(user, user.GetPassportDatasyncRevision());
            if (!passportData.Initialized()) {
                ERROR_LOG << GetRobotId() << ": cannot GetPassport for " << id << Endl;
                continue;
            }
            userPassportData.emplace(id, passportData);
        }
        TDuration timeout = TDuration::Seconds(10);
        for (auto&& [id, data] : userPassportData) {
            if (!data.Wait(timeout)) {
                ERROR_LOG << GetRobotId() << ": " << id << " wait timeout" << Endl;
                continue;
            }
            if (!data.HasValue()) {
                ERROR_LOG << GetRobotId() << ": " << id << " exception " << NThreading::GetExceptionMessage(data) << Endl;
                continue;
            }
            INFO_LOG << GetRobotId() << ": got data for " << id << Endl;
            auto session = userTagManager.BuildSession(false, true);
            auto fetchResult = userDb.FetchInfo(id, session);
            auto current = fetchResult.GetResultPtr(id);
            if (!current) {
                ERROR_LOG << GetRobotId() << ": cannot refetch " << id << ": " << session.GetStringReport() << Endl;
                continue;
            }
            auto user = *current;
            user.SetPassportNamesHash(userRegistrationManager.GetDocumentNumberHash(data.GetValue().GetNamesForHash()));
            auto upserted = userDb.UpdateUser(user, GetRobotUserId(), session);
            if (!upserted) {
                ERROR_LOG << GetRobotId() << ": cannot upsert " << id << ": " << session.GetStringReport() << Endl;
                continue;
            }
            if (!session.Commit()) {
                ERROR_LOG << GetRobotId() << ": cannot commit changed for " << id << ": " << session.GetStringReport() << Endl;
                continue;
            }
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TExpectedState TRegistratorRegular::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    if (!server->GetUserRegistrationManager()) {
        if (NotifierName) {
            NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "CRITICAL! Не сконфигурирован UserRegistrationManager");
        }
        return new IRTBackgroundProcessState();
    }

    if (!server->GetDriveAPI()->HasDocumentPhotosManager()) {
        if (NotifierName) {
            NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "CRITICAL! Не сконфигурирован DocumentPhotosManager");
        }
        return new IRTBackgroundProcessState();
    }

    const auto& registrationManager = server->GetUserRegistrationManager();
    const auto& assignmentsDB = server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments();

    auto newAssignments = assignmentsDB.GetUnfinalizedAssignments(true);

    auto actuality = Now();
    {
        TSet<TString> ids;
        for (auto&& assignment : newAssignments) {
            ids.emplace(assignment.GetLicenseBackId());
            ids.emplace(assignment.GetLicenseFrontId());
            ids.emplace(assignment.GetPassportBiographicalId());
            ids.emplace(assignment.GetPassportRegistrationId());
            ids.emplace(assignment.GetPassportSelfieId());
        }
        server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().FetchInfo(ids, actuality);
    }
    for (auto&& assignment : newAssignments) {
        try {
            if (registrationManager->HandleVerifiedAssignment(assignment, GetRobotUserId(), actuality)) {
                assignment.SetFinalizedAt(Now());
                assignment.SetComments(TVector<TString>()); // don't update comments
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!assignmentsDB.Upsert(assignment, session) || !session.Commit()) {
                    if (NotifierName) {
                        NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "Бэк не может сделать Upsert: " + assignment.GetId() + " Comments: " + assignment.SerializeToTableRecord().Get("comments"));
                    }
                    ERROR_LOG << "ASSIGNMENT FINALIZATION FAILED: unable to upsert db object" << Endl;
                }
            }
        } catch (const std::exception& e) {
            ERROR_LOG << "RegistratorRegularException during assignment " << assignment.GetId() << ": " << FormatExc(e) << Endl;
            if (NotifierName) {
                NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "CRITICAL! Словили исключение при обработке: " + assignment.GetId());
            }
        }
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRegistratorRegular::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("notifier_name", "Способ нотификации").SetVariants(server.GetNotifierNames());

    return scheme;
}

NJson::TJsonValue TRegistratorRegular::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "notifier_name", NotifierName);

    return result;
}

bool TRegistratorRegular::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "notifier_name", NotifierName);

    return true;
}
