#include "processor.h"

#include <drive/backend/abstract/settings.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TRegistrationCallResolver::TFactory::TRegistrator<TRegistrationCallResolver> TRegistrationCallResolver::Registrator(TRegistrationCallResolver::GetTypeName());

TExpectedState TRegistrationCallResolver::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    TVector<TDBTag> tags;
    if (!GetActiveTags(TagName, server, tags)) {
        ERROR_LOG << "Call resolver: GetUserIdsWithTag failed" << Endl;
        return new IRTBackgroundProcessState();
    }

    auto descr = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(TagName)->GetAs<TUserRobotCallTag::TDescription>();
    if (!descr) {
        ERROR_LOG << "Call resolver: description does not correspond to a valid TUserRobotCallTag::TDescrption" << Endl;
        return new IRTBackgroundProcessState();
    }
    auto maxWaitTime = descr->GetMaxWaitTime();

    for (auto&& tag : tags) {
        auto tagPtr = tag.MutableTagAs<TUserRobotCallTag>();
        if (!tagPtr) {
            continue;
        }

        const auto& userTagsManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();
        auto session = userTagsManager.BuildSession();
        auto optionalTagEvents = userTagsManager.GetEventsByTag(tag.GetTagId(), session);
        if (!optionalTagEvents) {
            ERROR_LOG << GetRobotId() << ": cannot GetEventsByTag " << tag.GetTagId() << ": " << session.GetStringReport() << Endl;
            continue;
        }
        const auto& tagEvents = *optionalTagEvents;
        bool isExpired = tagEvents.size() ? tagEvents.front().GetHistoryInstant() + maxWaitTime < Now() : true;
        bool toRemove = false;

        auto callStatus = server->GetDriveAPI()->GetOctopusClient()->GetCallStatus(tagPtr->GetCallSessionId());

        INFO_LOG << "Resolving call for user " << tag.GetObjectId() << Endl;

        if (callStatus != NRobotPhoneCall::ECallStatus::InProgress) {
            tagPtr->SetCallStatus(callStatus);
            if (!userTagsManager.UpdateTagData(tag, GetRobotUserId(), session)) {
                ERROR_LOG << "Unable to update tag data!" << Endl;
                continue;
            }
            toRemove = true;
        } else if (isExpired) {
            tagPtr->SetCallStatus(NRobotPhoneCall::ECallStatus::Error);
            if (!userTagsManager.UpdateTagData(tag, GetRobotUserId(), session)) {
                ERROR_LOG << "Unable to update tag data!" << Endl;
                continue;
            }
            toRemove = true;
        }

        if (toRemove) {
            if (!userTagsManager.RemoveTagSimple(tag, GetRobotUserId(), session, true) || !session.Commit()) {
                ERROR_LOG << "Unable to remove tag!" << Endl;
            }
            MaybeProceedRegistrationChat(server, tag.GetObjectId(), callStatus);
        }

    }

    return new IRTBackgroundProcessState();
}

void TRegistrationCallResolver::MaybeProceedRegistrationChat(const NDrive::IServer* server, const TString& userId, const NRobotPhoneCall::ECallStatus status) const {
    auto robot = server->GetChatRobot("registration");
    if (!robot->IsExistsForUser(userId, "")) {
        return;
    }
    TChatUserContext::TPtr context = new TChatUserContext();
    context->SetUserId(userId);
    context->SetChatId("registration");
    context->SetChatTopic("");
    NDrive::NChat::TMessage chatMessage("!call " + ToString(status), 0, NDrive::NChat::TMessage::EMessageType::Plaintext);
    robot->AcceptUserResponse(context, chatMessage, {}, userId);
}

bool TRegistrationCallResolver::GetActiveTags(const TString& tagName, const NDrive::IServer* server, TVector<TDBTag>& dbTags) const {
    auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();

    TVector<TString> tagNames;
    tagNames.push_back(tagName);

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, tagNames, dbTags, session)) {
        return false;
    }

    return true;
}

NDrive::TScheme TRegistrationCallResolver::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("tag_name", "Тег для роботного звонка").SetRequired(true);

    return scheme;
}

NJson::TJsonValue TRegistrationCallResolver::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "tag_name", TagName);

    return result;
}

bool TRegistrationCallResolver::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "tag_name", TagName);

    return true;
}
