#include "processor.h"

#include <drive/backend/abstract/notifier.h>
#include <drive/backend/abstract/settings.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/saas/api.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TRegistrationMonitoring::TFactory::TRegistrator<TRegistrationMonitoring> TRegistrationMonitoring::Registrator(TRegistrationMonitoring::GetTypeName());

TExpectedState TRegistrationMonitoring::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    auto observedUsers = server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().GetQueuedRegisteringUsers();

    // Initial check, user hasn't resubmitted anything yet
    {
        TMap<TString, ui32> photosByUser;
        auto photos = server->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetAllForUsers(observedUsers);
        for (auto&& photo : photos) {
            photosByUser[photo.GetUserId()] += 1;
        }

        TSet<TString> initialCheckUsers;
        for (auto&& it : photosByUser) {
            if (it.second == 5) {
                initialCheckUsers.insert(it.first);
            }
        }

        InitialAssignmentsSignal->Signal(initialCheckUsers.size());
        for (auto&& userId : initialCheckUsers) {
            observedUsers.erase(userId);
        }
    }

    // Documents reasked with usual flow
    {
        ui32 amtRobotReasks = CountAndCleanup({ "documents_were_reasked" }, server, observedUsers);
        RobotReaskSignal->Signal(amtRobotReasks);
    }

    // Documents were reasked by analytics processes
    {
        ui32 amtAnalyticsReasks = CountAndCleanup({ "blocked_old_license", "blocked_old_passport", "blocked_old_passport_reg" }, server, observedUsers);
        AnalyticsReaskSignal->Signal(amtAnalyticsReasks);
    }

    // Other (manual reask via admin portal)
    {
        OtherReaskSignal->Signal(observedUsers.size());
    }

    return new IRTBackgroundProcessState();
}

ui32 TRegistrationMonitoring::CountAndCleanup(const TVector<TString>& tagNames, const NDrive::IServer* server, TSet<TString>& observedUsers) const {
    TVector<TDBTag> tags;
    if (!GetActiveTags(tagNames, server, tags)) {
        ERROR_LOG << "Registration monitoring: GetUserIdsWithTag failed" << Endl;
        return 0;
    }

    ui32 amt = 0;
    for (auto&& tag : tags) {
        auto userId = tag.GetObjectId();
        auto it = observedUsers.find(userId);
        if (it != observedUsers.end()) {
            observedUsers.erase(it);
            amt += 1;
        }
    }

    return amt;
}

bool TRegistrationMonitoring::GetActiveTags(const TVector<TString>& tagNames, const NDrive::IServer* server, TVector<TDBTag>& dbTags) const {
    auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();

    if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, tagNames, dbTags, session)) {
        return false;
    }

    return true;
}
