#include "config.h"

#include <drive/backend/data/service_session.h>

#include <drive/library/cpp/mds/client.h>

TRegularPhotoReportWatcherState::TFactory::TRegistrator<TRegularPhotoReportWatcherState> TRegularPhotoReportWatcherState::Registrator(TRegularPhotoReportWatcher::GetTypeName());
TRegularPhotoReportWatcher::TFactory::TRegistrator<TRegularPhotoReportWatcher> TRegularPhotoReportWatcher::Registrator(TRegularPhotoReportWatcher::GetTypeName());

TString TRegularPhotoReportWatcherState::GetType() const {
    return TRegularPhotoReportWatcher::GetTypeName();
}

void TRegularPhotoReportWatcherState::SerializeToProto(NDrive::NProto::TCleaningReport& proto) const {
    proto.SetLastEventId(LastEventId);
    proto.SetLastSuccessfulTaskId(LastSuccessfulTaskId);
}

bool TRegularPhotoReportWatcherState::DeserializeFromProto(const NDrive::NProto::TCleaningReport& proto) {
    LastEventId = proto.GetLastEventId();
    LastSuccessfulTaskId = proto.GetLastSuccessfulTaskId();
    return true;
}

NJson::TJsonValue TRegularPhotoReportWatcherState::GetReport() const {
    NJson::TJsonValue result = TBase::GetReport();
    JWRITE(result, "last_event_id", LastEventId);
    JWRITE(result, "last_task_id", LastSuccessfulTaskId);
    return result;
}

NDrive::TScheme TRegularPhotoReportWatcherState::DoGetScheme() const {
    NDrive::TScheme result = TBase::DoGetScheme();
    result.Add<TFSNumeric>("last_event_id");
    result.Add<TFSNumeric>("last_task_id");
    return result;
}

TExpectedState TRegularPhotoReportWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const NDrive::IServer& server, TTagsModificationContext& context) const {
    auto result = MakeHolder<TRegularPhotoReportWatcherState>();

    const TRegularPhotoReportWatcherState* state = dynamic_cast<const TRegularPhotoReportWatcherState*>(stateExt.Get());
    const ui64 lastEventId = state ? state->GetLastEventId() : StartEventId;
    ui64 lastTaskId = state ? state->GetLastSuccessfulTaskId() : StartIndex;

    if (!server.GetDriveAPI()->HasMDSClient()) {
        return MakeUnexpected<TString>("Mds client not defined");
    }
    auto bucket = server.GetDriveAPI()->GetMDSClient().GetBucket(BucketName);
    if (!bucket) {
        return MakeUnexpected<TString>("Mds bucket not found '" + BucketName + "'");
    }

    auto carsInfo = context.GetFetchedCarsData();

    TMaybe<TSet<TString>> optionalFilteredUsers;
    if (!Roles.empty()) {
        TSet<TString> filteredUsers;
        if (!server.GetDriveAPI()->GetUsersData()->GetRoles().GetUsersWithAtLeastOneRoles(Roles, filteredUsers, true)) {
            return MakeUnexpected<TString>("cannot take users by roles");
        }
        optionalFilteredUsers = std::move(filteredUsers);
    }

    auto sessionBuilder = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("service", Now());
    if (!sessionBuilder) {
        ERROR_LOG << "Incorrect session builder 'service'" << Endl;
        return MakeUnexpected<TString>({});
    }
    const ui64 maxEventId = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetLockedMaxEventId();
    TVector<IEventsSession<TCarTagHistoryEvent>::TPtr> sessions = sessionBuilder->GetSessionsActualSinceId(lastEventId);
    std::sort(sessions.begin(), sessions.end(), [](IEventsSession<TCarTagHistoryEvent>::TPtr a, IEventsSession<TCarTagHistoryEvent>::TPtr b) -> bool {
        return a->GetLastEventId() < b->GetLastEventId();
    });
    TInstant currentTime = TInstant::Now();
    ui64 nextReal = lastEventId;
    size_t count = 1;
    for (const auto& serviceSession : sessions) {
        if (count > MaxSessionsCount) {
            break;
        }

        if (serviceSession->GetLastEventId() > maxEventId) {
            break;
        }

        if (serviceSession->GetClosed() && serviceSession->GetLastTS() + PhotoDelay > currentTime) {
            break;
        }

        nextReal = Max(nextReal, serviceSession->GetLastEventId());

        TServiceSession::TCompilation compilation(server.GetDriveAPI()->GetTagsManager().GetTagsMeta());
        if (!serviceSession->GetClosed() || serviceSession->GetLastEventId() <= lastEventId || !serviceSession->FillCompilation(compilation)) {
            continue;
        }

        if (!Tags.contains(compilation.GetTagName())) {
            continue;
        }

        auto carIt = carsInfo.find(serviceSession->GetObjectId());

        if (carIt == carsInfo.end()) {
            continue;
        }
        const auto& carInfo = carIt->second;

        const auto& userId = serviceSession->GetUserId();
        if (optionalFilteredUsers && !optionalFilteredUsers->contains(userId)) {
            continue;
        }
        auto userFetchResult = server.GetDriveAPI()->GetUsersData()->FetchInfo(userId);
        auto userInfo = userFetchResult.GetResultPtr(userId);
        if (!userInfo) {
            continue;
        }

        TString tagName = compilation.GetTagName();
        auto tagDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(compilation.GetTagName());
        if (!!tagDescription) {
            tagName = tagDescription->GetDisplayName();
        }

        TString lastEventInfo = "";
        TMaybe<TCarTagHistoryEvent> lastEv = serviceSession->GetLastEvent();
        if (lastEv.Defined()) {
            if (lastEv->GetHistoryAction() == EObjectHistoryAction::Remove) {
                lastEventInfo = "ВЫПОЛНЕНО(" + ToString(++lastTaskId) + ")";
            } else {
                lastEventInfo = "ОТМЕНЕНО " + lastEv->GetHistoryComment();
            }
        }

        TStringStream caption;
        caption << tagName << " " << lastEventInfo << Endl;
        caption << carInfo.GetNumber() << Endl;
        caption << compilation.GetServiceReport(&server) << Endl;
        caption << (userInfo->GetFullName() ? userInfo->GetFullName() : userInfo->GetLogin()) << Endl;
        caption << "Начало: " << serviceSession->GetStartTS().ToRfc822StringLocal() << Endl;
        caption << "Конец: " << serviceSession->GetLastTS().ToRfc822StringLocal() << Endl;

        auto images = compilation.GetImages();
        if (images.empty()) {
            NDrive::INotifier::Notify(server.GetNotifier(NotifierName), caption.Str() + "фото до/после не найдены");
        } else {
            NDrive::INotifier::Notify(server.GetNotifier(NotifierName), caption.Str());
        }

        for (const auto& image : images) {
            TString binaryImage;
            bool loadingSuccess = false;
            TMessagesCollector errors;
            if (bucket->GetFile(image.second.Path, binaryImage, errors) / 100 == 2) {
                loadingSuccess = true;
                NDrive::INotifier::TMessage message(image.second.Marker, binaryImage);
                message.SetTitle("image.jpg");
                message.SetAdditionalInfo("");
                loadingSuccess = NDrive::INotifier::SendPhoto(server.GetNotifier(NotifierName), message, NDrive::INotifier::TContext().SetServer(&server));
            }
            if (!loadingSuccess) {
                NDrive::INotifier::Notify(server.GetNotifier(NotifierName), image.second.Marker + "\nНе удалось загрузить фото " + errors.GetStringReport());
            }
        }
        count++;
    }
    result->SetLastEventId(nextReal);
    result->SetLastSuccessfulTaskId(lastTaskId);

    return result.Release();
}

NDrive::TScheme TRegularPhotoReportWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("notifier", "Нотификатор чата").SetVariants(server.GetNotifierNames()).SetRequired(true);
    scheme.Add<TFSArray>("tags", "Теги").SetElement<TFSString>().SetRequired(true);
    scheme.Add<TFSArray>("roles", "Роли").SetElement<TFSString>();
    scheme.Add<TFSArray>("departments", "Департаменты").SetElement<TFSString>();
    scheme.Add<TFSNumeric>("start_event_id").SetMin(0).SetDefault(0);
    scheme.Add<TFSNumeric>("start_index", "Стартовый номер работы(с нуля)").SetMin(0).SetDefault(0);
    scheme.Add<TFSString>("bucket_name", "Mds бакет фотографий").SetDefault("carsharing-tags").SetRequired(true);
    scheme.Add<TFSDuration>("photo_delay", "Ожидание фото").SetDefault(TDuration::Minutes(15));
    scheme.Add<TFSNumeric>("max_sessions_count").SetMin(0).SetDefault(30);
    return scheme;
}

bool TRegularPhotoReportWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    JREAD_CONTAINER(jsonInfo, "tags", Tags);
    JREAD_CONTAINER_OPT(jsonInfo, "roles", Roles);
    JREAD_CONTAINER_OPT(jsonInfo, "departments", Departments);
    JREAD_UINT_OPT(jsonInfo, "start_event_id", StartEventId);
    JREAD_UINT_OPT(jsonInfo, "start_index", StartIndex);
    JREAD_STRING(jsonInfo, "bucket_name", BucketName);
    JREAD_DURATION_OPT(jsonInfo, "photo_delay", PhotoDelay);
    JREAD_UINT_OPT(jsonInfo, "max_sessions_count", MaxSessionsCount);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRegularPhotoReportWatcher::DoSerializeToJson() const {
    NJson::TJsonValue jsonInfo = TBase::DoSerializeToJson();
    TJsonProcessor::Write(jsonInfo, "notifier", NotifierName);
    TJsonProcessor::WriteContainerArray(jsonInfo, "tags", Tags);
    TJsonProcessor::WriteContainerArray(jsonInfo, "roles", Roles);
    TJsonProcessor::WriteContainerArray(jsonInfo, "departments", Departments);
    TJsonProcessor::Write(jsonInfo, "start_event_id", StartEventId);
    TJsonProcessor::Write(jsonInfo, "start_index", StartIndex);
    TJsonProcessor::Write(jsonInfo, "bucket_name", BucketName);
    TJsonProcessor::WriteDurationString(jsonInfo, "photo_delay", PhotoDelay);
    TJsonProcessor::Write(jsonInfo, "max_sessions_count", MaxSessionsCount);
    return jsonInfo;
}
