#include "process.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/billing/accounts_manager.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/insurance/ingos/client.h>
#include <drive/backend/insurance/renins/client.h>
#include <drive/backend/insurance/task/builder.h>
#include <drive/backend/insurance/task/history.h>

#include <rtline/library/unistat/cache.h>


IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTInsuranceMessagesWatcher> TRTInsuranceMessagesWatcher::Registrator(TRTInsuranceMessagesWatcher::GetTypeName());

TExpectedState TRTInsuranceMessagesWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& driveApi = *(server.GetDriveAPI());
    TVector<TInsuranceNotification> insuranceNotifications = driveApi.GetActualInsuranceNotifications();

    auto notifier = server.GetNotifier(NotifierName);

    TVector<NDrive::TInsuranceInfo> insuranceData;
    ui64 objectsCount = 0;
    for (auto&& i : insuranceNotifications) {
        const TString& carId = i.GetCarId();
        auto policy = GetInsurancePolicy(carId, i.GetStart(), server);
        if (!policy) {
            TUnistatSignalsCache::SignalAdd("insurance-policy-" + GetRTProcessName(), "errors", 1);
            if (notifier) {
                notifier->Notify(NDrive::INotifier::TMessage("Не удалось получить полис для: [идентификатор машины:" + carId + "]. Дата поездки " + i.GetStart().ToStringLocal()));
            }
            continue;
        }

        if (policy->GetProvider() != ProviderType) {
            continue;
        }

        if (i.GetFinish() == i.GetStart()) {
            auto session = driveApi.BuildTx<NSQL::Writable>();
            if (!driveApi.CommitInsuranceNotification(i.GetNotificationId(), session)) {
                ERROR_LOG << GetRobotId() << ": cannot CommitInsuranceNotification for " << i.GetNotificationId() << ": " << session.GetStringReport() << Endl;
                continue;
            }
            if (!session.Commit()) {
                ERROR_LOG << GetRobotId() << ": cannot Commit transaction for " << i.GetNotificationId() << ": " << session.GetStringReport() << Endl;
                continue;
            }
            NDrive::TEventLog::Log("SkipEmptyNotification", NJson::ToJson(i));
            continue;
        }

        ++objectsCount;
        if (insuranceData.size() == MaxPackSize) {
            continue;
        }

        auto gCar = driveApi.GetCarsData()->FetchInfo({ carId }, TInstant::Zero());
        auto carIt = gCar.GetResult().find(carId);
        if (carIt == gCar.GetResult().end()) {
            ERROR_LOG << "Unknown car: " << carId << Endl;
            continue;
        }

        auto duration = i.GetFinish() - i.GetStart();
        if (!duration && SkipEmpty) {
            ERROR_LOG << GetRobotId() << ": empty ride " << i.GetSessionId() << Endl;
            continue;
        }

        NDrive::TInsuranceInfo info(i.GetNotificationId(), *policy);
        info.SetStartTime(i.GetStart()).SetEndTime(i.GetFinish()).SetSessionId(i.GetSessionId());
        info.SetVIN(carIt->second.GetVin()).SetCarModel(carIt->second.GetModel());
        insuranceData.emplace_back(std::move(info));
    }

    TUnistatSignalsCache::SignalAdd(::ToString(ProviderType) +  "-notifications", "count", objectsCount);

    auto session = driveApi.BuildTx<NSQL::Writable>();
    bool hasErrors = false;
    ui32 insuranceSum = 0;
    auto callback = [&driveApi, &session, &hasErrors, &insuranceSum, *this](bool isOk, const NDrive::TInsuranceInfo& orderInfo) {
        if (isOk) {
            if (driveApi.CommitInsuranceNotification(orderInfo.GetTaskId(), session)) {
                insuranceSum += orderInfo.GetCost(WithBaseCost);
                TUnistatSignalsCache::SignalAdd(::ToString(ProviderType) + "-export-commit-" + GetRTProcessName(), "ok", 1);
            } else {
                hasErrors = true;
                TUnistatSignalsCache::SignalAdd(::ToString(ProviderType) + "-export-commit-" + GetRTProcessName(), "error", 1);
            }
        } else {
            TUnistatSignalsCache::SignalAdd(::ToString(ProviderType) + "-export-" + GetRTProcessName(), "error", 1);
        }
    };

    switch(ProviderType) {
        case NDrive::EInsuranceProvider::Renins:
        {
            auto client = driveApi.GetReninsClient();
            if (!client) {
                ERROR_LOG << "No renins client configured" << Endl;
                return MakeUnexpected<TString>({});
            }
            if (PackMode) {
                client->PushDataPack(insuranceData, RequestTimeout, WithBaseCost, callback);
            } else {
                client->PushData(insuranceData, RequestsAtOnce, RequestTimeout, WithBaseCost, callback);
            }
            break;
        }
        case NDrive::EInsuranceProvider::Ingos:
        {
            auto client = driveApi.GetIngosClient();
            if (!client) {
                ERROR_LOG << "No ingos client configured" << Endl;
                return MakeUnexpected<TString>({});
            }
            client->PushData(insuranceData, RequestTimeout, WithBaseCost, callback);
            break;
        }
        case NDrive::EInsuranceProvider::Fake:
        {
            for (auto&& riding : insuranceData) {
                callback(true, riding);
            }
            break;
        }
    }

    if (hasErrors || !session.Commit()) {
        return MakeUnexpected<TString>({});
    }
    TUnistatSignalsCache::SignalAdd(::ToString(ProviderType) + "-" + GetRTProcessName(), "sum", insuranceSum);
    return stateExt;
}

NDrive::TScheme TRTInsuranceMessagesWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("request_timeout", "Таймаут клиента");
    scheme.Add<TFSNumeric>("max_pack", "Максимальный размер пакета");
    scheme.Add<TFSVariants>("provider_type", "Тип страховой").InitVariants<NDrive::EInsuranceProvider>();
    scheme.Add<TFSBoolean>("pack_mode", "Режим пакетной отправки");
    scheme.Add<TFSBoolean>("with_base_cost", "Добавлять сумму НС").SetDefault(true);
    scheme.Add<TFSBoolean>("skip_empty", "Не отсылать поезки с нулевой длительностью").SetDefault(SkipEmpty);
    scheme.Add<TFSNumeric>("requests_at_once", "Количество единовременно отправляемых запросов");
    scheme.Add<TFSVariants>("notifier_name", "Нотификатор").SetVariants(server.GetNotifierNames());
    return scheme;
}

bool TRTInsuranceMessagesWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_UINT_OPT(jsonInfo, "requests_at_once", RequestsAtOnce);
    JREAD_UINT_OPT(jsonInfo, "max_pack", MaxPackSize);
    JREAD_BOOL_OPT(jsonInfo, "pack_mode", PackMode);
    JREAD_BOOL_OPT(jsonInfo, "with_base_cost", WithBaseCost);
    JREAD_BOOL_OPT(jsonInfo, "skip_empty", SkipEmpty);
    JREAD_FROM_STRING_OPT(jsonInfo, "provider_type", ProviderType);
    JREAD_STRING_OPT(jsonInfo, "notifier_name", NotifierName);
    JREAD_DURATION_OPT(jsonInfo, "request_timeout", RequestTimeout);
    return true;
}

NJson::TJsonValue TRTInsuranceMessagesWatcher::DoSerializeToJson() const {
    NJson::TJsonValue jsonValue = TBase::DoSerializeToJson();
    JWRITE_DURATION(jsonValue, "request_timeout", RequestTimeout);
    JWRITE(jsonValue, "requests_at_once", RequestsAtOnce);
    JWRITE(jsonValue, "max_pack", MaxPackSize);
    JWRITE(jsonValue, "pack_mode", PackMode);
    JWRITE(jsonValue, "with_base_cost", WithBaseCost);
    JWRITE(jsonValue, "skip_empty", SkipEmpty);
    JWRITE(jsonValue, "provider_type", ::ToString(ProviderType));
    JWRITE_DEF(jsonValue, "notifier_name", NotifierName, "");
    return jsonValue;
}
