#include "tasks_builder.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/insurance/task/builder.h>
#include <drive/backend/insurance/task/history.h>
#include <drive/backend/insurance/task/split.h>
#include <drive/backend/saas/api.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/threading/future_cast.h>
#include <drive/library/cpp/tracks/client.h>

#include <rtline/library/unistat/cache.h>

#include <util/string/builder.h>
#include <util/string/join.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTInsuranceTasksWatcher> TRTInsuranceTasksWatcher::Registrator(TRTInsuranceTasksWatcher::GetTypeName());
TRTInsuranceBuildesState::TFactory::TRegistrator<TRTInsuranceBuildesState> TRTInsuranceBuildesState::Registrator(TRTInsuranceBuildesState::GetTypeName());

namespace {
    struct TInsuranceNotificationDetails {
        TString ObjectId;
        TString RideId;
        TString SessionId;
        TInsuranceNotification Notification;
        TAtomicSharedPtr<TCarTagHistoryEvent> Start;
        TAtomicSharedPtr<TCarTagHistoryEvent> Finish;
        NThreading::TFuture<NDrive::TTracks> Tracks;
    };

    TNamedSignalSimple SignalInsuranceTasksCount("insurance-tasks-count");
    TNamedSignalSimple SignalInsuranceTrackErrorsCount("insurance-track-errors-count");
    TNamedSignalSimple SignalRideIntervalsCount("insurance-riding-intervals-count");
    TNamedSignalHistogram SignalPartitionSize("insurance-riding-partition-size", NRTLineHistogramSignals::IntervalsRTLineReply);
}

void TRTInsuranceBuildesState::SerializeToProto(NDrive::NProto::THistoryProcessorData& proto) const {
    proto.SetLastEventId(LastEventId);
    if (LastError) {
        proto.SetLastError(LastError);
    }
}

bool TRTInsuranceBuildesState::DeserializeFromProto(const NDrive::NProto::THistoryProcessorData& proto) {
    LastEventId = proto.GetLastEventId();
    LastError = proto.GetLastError();
    return true;
}

TString TRTInsuranceBuildesState::GetType() const {
    return TRTInsuranceBuildesState::GetTypeName();
}

TExpectedState TRTInsuranceTasksWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    NDrive::TMetaTracksClient::TClients tracksClients;
    const auto tracksApi = server.GetRTLineAPI(TrackApiName);
    if (tracksApi) {
        tracksClients.push_back(MakeAtomicShared<NDrive::TTracksClient>(tracksApi->GetSearchClient()));
    } else {
        return MakeUnexpected<TString>("cannot find TrackApi: " + TrackApiName);
    }
    const auto secondaryTracksApi = SecondaryTrackApiName ? server.GetRTLineAPI(SecondaryTrackApiName) : nullptr;
    if (secondaryTracksApi) {
        tracksClients.push_back(MakeAtomicShared<NDrive::TTracksClient>(secondaryTracksApi->GetSearchClient()));
    } else if (SecondaryTrackApiName) {
        return MakeUnexpected<TString>("cannot find SecondaryTrackApi: " + SecondaryTrackApiName);
    }
    NDrive::TMetaTracksClient tracksClient(std::move(tracksClients));

    const TRTInsuranceBuildesState* state = dynamic_cast<const TRTInsuranceBuildesState*>(stateExt.Get());

    const TDeviceTagsManager& deviceTagsManager = server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    ui64 historyIdCursor = deviceTagsManager.GetHistoryManager().GetLockedMaxEventId();
    const ui64 lastEventId = state ? state->GetLastEventId() : StartEventId;

    if (historyIdCursor < lastEventId) {
        return stateExt;
    }

    if (historyIdCursor - lastEventId > MaxProcessingCount) {
        historyIdCursor = lastEventId + MaxProcessingCount;
    }

    auto sessionBuilder = deviceTagsManager.GetHistoryManager().GetSessionsBuilderSafe("billing", DataActuality);
    if (!sessionBuilder) {
        return MakeUnexpected<TString>("cannot GetSessionsBuilderSafe: billing");
    }

    NDrive::TTrackPartitionOptions trackPartitionOptions;
    trackPartitionOptions.AlwaysCutEdges = AlwaysCutEdges;
    trackPartitionOptions.Fault = FaultsInterval;
    trackPartitionOptions.Threshold = ZeroSpeedInterval;
    trackPartitionOptions.SkipEmpty = SkipEmpty;

    TVector<IEventsSession<TCarTagHistoryEvent>::TPtr> sessions = sessionBuilder->GetSessionsActualSinceId(lastEventId); // Except lastEventId
    TVector<TInsuranceNotification> notifications;
    auto addNotification = [&] (const TInsuranceNotification& notification) {
        auto duration = notification.GetFinish() - notification.GetStart();
        if (!duration) {
            NDrive::TEventLog::Log("SkipEmptyNotification", NJson::ToJson(notification));
            return;
        }
        if (SplitTimetable) {
            auto split = NDrive::Split(notification, SplitTimetable);
            notifications.insert(notifications.end(), split.begin(), split.end());
            if (split.size() > 1) {
                NDrive::TEventLog::Log("SplitIntervalByTimetable", NJson::TMapBuilder
                    ("session_id", notification.GetSessionId())
                    ("notification", NJson::ToJson(notification))
                    ("split", NJson::ToJson(split))
                );
            }
        } else {
            notifications.push_back(notification);
        }
    };

    TVector<TInsuranceNotificationDetails> deferred;
    TSet<TString> devices;
    TSet<TString> users;
    for (auto&& bSession : sessions) {
        TInsuranceTasksCompilation compilation;
        compilation.SetMaxAcceptedId(historyIdCursor).SetMinAcceptedId(Max(lastEventId, StartEventId));
        if (!bSession->FillCompilation(compilation)) {
            continue;
        }
        for (auto&& [start, finish] : compilation.GetRideIntervals()) {
            auto duration = finish->GetHistoryInstant() - start->GetHistoryInstant();
            if (!duration) {
                NDrive::TEventLog::Log("SkipEmptyRideInterval", NJson::TMapBuilder
                    ("session_id", bSession->GetSessionId())
                    ("interval", NJson::ToJson(std::tie(start, finish)))
                );
                continue;
            }
            SignalRideIntervalsCount.Signal(1);
            TInsuranceNotification notification;
            notification.SetCarId(bSession->GetObjectId()).SetUserId(bSession->GetUserId()).SetSessionId(bSession->GetSessionId());
            if (ZeroSpeedInterval == TDuration::Zero()) {
                notification.SetStart(start->GetHistoryInstant()).SetFinish(finish->GetHistoryInstant());
                addNotification(notification);
            } else {
                NDrive::TTrackQuery trackQuery;
                trackQuery.RideId = ::ToString(start->GetHistoryInstant().Seconds()) + "-" + ::ToString(start->GetHistoryAction()) + "-" + bSession->GetSessionId() + "-new";
                auto asyncTracks = tracksClient.GetTracks(trackQuery, TrackApiTimeout);

                TInsuranceNotificationDetails details;
                details.ObjectId = bSession->GetObjectId();
                details.RideId = trackQuery.RideId;
                details.SessionId = bSession->GetSessionId();
                details.Notification = notification;
                details.Start = start;
                details.Finish = finish;
                details.Tracks = asyncTracks;
                deferred.push_back(std::move(details));
            }
            devices.emplace(bSession->GetObjectId());
            users.emplace(bSession->GetUserId());
        }
    }
    if (!deferred.empty()) {
        for (auto&& details : deferred) {
            const auto& asyncTracks = details.Tracks;
            const auto& objectId = details.ObjectId;
            const auto& start = details.Start;
            const auto& finish = details.Finish;
            auto notification = details.Notification;
            {
                if (!asyncTracks.Wait(TrackApiTimeout) || !asyncTracks.HasValue() || asyncTracks.GetValue().empty()) {
                    NDrive::TEventLog::Log("GetTrackError", NJson::TMapBuilder
                        ("session_id", details.SessionId)
                        ("ride_id", details.RideId)
                        ("tracks", NJson::ToJson(asyncTracks))
                    );
                    SignalInsuranceTrackErrorsCount.Signal(1);
                    notification.SetStart(start->GetHistoryInstant()).SetFinish(finish->GetHistoryInstant());
                    addNotification(notification);
                } else {
                    const NDrive::TTracks& tracks = asyncTracks.GetValue();
                    const NDrive::TTrack& track = tracks.front();
                    auto interval = TInterval<TInstant>(start->GetHistoryInstant(), finish->GetHistoryInstant());
                    auto partition = NDrive::BuildTrackPartition(interval, track, trackPartitionOptions);
                    NDrive::TEventLog::Log("BuildTrackPartition", NJson::TMapBuilder
                        ("session_id", details.SessionId)
                        ("ride_id", details.RideId)
                        ("interval", NJson::ToJson(std::tie(start, finish)))
                        ("partition", NJson::ToJson(partition))
                        ("track_start", NJson::PointerToJson(!track.Coordinates.empty() ? &track.Coordinates.front() : nullptr))
                        ("track_finish", NJson::PointerToJson(!track.Coordinates.empty() ? &track.Coordinates.back() : nullptr))
                    );
                    SignalPartitionSize.Signal(partition.Intervals.size());
                    bool alwaysNonEmptyPartition = false;
                    if (NDrive::GetDuration(partition) == TDuration::Zero()) {
                        auto policy = GetInsurancePolicy(objectId, start->GetHistoryInstant(), server);
                        alwaysNonEmptyPartition = policy && AlwaysNonEmptyProviders.contains(policy->GetProvider());
                    }
                    if (alwaysNonEmptyPartition && NDrive::GetDuration(partition) == TDuration::Zero()) {
                        partition = NDrive::BuildSimplePartition(interval, track, trackPartitionOptions);
                        NDrive::TEventLog::Log("BuildSimplePartition", NJson::TMapBuilder
                            ("session_id", details.SessionId)
                            ("ride_id", details.RideId)
                            ("interval", NJson::ToJson(std::tie(start, finish)))
                            ("partition", NJson::ToJson(partition))
                        );
                    }
                    if (alwaysNonEmptyPartition && NDrive::GetDuration(partition) == TDuration::Zero()) {
                        partition.Intervals.emplace_back(
                            start->GetHistoryInstant(),
                            start->GetHistoryInstant() + TDuration::Seconds(1)
                        );
                        NDrive::TEventLog::Log("BuildStubPartition", NJson::TMapBuilder
                            ("session_id", details.SessionId)
                            ("ride_id", details.RideId)
                            ("interval", NJson::ToJson(std::tie(start, finish)))
                            ("partition", NJson::ToJson(partition))
                        );
                    }
                    for (auto&& interval : partition.Intervals) {
                        notification.SetStart(interval.GetMin()).SetFinish(interval.GetMax());
                        addNotification(notification);
                    }
                }
            }
        }
    }

    auto gUsers = server.GetDriveAPI()->GetUsersData()->FetchInfo(users);
    auto gDevices = server.GetDriveAPI()->GetCarsData()->FetchInfo(devices);

    NDrive::INotifier::TPtr notifier = server.GetNotifier(NotifierName);
    for (ui32 i = 0; i < notifications.size();) {
        bool fail;
        TString errorMessage = "";
        TVector<TString> messages;
        const ui32 batchSize = 100;
        for (ui32 att = 0; att < 10; ++att) {
            fail = false;
            auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
            for (ui32 j = i; j < Min<ui32>(i + batchSize, notifications.size()); ++j) {
                const TInsuranceNotification& n = notifications[j];
                if (!server.GetDriveAPI()->InsuranceNotification(n.GetSessionId(), n.GetStart(), n.GetFinish(), n.GetUserId(), n.GetCarId(), GetRobotUserId(), session)) {
                    ERROR_LOG << GetRobotId() << ": " << session.GetStringReport() << Endl;
                    errorMessage = session.GetStringReport();
                    fail = true;
                    break;
                }
                const TDriveUserData* userData = gUsers.GetResultPtr(n.GetUserId());
                const TDriveCarInfo* deviceData = gDevices.GetResultPtr(n.GetCarId());
                TString message = TStringBuilder()
                                  << (userData ? userData->GetHRReport() : n.GetUserId()) << ": "
                                  << (deviceData ? deviceData->GetHRReport() : n.GetCarId()) << " (" << n.GetStart() << "-" << n.GetFinish() << ")";
                messages.push_back(message);
            }
            if (!DryRun && !session.Commit()) {
                fail = true;
            }
            if (fail) {
                errorMessage = session.GetStringReport();
            } else {
                break;
            }
        }
        if (fail) {
            if (notifier) {
                notifier->Notify(NDrive::INotifier::TMessage("Не могу построить задания для отправки в страховую", errorMessage));
            } else {
                ERROR_LOG << GetRobotId() << ": " << "Не могу построить задания для отправки в страховую: " << errorMessage << Endl;
            }
            return MakeUnexpected<TString>(std::move(errorMessage));
        } else {
            SignalInsuranceTasksCount.Signal(messages.size());
            if (notifier) {
                NDrive::INotifier::MultiLinesNotify(notifier, "Задания для отправки в страховую", messages);
            } else {
                INFO_LOG << GetRobotId() << ": " << "Задания для отправки в страховую: " << JoinSeq("\r\n", messages) << Endl;
            }
        }
        i = i + batchSize;
    }

    if (DryRun) {
        return MakeUnexpected<TString>("DryRun");
    }
    auto result = MakeHolder<TRTInsuranceBuildesState>();
    result->SetLastEventId(historyIdCursor);
    return result;
}

NDrive::TScheme TRTInsuranceTasksWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("always_cut_edges", "Всегда резать границы поездки");
    scheme.Add<TFSVariants>("always_non_empty_providers", "DRIVEBACK-2544").InitVariants<NDrive::EInsuranceProvider>().SetMultiSelect(true);
    scheme.Add<TFSNumeric>("zero_speed_interval", "Минимальный интервал с нулевой скоростью (сек)");
    scheme.Add<TFSNumeric>("faults_interval", "Интервал игнорирования сигнала (сек)");
    scheme.Add<TFSBoolean>("skip_empty", "Игнорировать пустые интервалы").SetDefault(false);
    scheme.Add<TFSNumeric>("start_event_id", "Минимальный id для конца ride-интервала").SetDefault(0);
    scheme.Add<TFSNumeric>("max_processing_count", "Максимальный интервал id для единовременной обработки").SetDefault(50000);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSString>("track_api_name", "Сервис с треками");
    scheme.Add<TFSString>("secondary_track_api_name", "Запасной сервис с треками");
    scheme.Add<TFSNumeric>("track_api_timeout", "Таймаут к сервису с треками");
    scheme.Add<TFSBoolean>("dry_run", "Холостой режим");
    scheme.Add<TFSJson>("split_timetable", "Расписание 'Ночного режима' DRIVEBACK-3242");
    return scheme;
}

bool TRTInsuranceTasksWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_STRING_OPT(jsonInfo, "notifier", NotifierName);
    JREAD_UINT_OPT(jsonInfo, "start_event_id", StartEventId);
    JREAD_UINT_OPT(jsonInfo, "max_processing_count", MaxProcessingCount);
    JREAD_BOOL(jsonInfo, "dry_run", DryRun);
    JREAD_DURATION_OPT(jsonInfo, "zero_speed_interval", ZeroSpeedInterval);
    JREAD_BOOL_OPT(jsonInfo, "skip_empty", SkipEmpty);
    JREAD_DURATION_OPT(jsonInfo, "faults_interval", FaultsInterval);
    JREAD_STRING(jsonInfo, "track_api_name", TrackApiName);
    JREAD_STRING_OPT(jsonInfo, "secondary_track_api_name", SecondaryTrackApiName);
    JREAD_DURATION_OPT(jsonInfo, "track_api_timeout", TrackApiTimeout);
    return
        NJson::ParseField(jsonInfo["split_timetable"], SplitTimetable) &&
        NJson::ParseField(jsonInfo["always_non_empty_providers"], AlwaysNonEmptyProviders) &&
        NJson::ParseField(jsonInfo["always_cut_edges"], AlwaysCutEdges)
        ;
}

NJson::TJsonValue TRTInsuranceTasksWatcher::DoSerializeToJson() const {
    NJson::TJsonValue jsonValue = TBase::DoSerializeToJson();
    NJson::InsertField(jsonValue, "always_cut_edges", AlwaysCutEdges);
    NJson::InsertField(jsonValue, "always_non_empty_providers", AlwaysNonEmptyProviders);
    NJson::InsertNonNull(jsonValue, "split_timetable", SplitTimetable);
    JWRITE_DURATION(jsonValue, "zero_speed_interval", ZeroSpeedInterval);
    JWRITE_DURATION(jsonValue, "faults_interval", FaultsInterval);
    JWRITE(jsonValue, "skip_empty", SkipEmpty);
    JWRITE(jsonValue, "start_event_id", StartEventId);
    JWRITE(jsonValue, "max_processing_count", MaxProcessingCount);
    JWRITE(jsonValue, "notifier", NotifierName);
    JWRITE(jsonValue, "dry_run", DryRun);
    JWRITE(jsonValue, "track_api_name", TrackApiName);
    JWRITE(jsonValue, "secondary_track_api_name", SecondaryTrackApiName);
    JWRITE_DURATION(jsonValue, "track_api_timeout", TrackApiTimeout);
    return jsonValue;
}
