#include "process.h"

#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/database/transaction/tx.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/rental/rental_offer_holder_tag.h>
#include <drive/backend/offers/actions/rental_offer.h>
#include <drive/backend/tags/tags_manager.h>

TExpectedState TStartRentalProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    TStartRentalProcess::Execute(server, GetRobotUserId(), StartRentalTimeout);
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

void TStartRentalProcess::Execute(const NDrive::IServer& server, const TString& userId, TDuration threshold) {
    const auto driveApi = Yensured(server.GetDriveAPI());

    TUserPermissions::TPtr permissions = nullptr;
    TMaybe<TVector<TDBTag>> rentalOfferHolderTags;
    {
        auto session = driveApi->BuildTx<NSQL::ReadOnly>();
        auto optionalTags = driveApi->GetTagsManager().GetTraceTags().RestoreTagsRobust({}, { TChargableSessionTag::Type() }, session);
        R_ENSURE(optionalTags, {}, "cannot restore session tags", session);

        TSet<TString> rentalOfferTagIds;

        for (auto& tag: *optionalTags) {
            auto pTag = tag.GetTagAs<TChargableSessionTag>();
            if (pTag && NEntityTagsManager::EEntityType::User == pTag->GetSessionTagEntityType()) {
                rentalOfferTagIds.insert(ToString(tag.GetTagAs<TChargableSessionTag>()->GetSessionTagId()));
            }
        }

        permissions = driveApi->GetUserPermissions(userId);
        R_ENSURE(permissions, {}, "cannot get permissions for user robot id: " + userId, session);
        rentalOfferHolderTags = driveApi->GetTagsManager().GetUserTags().RestoreTags(rentalOfferTagIds, session);
        R_ENSURE(rentalOfferHolderTags, {}, "cannot get rental offer holder tags, user robot id: " + userId, session);
    }

    auto now = TInstant::Now();

    for (auto& tag: *rentalOfferHolderTags) {
        if (auto pTag = std::dynamic_pointer_cast<TRentalOfferHolderTag>(tag.GetData())) {
            if (auto offer = std::dynamic_pointer_cast<TRentalOffer>(pTag->GetOffer())) {
                if (offer->HasStatus() && TRentalOfferHolderTag::GetPaidOfferStatus() == ToLowerUTF8(offer->GetStatusRef())) {
                    auto offerStartRentalTimeout = offer->HasStartRentalTimeout() ? offer->GetStartRentalTimeoutRef() : threshold;
                    if (offer->GetSince() > now - offerStartRentalTimeout) {

                        auto tx = driveApi->BuildTx<NSQL::Writable>();
                        const auto offerLock = TRentalOfferHolderTag::LockOffer(offer, tx);

                        if (!offerLock) {
                            ERROR_LOG << userId << ": " << "TryLock failed: rental_offer_" << offer->GetOfferId() << Endl;
                            continue;
                        } else if (!*offerLock) {
                            WARNING_LOG << userId << ": " << "rental_offer_" << offer->GetOfferId() << " is already locked" << Endl;
                            continue;
                        }

                        // check offer removed/updated
                        {
                            auto restoredRentalOfferHolderTag = driveApi->GetTagsManager().GetUserTags().RestoreTag(tag.GetTagId(), tx);
                            if (!restoredRentalOfferHolderTag) {
                                ERROR_LOG << userId << ": " << "cannot restore rental_offer " << offer->GetOfferId() << Endl;
                                continue;
                            } else if (!*restoredRentalOfferHolderTag) {
                                WARNING_LOG << userId << ": " << "rental_offer " << offer->GetOfferId() << " is removed" << Endl;
                                continue;
                            }

                            bool startRentalConditionsChanged = true;
                            if (auto pRestoredTag = std::dynamic_pointer_cast<TRentalOfferHolderTag>(restoredRentalOfferHolderTag->GetData())) {
                                if (auto restoredOffer = std::dynamic_pointer_cast<TRentalOffer>(pTag->GetOffer())) {
                                    if (restoredOffer->HasStatus() && TRentalOfferHolderTag::GetPaidOfferStatus() == restoredOffer->GetStatusRef()) {
                                        offerStartRentalTimeout = offer->HasStartRentalTimeout() ? offer->GetStartRentalTimeoutRef() : threshold;
                                        if (restoredOffer->GetSince() > now - offerStartRentalTimeout) {
                                            startRentalConditionsChanged = false;
                                        }
                                    }
                                }
                            }

                            if (startRentalConditionsChanged) {
                                WARNING_LOG << userId << " start rental conditions changed " << offer->GetOfferId() << Endl;
                                continue;
                            }
                        }

                        if (!TRentalOfferHolderTag::Start(tag, *permissions, &server, tx, "StartRentalProcess::DoExecute", offer->GetObjectId())) {
                            ERROR_LOG << userId << " cannot start rental " << offer->GetOfferId() << Endl;
                            continue;
                        }

                        if (!tx.Commit()) {
                            ERROR_LOG << userId << " cannot commit transaction " << offer->GetOfferId() << Endl;
                        }
                    }
                }
            }
        }
    }
}

NDrive::TScheme TStartRentalProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSDuration>("start_rental_timeout").SetDefault(StartRentalTimeout);
    return scheme;
}

bool TStartRentalProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return NJson::ParseField(value["start_rental_timeout"], StartRentalTimeout);
}

NJson::TJsonValue TStartRentalProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["start_rental_timeout"] = NJson::ToJson(NJson::Hr(StartRentalTimeout));
    return result;
}

TStartRentalProcess::TFactory::TRegistrator<TStartRentalProcess> TStartRentalProcess::Registrator(TStartRentalProcess::GetTypeName());
