#include "process.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/salary.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/yt/common/writer.h>

#include <rtline/library/unistat/cache.h>


IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTFuelingSalaryWatcher> TRTFuelingSalaryWatcher::Registrator(TRTFuelingSalaryWatcher::GetTypeName());
TRTHistoryWatcherState::TFactory::TRegistrator<TRTFuelingSalaryState> TRTFuelingSalaryState::Registrator(TRTFuelingSalaryWatcher::GetTypeName());

TString TRTFuelingSalaryState::GetType() const {
    return TRTFuelingSalaryWatcher::GetTypeName();
}

TExpectedState TRTFuelingSalaryWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    auto result = MakeHolder<TRTFuelingSalaryState>();

    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const TRTFuelingSalaryState* state = dynamic_cast<const TRTFuelingSalaryState*>(stateExt.Get());

    const TUserTagsManager& userTagsManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
    const NDrive::NBilling::TAccountsManager& accountsManager = server.GetDriveAPI()->GetBillingManager().GetAccountsManager();

    const ui64 historyIdCursor = userTagsManager.GetHistoryManager().GetLockedMaxEventId();
    const ui64 lastEventId = state ? state->GetLastEventId() : 0;
    ui64 nextLastEventId = lastEventId;

    auto registeredTags = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags();
    auto session = userTagsManager.BuildSession();
    auto optionalEvents = userTagsManager.GetEventsSince(lastEventId + 1, session, 10000);
    if (!optionalEvents) {
        return MakeUnexpected("cannot GetEventsSince: " + session.GetStringReport());
    }
    if (!session.Rollback()) {
        ERROR_LOG << GetRobotId() << ": cannot rollback: " << session.GetStringReport() << Endl;
    }

    auto notifier = server.GetNotifier(NotifierName);

    ui32 salaryTags = 0;
    ui32 salarySum = 0;
    ui32 fineSum = 0;

    try {
        auto ytClient = NYT::CreateClient(YTCluster);
        TYTWritersSet<TTableSelectorDay> writers(ytClient, YTDir + "/transactions");
        TYTWritersSet<TTableSelectorDay> refunds(ytClient, YTDir + "/refunds");

        for (auto&& tagEvent : *optionalEvents) {
            if (tagEvent.GetHistoryEventId() > historyIdCursor) {
                break;
            }

            nextLastEventId = Max<ui64>(nextLastEventId, tagEvent.GetHistoryEventId());

            if (!SalaryTags.contains(tagEvent->GetName())) {
                continue;
            }

            if (tagEvent.GetHistoryAction() != EObjectHistoryAction::Remove) {
                continue;
            }

            if (tagEvent.GetHistoryUserId() != GetRobotUserId()) {
                continue;
            }

            const TOperationTag* operationTag = tagEvent.GetTagAs<const TOperationTag>();
            if (!operationTag) {
                ERROR_LOG << "Incorrect tag type: " << tagEvent->GetName() << Endl;
                continue;
            }

            TInstant ts = operationTag->GetOperationTs() == TInstant::Zero() ? tagEvent.GetHistoryInstant() : operationTag->GetOperationTs();

            if (ts < FirstInstant) {
                continue;
            }

            auto descIt = registeredTags.find(tagEvent->GetName());
            if (descIt == registeredTags.end()) {
                ERROR_LOG << "Can't find description for: " << tagEvent->GetName() << Endl;
                continue;
            }

            auto billingDesc = std::dynamic_pointer_cast<const TBillingTagDescription>(descIt->second);


            ++salaryTags;
            const TString& userId = tagEvent.TConstDBTag::GetObjectId();
            for (auto account : accountsManager.GetUserAccounts(userId)) {
                if (Yensured(billingDesc)->GetAvailableAccounts().contains(account->GetUniqueName())) {
                    auto salaryAccount = account->GetRecordAs<NDrive::NBilling::TSalaryAccountRecord>();
                    if (!salaryAccount) {
                        TUnistatSignalsCache::SignalAdd("salary-export-" + GetRTProcessName(), "error", 1);
                        result->SetLastEventId(lastEventId);
                        result->SetLastError("Can't find account for " + userId);
                        if (notifier) {
                            notifier->Notify(NDrive::INotifier::TMessage("Can't find account for " + userId));
                        }
                        return result.Release();
                    }
                    if (!salaryAccount->GetClientId()) {
                        TUnistatSignalsCache::SignalAdd("salary-export-" + GetRTProcessName(), "error", 1);
                        result->SetLastEventId(lastEventId);
                        result->SetLastError("Can't find client_id for " + userId);
                        if (notifier) {
                            notifier->Notify(NDrive::INotifier::TMessage("Can't find client_id for " + userId));
                        }
                        return result.Release();
                    }
                    ui32 amount = operationTag->GetAmount();
                    // Correct because of OEBS max delay.
                    if (ts < StartInstant - TDuration::Hours(20)) {
                        ts = StartInstant;
                    }
                    if (billingDesc->GetAction() == TBillingTagDescription::EAction::Debit) {
                        salarySum += amount;
                        TUnistatSignalsCache::SignalAdd("salary-export-" + GetRTProcessName(), "ok", amount);
                        NYT::TNode resultNode = NBillingExports::CreateSalaryNode(tagEvent.GetTagId(), tagEvent.GetTagId(), *this, salaryAccount->GetClientId(), ts, amount);
                        writers.GetWriter(StartInstant)->AddRow(resultNode);
                    } else if (billingDesc->GetAction() == TBillingTagDescription::EAction::Credit) {
                        fineSum += amount;
                        TUnistatSignalsCache::SignalAdd("salary-export-refunds-" + GetRTProcessName(), "ok", amount);
                        NYT::TNode resultNode = NBillingExports::CreateRefundNode(tagEvent.GetTagId(), tagEvent.GetTagId(), salaryAccount->GetClientId(), amount, GetBalancePaymentType(), ts);
                        refunds.GetWriter(StartInstant)->AddRow(resultNode);
                    }
                    break;
                }
            }
        }
        writers.Finish();
        refunds.Finish();
    } catch (const std::exception& e) {
        ERROR_LOG << "YTError " << FormatExc(e) << Endl;
        result->SetLastEventId(lastEventId);
        result->SetLastError("YTError " + FormatExc(e));
        return result.Release();
    }

    if (notifier && salaryTags > 0) {
        TStringStream ss;
        ss << "Save salary data. Tags: " << salaryTags << ". Sum: " << salarySum * 0.01 << ". Fines: " << fineSum * 0.01;
        notifier->Notify(NDrive::INotifier::TMessage(ss.Str()));
    }

    TUnistatSignalsCache::SignalLastX("salary-export-" + GetRTProcessName(), "error", 0);
    result->SetLastEventId(nextLastEventId);
    return result.Release();
}

NDrive::TScheme TRTFuelingSalaryWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSArray>("salary_tags", "Теги для обработки").SetElement<TFSString>();
    scheme.Add<TFSString>("yt_cluster", "Кластер YT");
    scheme.Add<TFSString>("notifier", "Телеграм для нотификации");
    scheme.Add<TFSString>("yt_dir", "Директория для экспорта (в ней должны быть transactions и refunds");
    scheme.Add<TFSNumeric>("start_instant", "Начало периода начислений").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSString>("balance_product", "BalanceProduct");
    scheme.Add<TFSString>("balance_service_id", "BalanceServiceID");
    scheme.Add<TFSString>("balance_partner_id", "BalancePartnerID");
    scheme.Add<TFSString>("balance_payment_type", "BalancePaymentType");
    return scheme;
}

bool TRTFuelingSalaryWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_INSTANT_OPT(jsonInfo, "start_instant", FirstInstant)
    JREAD_STRING(jsonInfo, "yt_cluster", YTCluster);
    JREAD_STRING(jsonInfo, "yt_dir", YTDir);
    JREAD_STRING_OPT(jsonInfo, "notifier", NotifierName);
    JREAD_STRING(jsonInfo, "balance_product", Product);
    JREAD_STRING(jsonInfo, "balance_service_id", ServiceID);
    JREAD_STRING(jsonInfo, "balance_partner_id", PartnerID);
    JREAD_STRING(jsonInfo, "balance_payment_type", PaymentType);
    for (auto&& tag : jsonInfo["salary_tags"].GetArraySafe()) {
        SalaryTags.insert(tag.GetString());
    }
    return true;
}

NJson::TJsonValue TRTFuelingSalaryWatcher::DoSerializeToJson() const {
    NJson::TJsonValue tagsJson(NJson::JSON_ARRAY);
    for (auto&& i : SalaryTags) {
        tagsJson.AppendValue(i);
    }

    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["salary_tags"] = tagsJson;
    result["yt_dir"] = YTDir;
    result["yt_cluster"] = YTCluster;
    result["notifier"] = NotifierName;
    result["balance_product"] = Product;
    result["balance_service_id"] = ServiceID;
    result["balance_partner_id"] = PartnerID;
    result["balance_payment_type"] = PaymentType;
    result["start_instant"] = FirstInstant.Seconds();
    return result;
}
