#include "common.h"

#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/surge/surge_snapshot.h>

#include <drive/telematics/api/sensor/interface.h>

void TSensorsHelper::PatchScheme(NDrive::TScheme& scheme) const {
    scheme.Add<TFSNumeric>("sensor_id", "Идентификатор сенсора");
    scheme.Add<TFSNumeric>("sensor_sub_id", "Вторичный идентификатор");
}

bool TSensorsHelper::DeserializeFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "sensor_id", SensorId, /* required = */ true)
        && NJson::ParseField(json, "sensor_sub_id", SensorSubId, /* required = */ false);
}

void TSensorsHelper::PatchJson(NJson::TJsonValue& json) const {
    NJson::InsertField(json, "sensor_id", SensorId);
    NJson::InsertField(json, "sensor_sub_id", SensorSubId);
}

TSensorsHelper::TOptCarValues TSensorsHelper::GetSensorValues(const IRTCarsProcess::TTagsModificationContext& context, const bool useSensorApi, const TDuration freshness, const TInstant startTS) const {
    const NDrive::IServer& server = context.GetServer();
    typename TOptCarValues::TValueType sensorValues;
    if (SensorId < IDLE_TIME_AS_SENSOR) {
        auto snapshots = server.GetSnapshotsManager().GetSnapshotsVector(context.GetFilteredCarIds());
        const NDrive::TSensorId id(SensorId, SensorSubId);

        NDrive::ISensorApi::TSensors sensors;
        if (useSensorApi) {
            const auto* sensorApi = server.GetSensorApi();
            if (!sensorApi) {
                return {};
            }
            sensors = sensorApi->GetSensors(server.GetDriveAPI()->GetIMEIs(MakeVector(context.GetFilteredCarIds()), context.GetFetchedCarsData()), { id }, {}, 3).ExtractValueSync();
        }

        for (auto&& i : snapshots.GetSelection()) {
            auto sensor = i.second.GetSensor(id, freshness);
            if (sensor) {
                try {
                    double value = sensor->ConvertTo<double>();
                    sensorValues.emplace(i.first, value);
                } catch (...) {
                    continue;
                }
            } else if (useSensorApi) {
                TString imei = server.GetDriveAPI()->GetIMEI(i.first, context.GetFetchedCarsData());
                if (!imei) {
                    continue;
                }
                auto itSensor = sensors.find(imei);
                if (itSensor == sensors.end() || itSensor->second.empty() || itSensor->second.size() != 1 || itSensor->second.front().GetSinceDelta(startTS) > freshness) {
                    continue;
                }
                try {
                    double value = itSensor->second.front().ConvertTo<double>();
                    sensorValues.emplace(i.first, value);
                } catch (...) {
                    continue;
                }
            }
        }
    } else if (SensorId == IDLE_TIME_AS_SENSOR) {
        const TInstant now = ModelingNow();
        auto surgeInfos = server.GetSurgeConstructor()->GetInfoActual();
        for (auto&& i : context.GetFilteredCarIds()) {
            auto it = surgeInfos.find(i);
            if (it != surgeInfos.end() && it->second.HasIdleStart()) {
                sensorValues.emplace(i, (now - it->second.GetIdleStartUnsafe()).Seconds());
            }
        }
    }
    return sensorValues;
}

void TNotifyHelper::PatchScheme(const IServerBase& server, NDrive::TScheme& scheme) const {
    scheme.Add<TFSVariants>("notifier", "Нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("report_cars", "Информация о конкретных машинах").SetDefault(true);
}

bool TNotifyHelper::DeserializeFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "notifier", NotifierName, /* required = */ false)
        && NJson::ParseField(json, "report_cars", ReportCarsList, /* required = */ false);
}

void TNotifyHelper::PatchJson(NJson::TJsonValue& json) const {
    NJson::InsertField(json, "notifier", NotifierName);
    NJson::InsertField(json, "report_cars", ReportCarsList);
}

void TNotifyHelper::Notify(const NDrive::IServer& server, const TVector<TString>& lines, const TString& header /* = "" */, const bool force /* = false */) const {
    if (lines.size() && (ReportCarsList || force)) {
        NDrive::INotifier::MultiLinesNotify(server.GetNotifier(NotifierName), header ? header : DefaultNotifyHeader, lines);
    } else if (header) {
        NDrive::INotifier::Notify(server.GetNotifier(GetNotifierName()), header);
    }
}
