#include "config.h"


TRTSensorToTagsWatcher::TFactory::TRegistrator<TRTSensorToTagsWatcher> TRTSensorToTagsWatcher::Registrator(TRTSensorToTagsWatcher::GetTypeName());
const TSet<IRTSensorToTagsWatcher::EAbility> IRTSensorToTagsWatcher::FullAbilitiesSet = {IRTSensorToTagsWatcher::EAbility::Add, IRTSensorToTagsWatcher::EAbility::Evolve, IRTSensorToTagsWatcher::EAbility::Remove};

namespace {
    static const TString RemoveTagConstant = "$remove$";
    static const TString NoActionConstant = "$ignore$";
}

class TNewTagFeatures {
    R_FIELD(TString, Name);
    R_FIELD(i32, Priority, 0);
    R_READONLY(TVector<TDBTag>, ForRemove);
    R_READONLY(TVector<TDBTag>, AvailableForRemove);
    R_READONLY(TDBTag, ForEvolution);
    R_READONLY(TString, CarInfo);
public:
    TNewTagFeatures() = default;
    TNewTagFeatures(const TString& name, const i32 priority, const TVector<TDBTag>& tags, const TString& carInfo)
        : Name(name)
        , Priority(priority)
        , CarInfo(carInfo)
    {
        if (name == RemoveTagConstant) {
            ForRemove = tags;
        } else if (tags.size()) {
            Split(tags, ForEvolution, ForRemove);
        }
        for (auto&& i : ForRemove) {
            if (!i->GetPerformer()) {
                AvailableForRemove.emplace_back(i);
            }
        }
    }

    TString GetId() const {
        return Name + (Priority ? ("(" + ::ToString(Priority) + ")") : "");
    }

    i32 GetBlocksCount() const {
        i32 maxPriorityRemove = 0;
        for (auto&& i : ForRemove) {
            maxPriorityRemove = Max(maxPriorityRemove, i->GetTagPriority(0));
        }
        if (ForEvolution.HasData()) {
            maxPriorityRemove = Max(maxPriorityRemove, ForEvolution->GetTagPriority(0));
        }
        if (GetPriority() > 0) {
            if (maxPriorityRemove <= 0) {
                return 1;
            }
        } else if (maxPriorityRemove > 0) {
            return -1;
        }
        return 0;
    }

    bool IsSame(const TDBTag& tag) const {
        if (!tag.HasData()) {
            return false;
        }
        return tag->GetName() == Name && tag->GetTagPriority(0) == Priority;
    }

    ITag::TPtr Construct(const NDrive::IServer& server) const {
        auto d = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(GetName());
        if (d) {
            ITag::TPtr result = ITag::TFactory::Construct(d->GetType());
            result->SetName(GetName());
            result->SetTagPriority(GetPriority());
            return result;
        }
        return nullptr;
    }
private:
    void Split(const TVector<TDBTag>& tags, TDBTag& evolveTag, TVector<TDBTag>& tagsForRemove) {
        TVector<TDBTag> performed;
        TVector<TDBTag> free;
        for (auto&& tag : tags) {
            if (tag->GetPerformer()) {
                performed.emplace_back(tag);
            } else {
                free.emplace_back(tag);
            }
        }
        if (performed.empty()) {
            SplitSimple(free, evolveTag, tagsForRemove);
        } else {
            SplitSimple(performed, evolveTag, tagsForRemove);
            tagsForRemove.insert(tagsForRemove.end(), free.begin(), free.end());
        }
    }

    void SplitSimple(const TVector<TDBTag>& tags, TDBTag& evolveTag, TVector<TDBTag>& tagsForRemove) {
        for (auto&& i : tags) {
            if (!evolveTag.HasData()) {
                evolveTag = i;
            } else if (evolveTag->GetName() == GetName()) {
                if (Abs(evolveTag->GetTagPriority(0) - GetPriority()) > Abs(i->GetTagPriority(0) - GetPriority())) {
                    tagsForRemove.emplace_back(evolveTag);
                    evolveTag = i;
                } else {
                    tagsForRemove.emplace_back(i);
                }
            } else if (i->GetName() == GetName()) {
                tagsForRemove.emplace_back(evolveTag);
                evolveTag = i;
            } else {
                tagsForRemove.emplace_back(i);
            }
        }
    }
};


bool IRTSensorToTagsWatcher::GetTagByValue(const double value, TString& tagName, i32& priority) const {
    for (ui32 i = 0; i < CriticalValues.size(); ++i) {
        if (value < CriticalValues[i]) {
            tagName = Tags[i];
            priority = Priorities[i];
            return true;
        }
    }
    if (Tags.size() > CriticalValues.size()) {
        tagName = Tags[CriticalValues.size()];
        priority = Priorities[CriticalValues.size()];
        return true;
    }
    return false;
}

TExpectedState IRTSensorToTagsWatcher::SensorWatcherDoExecute(TSensorWatcherContext& context) const {
    const NDrive::IServer& server = context.GetRTContext().GetServer();
    if (Abilities.empty()) {
        return new IRTBackgroundProcessState;
    }

    auto sensorValues = SensorHelper.GetSensorValues(context.GetRTContext(), UseSensorApi, GetFreshness(), TBase::StartInstant);
    if (!sensorValues) {
        return MakeUnexpected<TString>("no SensorApi");
    }

    TMap<TString, TVector<TDBTag>> tagsByDevice;
    TSet<TString> blocked;
    {
        TVector<TDBTag> dbTags;
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(context.GetFilteredCarIds(), Tags, dbTags, session)) {
            return MakeUnexpected<TString>("cannot RestoreTags: " + session.GetStringReport());
        }
        for (auto&& i : dbTags) {
            tagsByDevice[i.GetObjectId()].emplace_back(i);
            if (i->GetTagPriority(0) > 0) {
                blocked.emplace(i.GetObjectId());
            }
        }
    }

    TMap<TString, TNewTagFeatures> tagToBe;
    for (auto&& i : *sensorValues) {
        const double value = i.second;
        TString tagName;
        TString tagNameUp;
        TString tagNameDown;
        i32 priority = 0;
        i32 priorityUp = 0;
        i32 priorityDown = 0;
        if (!GetTagByValue(value + PrecisionUp, tagNameUp, priorityUp) || !GetTagByValue(value + PrecisionDown, tagNameDown, priorityDown) || !GetTagByValue(value, tagName, priority)) {
            continue;
        }
        if (priority != priorityDown || priorityDown != priorityUp) {
            continue;
        }
        if (tagName != tagNameUp || tagName != tagNameDown) {
            continue;
        }
        if (tagName == NoActionConstant) {
            continue;
        }
        auto result = CheckDevice(i.first, value, context);
        if (!result) {
            return MakeUnexpected<TString>("cannot CheckDevice: " + i.first);
        }
        switch (*result) {
            case ESpecialAction::Ignore:
                continue;
            case ESpecialAction::Remove:
                tagName = RemoveTagConstant;
            case ESpecialAction::Add:
                break;
        }
        auto it = tagsByDevice.find(i.first);
        TVector<TDBTag> tags = (it == tagsByDevice.end()) ? TVector<TDBTag>() : it->second;
        TString carInfo = i.first;
        const TDriveCarInfo* dci = context.GetFetchedCarsData(i.first);
        if (dci) {
            carInfo = dci->GetHRReport();
        }
        TNewTagFeatures newTagInfo(tagName, priority, tags, carInfo);
        if (newTagInfo.GetBlocksCount() == -1) {
            blocked.erase(i.first);
        }
        tagToBe.emplace(i.first, std::move(newTagInfo));
    }
    const ui32 blockedLimit = GetBlocksCountLimit() ? GetBlocksCountLimit() : Max<ui32>();
    if (Abilities.contains(EAbility::Remove)) {
        TVector<TString> lines;
        {
            for (auto&& i : tagToBe) {
                auto& list = GetModifyPerformed() ? i.second.GetForRemove() : i.second.GetAvailableForRemove();
                if (!GetDryRunMode()) {
                    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RemoveTags(list, GetRobotUserId(), &server, session, GetModifyPerformed(), true)) {
                        return MakeUnexpected<TString>("cannot RemoveTags: " + session.GetStringReport());
                    }
                    if (!session.Commit()) {
                        return MakeUnexpected<TString>("cannot commit: " + session.GetStringReport());
                    }
                }
                for (auto&& dbTag : list) {
                    lines.emplace_back(i.second.GetCarInfo() + ": удален " + dbTag->GetName());
                }
            }
        }
        NotifyHelper.Notify(server, lines, GetRTProcessName() + "(-" + ::ToString(lines.size()) + ")");
    }

    if (Abilities.contains(EAbility::Add) || Abilities.contains(EAbility::Evolve)) {
        TMap<TString, ui32> addTags;
        TVector<TString> lines;
        {
            for (auto&& i : tagToBe) {
                const TNewTagFeatures& newTagInfo = i.second;
                if (newTagInfo.GetName() == RemoveTagConstant || !newTagInfo.GetName()) {
                    continue;
                }
                ITag::TPtr tag = newTagInfo.Construct(server);
                if (!tag) {
                    lines.emplace_back(newTagInfo.GetCarInfo() + ": не могу создать " + newTagInfo.GetId());
                    continue;
                }
                if (newTagInfo.GetBlocksCount() == 1) {
                    if (blocked.size() >= blockedLimit && !blocked.contains(i.first)) {
                        lines.emplace_back(newTagInfo.GetCarInfo() + ": должен быть добавлен " + newTagInfo.GetId());
                        ++addTags["Ожидается " + newTagInfo.GetId()];
                        continue;
                    } else {
                        blocked.emplace(i.first);
                    }
                }
                if (GetDryRunMode()) {
                    continue;
                }
                auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!newTagInfo.GetForEvolution().HasData()) {
                    if (!Abilities.contains(EAbility::Add)) {
                        continue;
                    }
                    ++addTags["Добавлено " + newTagInfo.GetId()];
                    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().AddTag(tag, GetRobotUserId(), i.first, &server, session, EUniquePolicy::SkipIfExists)) {
                        return MakeUnexpected<TString>("cannot AddTag: " + session.GetStringReport());
                    }
                    lines.emplace_back(i.second.GetCarInfo() + ": добавлен " + i.second.GetId());
                } else if (!newTagInfo.IsSame(newTagInfo.GetForEvolution())) {
                    if (!Abilities.contains(EAbility::Evolve)) {
                        continue;
                    }
                    auto evolveTag = newTagInfo.GetForEvolution();
                    if (!GetModifyPerformed() && !!evolveTag->GetPerformer()) {
                        lines.emplace_back(i.second.GetCarInfo() + ": " + evolveTag->GetName() + " исполняется");
                        ++addTags["В работе " + newTagInfo.GetId()];
                    } else {
                        tag->SetPerformer(evolveTag->GetPerformer());
                        if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().DirectEvolveTag(GetRobotUserId(), evolveTag, tag, session)) {
                            return MakeUnexpected<TString>("cannot DirectEvolveTag: " + session.GetStringReport());
                        }
                        ++addTags["Эволюция " + newTagInfo.GetId()];
                        lines.emplace_back(i.second.GetCarInfo() + ": " + evolveTag->GetName() + " преобразован в " + i.second.GetId());
                    }
                }
                if (!session.Commit()) {
                    return MakeUnexpected<TString>("cannot Commit: " + session.GetStringReport());
                }
            }
        }
        if (lines.size()) {
            {
                TVector<TString> tagsNotify;
                Transform(addTags.begin(), addTags.end(), std::back_inserter(tagsNotify), [] (const auto& item) { return item.first +":" + ToString(item.second) + "\n"; });
                NotifyHelper.Notify(server, tagsNotify, /* header = */ "", /* force = */ true);
            }
            NotifyHelper.Notify(server, lines);
        }
    }

    return new IRTBackgroundProcessState();
}

bool IRTSensorToTagsWatcher::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    NotifyHelper.SetDefaultNotifyHeader(GetRTProcessName());
    return true;
}

NDrive::TScheme IRTSensorToTagsWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    SensorHelper.PatchScheme(scheme);
    scheme.Add<TFSString>("critical_values", "Критические значения сенсора");
    scheme.Add<TFSString>("critical_values_tags", "Теги, соответствующие критическим значениям");
    NotifyHelper.PatchScheme(server, scheme);
    scheme.Add<TFSBoolean>("dry_run_mode", "Только нотификации").SetDefault(false);
    scheme.Add<TFSString>("priorities", "Приоритеты соответствующих тегов");
    scheme.Add<TFSNumeric>("precision_up", "Точность для сравнения вверх").SetMin(0).SetPrecision(2).SetDefault(0);
    scheme.Add<TFSNumeric>("precision_down", "Точность для сравнения вниз").SetMin(0).SetPrecision(2).SetDefault(0);
    scheme.Add<TFSNumeric>("blocks_count_limit", "Максимальное количество заблокированных машин").SetMin(0).SetDefault(100);
    scheme.Add<TFSBoolean>("modify_performed", "модифицировать исполняемые").SetDefault(true);
    scheme.Add<TFSVariants>("abilities", "Возможные действия").InitVariants<EAbility>().SetMultiSelect(true);
    scheme.Add<TFSBoolean>("use_sensor_api", "Использовать SensorApi").SetDefault(false);
    return scheme;
}

bool IRTSensorToTagsWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo) || !SensorHelper.DeserializeFromJson(jsonInfo) || !NotifyHelper.DeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_BOOL_OPT(jsonInfo, "dry_run_mode", DryRunMode);
    JREAD_BOOL_OPT(jsonInfo, "modify_performed", ModifyPerformed);
    JREAD_CONTAINER_OPT(jsonInfo, "critical_values_tags", Tags);
    JREAD_CONTAINER_OPT(jsonInfo, "critical_values", CriticalValues);
    JREAD_CONTAINER_OPT(jsonInfo, "priorities", Priorities);
    if (Tags.empty() || CriticalValues.empty() || Priorities.empty()) {
        return false;
    }

    if (Priorities.size() != Tags.size()) {
        return false;
    }

    if (Tags.size() != CriticalValues.size() && Tags.size() != CriticalValues.size() + 1) {
        return false;
    }
    JREAD_INT_OPT(jsonInfo, "blocks_count_limit", BlocksCountLimit);
    Abilities.clear();
    JREAD_CONTAINER_OPT(jsonInfo, "abilities", Abilities);
    JREAD_DOUBLE_OPT(jsonInfo, "precision_up", PrecisionUp);
    JREAD_DOUBLE_OPT(jsonInfo, "precision_down", PrecisionDown);
    JREAD_BOOL_OPT(jsonInfo, "use_sensor_api", UseSensorApi);
    return true;
}

NJson::TJsonValue IRTSensorToTagsWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    SensorHelper.PatchJson(result);
    NotifyHelper.PatchJson(result);
    JWRITE(result, "dry_run_mode", DryRunMode);
    JWRITE(result, "modify_performed", ModifyPerformed);
    JWRITE(result, "precision_up", PrecisionUp);
    JWRITE(result, "precision_down", PrecisionDown);
    JWRITE(result, "blocks_count_limit", BlocksCountLimit);
    JWRITE(result, "use_sensor_api", UseSensorApi);
    TJsonProcessor::WriteContainerArray(result, "critical_values_tags", Tags);
    TJsonProcessor::WriteContainerArray(result, "critical_values", CriticalValues);
    TJsonProcessor::WriteContainerArray(result, "priorities", Priorities);
    TJsonProcessor::WriteContainerArrayStrings(result, "abilities", Abilities);
    return result;
}

TString TRTSensorToTagsWatcher::GetType() const {
    return GetTypeName();
}

TString TRTSensorToTagsWatcher::GetTypeName() {
    return "sensor_to_tags";
}
