#pragma once

#include "common.h"


class IRTSensorToTagsWatcher: public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;

public:
    enum class EAbility {
        Add,
        Evolve,
        Remove
    };

    enum class ESpecialAction {
        Remove,
        Ignore,
        Add
    };

    class TSensorWatcherContext {
    public:
        TSensorWatcherContext(const TTagsModificationContext& context)
            : RTContext(context)
        {
        }

        virtual TMaybe<ESpecialAction> CheckDevice(const TString& /* carId */, const double /* value */) const {
            return ESpecialAction::Add;
        }

        virtual const TSet<TString>& GetFilteredCarIds() const {
            return RTContext.GetFilteredCarIds();
        }

        virtual const TMap<TString, TDriveCarInfo>& GetFetchedCarsData() const {
            return RTContext.GetFetchedCarsData();
        }

        virtual const TDriveCarInfo* GetFetchedCarsData(const TString& carId) const {
            return RTContext.GetFetchedCarsData(carId);
        };

        virtual bool Init(const NDrive::IServer& /* server */) {
            return true;
        }

        const TTagsModificationContext& GetRTContext() {
            return RTContext;
        }

    protected:
        const TTagsModificationContext& RTContext;
    };

protected:
    TExpectedState SensorWatcherDoExecute(TSensorWatcherContext& context) const;

    virtual TMaybe<ESpecialAction> CheckDevice(const TString& carId, const double value, const TSensorWatcherContext& context) const {
        return context.CheckDevice(carId, value);
    }

    virtual bool DoStart(const TRTBackgroundProcessContainer& container) override;

public:
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    bool GetTagByValue(const double value, TString& tagName, i32& priority) const;

protected:
    double PrecisionUp = 0;
    double PrecisionDown = 0;

private:
    static const TSet<EAbility> FullAbilitiesSet;
    R_READONLY(TVector<double>, CriticalValues);
    R_READONLY(TVector<TString>, Tags);
    R_READONLY(TVector<i32>, Priorities);
    R_READONLY(bool, UseSensorApi, false);
    R_READONLY(TSet<EAbility>, Abilities, FullAbilitiesSet);
    R_READONLY(bool, DryRunMode, false);
    R_READONLY(bool, ModifyPerformed, true);
    R_READONLY(ui32, BlocksCountLimit, 0);
    R_FIELD(TSensorsHelper, SensorHelper);
    R_FIELD(TNotifyHelper, NotifyHelper);
};

template<class TContext>
class IRTSensorToTagsWatcherTemplate: public IRTSensorToTagsWatcher {
protected:
    virtual bool CustomizeWatcherContext(TContext& /* watcherContext */) const {
        return true;
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /* state */, const NDrive::IServer& server, TTagsModificationContext& context) const override {
        TContext watcherContext(context);
        if (CustomizeWatcherContext(watcherContext) && !watcherContext.Init(server)) {
            return MakeUnexpected<TString>("Fail to init sensor watcher context");
        }
        return SensorWatcherDoExecute(watcherContext);
    }
};

class TRTSensorToTagsWatcher: public IRTSensorToTagsWatcherTemplate<IRTSensorToTagsWatcher::TSensorWatcherContext> {
    using TBase = IRTSensorToTagsWatcherTemplate<IRTSensorToTagsWatcher::TSensorWatcherContext>;
public:
    virtual TString GetType() const override;

    static TString GetTypeName();

private:
    static TFactory::TRegistrator<TRTSensorToTagsWatcher> Registrator;
};
