#include "maintenance.h"

#include <drive/backend/data/device_tags.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/maintenance/manager.h>


TRTMaintenanceWatcher::TFactory::TRegistrator<TRTMaintenanceWatcher> TRTMaintenanceWatcher::Registrator(TRTMaintenanceWatcher::GetTypeName());

TString TRTMaintenanceWatcher::GetTypeName() {
    return "maintenance_watcher";
}

TString TRTMaintenanceWatcher::GetType() const {
    return GetTypeName();
}

bool TRTMaintenanceWatcher::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    NotifyHelper.SetDefaultNotifyHeader(GetRTProcessName());
    return true;
}

NDrive::TScheme TRTMaintenanceWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    SensorHelper.PatchScheme(scheme);
    NotifyHelper.PatchScheme(server, scheme);
    scheme.Add<TFSDuration>("sensor_fresheness", "Максимально допустимый возраст данных сенсора").SetDefault(TDuration::Minutes(1));
    scheme.Add<TFSString>("tag_name", "Тег для обозначения ТО").SetRequired(true);
    scheme.Add<TFSNumeric>("critical_value", "Критическое значение, после которого уводить с линии");
    scheme.Add<TFSNumeric>("critical_priority", "Приоритет тега, с которым уводить с линии");
    scheme.Add<TFSNumeric>("precision", "Точность для сравнения").SetMin(0).SetDefault(0);
    scheme.Add<TFSDuration>("critical_period", "Критическое превышение по времени, после которого уводить с линии");
    scheme.Add<TFSBoolean>("remove_tags", "Удалять теги").SetDefault(RemoveTags);
    scheme.Add<TFSBoolean>("dry_run_mode", "Только нотификации").SetDefault(false);
    scheme.Add<TFSNumeric>("blocks_count_limit", "Максимальное количество заблокированных машин").SetMin(0).SetDefault(100);
    scheme.Add<TFSBoolean>("modify_performed", "модифицировать исполняемые").SetDefault(true);
    return scheme;
}

bool TRTMaintenanceWatcher::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    return TBase::DoDeserializeFromJson(json)
        && SensorHelper.DeserializeFromJson(json)
        && NotifyHelper.DeserializeFromJson(json)
        && NJson::ParseField(json, "sensor_fresheness", SensorFreshness, /* required = */ false)
        && NJson::ParseField(json, "tag_name", Tag, /* required = */ true)
        && NJson::ParseField(json, "critical_value", CriticalValue, /* required = */ true)
        && NJson::ParseField(json, "critical_priority", Priority, /* required = */ false)
        && NJson::ParseField(json, "precision", Precision, /* required = */ false)
        && NJson::ParseField(json, "critical_period", CriticalPeriod, /* required = */ false)
        && NJson::ParseField(json, "remove_tags", RemoveTags, /* required = */ false)
        && NJson::ParseField(json, "dry_run_mode", DryRunMode, /* required = */ false)
        && NJson::ParseField(json, "blocks_count_limit", BlocksCountLimit, /* required = */ false)
        && NJson::ParseField(json, "modify_performed", ModifyPerformed, /* required = */ false);
}

NJson::TJsonValue TRTMaintenanceWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    SensorHelper.PatchJson(result);
    NotifyHelper.PatchJson(result);
    NJson::InsertField(result, "sensor_fresheness", SensorFreshness);
    NJson::InsertField(result, "tag_name", Tag);
    NJson::InsertField(result, "critical_value", CriticalValue);
    NJson::InsertField(result, "critical_priority", Priority);
    NJson::InsertField(result, "precision", Precision);
    NJson::InsertField(result, "critical_period", NJson::Hr(CriticalPeriod));
    NJson::InsertField(result, "remove_tags", RemoveTags);
    NJson::InsertField(result, "dry_run_mode", DryRunMode);
    NJson::InsertField(result, "blocks_count_limit", BlocksCountLimit);
    NJson::InsertField(result, "modify_performed", ModifyPerformed);
    return result;
}

namespace {
    class TMaintenanceDecision {
    private:
        R_FIELD(i32, Priority, 0);
        R_FIELD(TMaintenanceTag::EReason, Reason, TMaintenanceTag::EReason::Unknown);
        R_FIELD(ui64, CurrentMileage, 0);
        R_FIELD(ui64, CurrentMileageParam, 0);
        R_FIELD(ui64, CurrentCriticalMileage, 0);
        R_FIELD(TInstant, TimestampParam, TInstant::Zero());
        R_FIELD(TInstant, CriticalTimestampParam, TInstant::Zero());

    public:
        TMaintenanceDecision(const i64 priority, const TMaintenanceTag::EReason reason)
            : Priority(priority)
            , Reason(reason)
        {
        }

        bool PatchTag(TMaintenanceTag& tag) {
            bool update = false;
            if (Priority > 0 && tag.GetTagPriority(0) != Priority) {
                tag.SetTagPriority(Priority);
                update |= true;
            }
            if (!tag.HasTagPriority()) {
                tag.SetTagPriority(0);
                update |= true;
            }
            if (tag.GetReason() != Reason) {
                tag.SetReason(Reason);
                update |= true;
            }
            auto checkValues = [&update](auto& tagVal, const auto val) {
                if (val && (!tagVal || *tagVal != val)) {
                    tagVal = val;
                    update |= true;
                }
            };
            checkValues(tag.OptionalCurrentMileage(), CurrentMileage);
            checkValues(tag.OptionalRequiredMileage(), CurrentMileageParam);
            checkValues(tag.OptionalCriticalMileage(), CurrentCriticalMileage);
            checkValues(tag.OptionalRequiredTimestamp(), TimestampParam);
            checkValues(tag.OptionalCriticalTimestamp(), CriticalTimestampParam);
            return update;
        }
    };

    class TWatcherContext {
    private:
        using TSensorValues = TMap<TString, double>;
        using TMaintenanceInfos = TMap<TString, TMaintenanceInfo>;
        using TModelsMap = TMap<TString, TDriveModelData>;
        using TTransferDates = TMap<TString, TInstant>;

    private:
        R_FIELD(ui16, Priority, 0);
        R_FIELD(ui64, CriticalValue, 0);
        R_FIELD(ui32, Precision, 0);
        R_FIELD(TDuration, CriticalPeriod, TDuration::Zero());
        R_FIELD(TSensorValues, SensorValues);
        R_FIELD(TMaintenanceInfos, PlanMaintenanceInfo);
        R_FIELD(TMaintenanceInfos, LastMaintenanceInfo);
        R_FIELD(TModelsMap, CarModels);
        R_FIELD(TTransferDates, TransferDates);

    public:
        TWatcherContext(const ui16 priority, const ui64 criticalValue, const ui32 precision, const TDuration criticalPeriod)
            : Priority(priority)
            , CriticalValue(criticalValue)
            , Precision(precision)
            , CriticalPeriod(criticalPeriod)
        {
        }

        bool Initialize(const IRTCarsProcess::TTagsModificationContext& carsContext, NDrive::TEntitySession& session) {
            return GetMaintenanceInfos(carsContext, session)
                && GetModelConstants(carsContext.GetServer(), session)
                && GetTransferDates(carsContext, session);
        }

        bool InitializeSensorValues(const TSensorsHelper& helper, const IRTCarsProcess::TTagsModificationContext& carsContext, const TDuration freshness, const TInstant start) {
            auto sensorValues = helper.GetSensorValues(carsContext, /* useSensorApi = */ false, freshness, start);
            if (!sensorValues) {
                return false;
            }
            SetSensorValues(std::move(*sensorValues));
            return true;
        }

        TMaybe<TMaintenanceDecision> MakeMaintenanceDecision(const TDriveCarInfo& car) const {
            auto sensorValue = GetSensorValues().FindPtr(car.GetId());
            auto carModel = GetCarModels().FindPtr(car.GetModel());
            if (!sensorValue || !carModel) {
                NDrive::TEventLog::Log("CovidPassClientResponse", NJson::TMapBuilder
                    ("car_id", car.GetId())
                    ("error", "fail to fetch data")
                    ("has_sensor_value", !!sensorValue)
                    ("has_model_info", !!carModel)
                );
                return {};
            }
            auto lastMaintenanceInfo = GetLastMaintenanceInfo().FindPtr(car.GetVin());
            if (!lastMaintenanceInfo || (lastMaintenanceInfo->HasReadyDate() && !lastMaintenanceInfo->HasMileage())) {
                TMaybe<TMaintenanceDecision> result;
                if (carModel->HasMaintenanceMileage()) {
                    result = MakeDecision((ui64)*sensorValue, carModel->GetMaintenanceMileageRef(), TMaintenanceTag::EReason::Plan);
                }
                if ((!result || result->GetPriority() < 0) && carModel->HasFirstMaintenanceMileage()) {
                    result = MakeDecision((ui64)*sensorValue, carModel->GetFirstMaintenanceMileageRef(), TMaintenanceTag::EReason::First);
                }
                if ((!result || result->GetPriority() < 0) && carModel->HasIntermediateMaintenanceMileage()) {
                    result = MakeDecision((ui64)*sensorValue, carModel->GetIntermediateMaintenanceMileageRef(), TMaintenanceTag::EReason::Intermediate);
                }
                if ((!result || result->GetPriority() < 0) && carModel->HasMaintenancePeriod()) {
                    if (lastMaintenanceInfo) {
                        result = MakeDecision(lastMaintenanceInfo->GetReadyDateRef() + carModel->GetMaintenancePeriodRef());
                    } else if (auto transferDate = TransferDates.FindPtr(car.GetId()); transferDate && *transferDate) {
                        result = MakeDecision(*transferDate + carModel->GetMaintenancePeriodRef());
                    }
                }
                return result;
            }
            if (!lastMaintenanceInfo->HasMileage() || !lastMaintenanceInfo->HasReadyDate()) {
                return {};
            }
            TMaybe<TMaintenanceDecision> result;
            if (carModel->HasMaintenanceMileage()) {
                ui64 mileageParam = carModel->GetMaintenanceMileageRef();
                if (auto planMaintenanceInfo = GetPlanMaintenanceInfo().FindPtr(car.GetVin())) {
                    if (!planMaintenanceInfo->HasReadyDate() && planMaintenanceInfo->IsEqual(*lastMaintenanceInfo)) {
                        return {}; // in process;
                    }
                    mileageParam += planMaintenanceInfo->GetMileageDef(0);
                }
                result = MakeDecision((ui64)*sensorValue, mileageParam, TMaintenanceTag::EReason::Plan);
            }
            if ((!result || result->GetPriority() < 0) && carModel->HasIntermediateMaintenanceMileage()) {
                return MakeDecision((ui64)*sensorValue, carModel->GetIntermediateMaintenanceMileageRef() + lastMaintenanceInfo->GetMileageRef(), TMaintenanceTag::EReason::Intermediate);
            }
            if ((!result || result->GetPriority() < 0) && carModel->HasMaintenancePeriod()) {
                result = MakeDecision(lastMaintenanceInfo->GetReadyDateRef() + carModel->GetMaintenancePeriodRef());
            }
            return result;
        }


    private:
        TMaybe<TMaintenanceDecision> MakeDecision(const ui64 mileage, const ui64 param, const TMaintenanceTag::EReason reason) const {
            auto standart = CheckMileage(mileage, param);
            auto critical = CheckMileage(mileage, param + CriticalValue);
            if (!standart || !critical) {
                return {};
            }
            i64 priority = 0;
            if (!*standart) {
                priority = -1;
            }
            if (*critical) {
                priority = Priority;
            }
            return TMaintenanceDecision(priority, reason).SetCurrentMileage(mileage).SetCurrentMileageParam(param).SetCurrentCriticalMileage(param + CriticalValue);
        }

        TMaybe<TMaintenanceDecision> MakeDecision(const TInstant param) const {
            i64 priority = -1;
            if (param + CriticalPeriod < ModelingNow()) {
                priority = Priority;
            } else if (param < ModelingNow()) {
                priority = 0;
            }
            return TMaintenanceDecision(priority, TMaintenanceTag::EReason::Timeout)
                .SetTimestampParam(param)
                .SetCriticalTimestampParam(param + CriticalPeriod);
        }

        bool GetTransferDates(const IRTCarsProcess::TTagsModificationContext& carsContext, NDrive::TEntitySession& session) {
            auto attachmentsMap = carsContext.GetServer().GetDriveAPI()->GetCarAttachmentAssignments().GetAttachmentOfType(carsContext.GetFilteredCarIds(), EDocumentAttachmentType::CarRegistryDocument, session);
            if (!attachmentsMap) {
                return false;
            }
            for (auto&& [carId, attachments] : *attachmentsMap) {
                if (attachments.empty()) {
                    continue;
                }
                if (auto regDocument = dynamic_cast<const TCarRegistryDocument*>(attachments.front().Get())) {
                    TransferDates.emplace(carId, regDocument->GetTransferDate());
                }
            }
            return true;
        }

        bool GetMaintenanceInfos(const IRTCarsProcess::TTagsModificationContext& carsContext, NDrive::TEntitySession& session) {
            const auto& cars = carsContext.GetFetchedCarsData();
            TSet<TString> vins;
            Transform(cars.begin(), cars.end(), std::inserter(vins, vins.begin()), [](const auto& item) { return item.second.GetVin(); });
            TSet<TString> intermediate;
            {
                auto infos = carsContext.GetServer().GetDriveAPI()->GetMaintenanceDB().GetObjects(vins, session);
                if (!infos) {
                    return {};
                }
                Transform(infos->begin(), infos->end(), std::inserter(LastMaintenanceInfo, LastMaintenanceInfo.begin()), [](auto&& info) -> std::pair<TString, TMaintenanceInfo> { return { info.GetVIN(), info }; });
                for (auto&& info : *infos) {
                    TString vin = info.GetVIN();
                    if (info.IsIntermediate()) {
                        intermediate.insert(vin);
                    } else {
                        PlanMaintenanceInfo.emplace(vin, info);
                    }
                    LastMaintenanceInfo.emplace(std::move(vin), std::move(info));
                }
            }
            NSQL::TQueryOptions options(/* limit = */ 0, /* descending = */ true);
            options.SetOrderBy({ "start_date" });
            options.SetGenericCondition("vin", intermediate);
            options.SetGenericCondition("is_intermediate", NSQL::Not(true));
            auto infos = carsContext.GetServer().GetDriveAPI()->GetMaintenanceDB().GetHistoryManager().GetEvents({}, {}, session, options);
            if (!infos) {
                return {};
            }
            for (auto&& info : *infos) {
                if (intermediate.empty()) {
                    break;
                }
                TString vin = info.GetVIN();
                if (intermediate.erase(vin)) {
                    PlanMaintenanceInfo.emplace(std::move(vin), std::move(info));
                }
            }
            return true;
        }

        bool GetModelConstants(const NDrive::IServer& server, NDrive::TEntitySession& session) {
            auto mgr = server.GetDriveAPI()->GetModelsData();
            if (!mgr) {
                return false;
            }
            auto models = mgr->FetchInfo(session);
            if (!models) {
                return false;
            }
            CarModels = std::move(models.GetResult());
            return true;
        }

        TMaybe<bool> CheckMileage(const ui64 mileage, const ui64 param) const {
            if (mileage + Precision < param) {
                return false;
            }
            if (mileage > param + Precision) {
                return true;
            }
            return {};
        }
    };

    class TTagsContext {
    public:
        bool GetCarTags(const IRTCarsProcess::TTagsModificationContext& carsContext, const TString& tagName, NDrive::TEntitySession& session) {
            auto dbTags = carsContext.GetServer().GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTagsRobust(std::cref(carsContext.GetFilteredCarIds()), { tagName }, session);
            if (!dbTags) {
                return false;
            }
            for (auto&& tag : *dbTags) {
                const TString id = tag.GetObjectId();
                if (!TagsByDevice.FindPtr(id)) {
                    if (tag->GetTagPriority(0) > 0) {
                        Blocked.emplace(id);
                    }
                    TagsByDevice.emplace(id, std::move(tag));
                } else {
                    WrongTagsCount.emplace(id);
                }
            }
            return true;
        }

    private:
        using TTagsMap = TMap<TString, TDBTag>;

        R_FIELD(TTagsMap, TagsByDevice);
        R_FIELD(TSet<TString>, Blocked);
        R_FIELD(TSet<TString>, WrongTagsCount);
    };

}

TExpectedState TRTMaintenanceWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*stateExt*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    TWatcherContext watcherContext(Priority, CriticalValue, Precision, CriticalPeriod);
    TTagsContext tagsContext;
    {
        if (!watcherContext.InitializeSensorValues(SensorHelper, context, SensorFreshness, TBase::StartInstant)) {
            return MakeUnexpected<TString>("Fail to get sensors");
        }
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!watcherContext.Initialize(context, session)) {
            return MakeUnexpected<TString>("Fail to get car info: " + session.GetStringReport());
        }
        if (!tagsContext.GetCarTags(context, Tag, session)) {
            return MakeUnexpected<TString>("cannot RestoreTags: " + session.GetStringReport());
        }
    }
    context.SetAddTagPolicy(EUniquePolicy::Rewrite);

    TVector<TString> notifyLines;
    const ui32 blockedLimit = GetBlocksCountLimit() ? GetBlocksCountLimit() : Max<ui32>();
    for (auto&& [id, car] : context.GetFetchedCarsData()) {
        if (tagsContext.GetWrongTagsCount().contains(id)) {
            notifyLines.emplace_back(id + ": не обработан из-за неверных данных");
            continue;
        }
        auto toDecision = watcherContext.MakeMaintenanceDecision(car);
        if (!toDecision) {
            continue;
        }
        if (toDecision->GetPriority() < 0) {
            if (!RemoveTags) {
                continue;
            }
            if (auto it = tagsContext.GetTagsByDevice().FindPtr(id)) {
                auto& tag = *it;
                if (!tag->GetPerformer() || GetModifyPerformed()) {
                    if (tag->GetTagPriority(0) > 0) {
                        tagsContext.MutableBlocked().erase(id);
                    }
                    notifyLines.emplace_back(id + ": удален " + tag->GetName());
                    context.RemoveTag(id, tag);
                }
            }
        } else {
            ITag::TPtr tag;
            {
                auto desc = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(Tag);
                if (desc) {
                    tag = ITag::TFactory::Construct(desc->GetType());
                    tag->SetName(Tag);
                }
            }
            if (!tag) {
                notifyLines.emplace_back("Указан невалидный тег: " + Tag);
                continue;
            }
            auto mTag = std::dynamic_pointer_cast<TMaintenanceTag>(tag);
            if (mTag) {
                toDecision->PatchTag(*mTag);
            } else {
                tag->SetTagPriority(toDecision->GetPriority());
            }
            if (auto it = tagsContext.MutableTagsByDevice().FindPtr(id)) {
                auto& oldTag = *it;
                if (oldTag->GetPerformer() && !GetModifyPerformed()) {
                    continue;
                }
                if (oldTag->GetTagPriority(0) > tag->GetTagPriority(0) && (!mTag || mTag->GetReason() != TMaintenanceTag::EReason::Plan)) {
                    continue;
                }
                auto mOldTag = oldTag.MutableTagAs<TMaintenanceTag>();
                if (mOldTag && mTag) {
                    if (mOldTag->GetReason() == mTag->GetReason() && oldTag->GetTagPriority(0) == tag->GetTagPriority(0)) {
                        continue;
                    }
                    if (mOldTag->GetReason() != mTag->GetReason() && mOldTag->GetReason() == TMaintenanceTag::EReason::Plan) {
                        continue;
                    }
                }
            }
            if (tag->GetTagPriority(0) > 0 && !tagsContext.GetBlocked().contains(id) && tagsContext.GetBlocked().size() > blockedLimit) {
                notifyLines.emplace_back(id +  ": ожидает изменения " + tag->GetName());
                continue;
            }
            notifyLines.emplace_back(id + ": добавлен/изменён " + Tag);
            context.AddTag(id, tag);
        }
    }

    if (!context.ApplyModification(GetDryRunMode())) {
        return MakeUnexpected<TString>("cannot apply modifycations");
    }
    NotifyHelper.Notify(server, notifyLines);
    return new IRTBackgroundProcessState();
}
