#include "common.h"

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/cast.h>
#include <rtline/library/json/field.h>
#include <rtline/library/json/parse.h>

namespace {
    NServiceRouting::TTimeWindow GetNearestTimeWindow(const TTimeRestriction& restriction, TInstant reference) {
        auto leftBound = restriction.GetTimeStart(reference);
        auto rightBound = restriction.GetTimeFinish(reference);
        if (rightBound < leftBound) {
            rightBound += TDuration::Days(1);
        }
        Y_ENSURE(leftBound <= rightBound);
        while (rightBound < reference) {
            leftBound += TDuration::Days(1);
            rightBound += TDuration::Days(1);
        }
        while (rightBound - TDuration::Days(1) >= reference) {
            leftBound -= TDuration::Days(1);
            rightBound -= TDuration::Days(1);
        }
        return std::make_pair(leftBound, rightBound);
    }

    void Adjust(NServiceRouting::TTimeWindow& window, TInstant reference) {
        window.first = std::max(window.first, reference);
    }
}

namespace NServiceRouting {
    TString Format(TTimeWindow window) {
        return window.first.ToString() + "/" + window.second.ToString();
    }

    bool TShiftDescription::IsActual(TInstant timestamp) const {
        return CalculateBoundaries.IsActualNow(timestamp);
    }

    TTimeWindow TShiftDescription::GetShiftBoundaries(TInstant reference) const {
        return GetNearestTimeWindow(ShiftBoundaries, reference);
    }

    TTimeWindow TShiftDescription::GetAdjustedShiftBoundaries(TInstant reference) const {
        auto result = GetShiftBoundaries(reference);
        Adjust(result, reference);
        return result;
    }

    TTimeWindow TShiftDescription::GetRulesBasedTimeWindow(TInstant reference, const TSet<TString>& areasIdsInPoint) const {
        for (const auto& rule : TimeWindowRules) {
            if (rule.IsSatisfied(areasIdsInPoint)) {
                return GetNearestTimeWindow(rule.GetTimeWindow(), reference);
            }
        }
        return GetShiftBoundaries(reference);
    }

    TTimeWindow TShiftDescription::GetAdjustedRulesBasedTimeWindow(TInstant reference, const TSet<TString>& areasIdsInPoint) const {
        auto result = GetRulesBasedTimeWindow(reference, areasIdsInPoint);
        Adjust(result, reference);
        return result;
    }

    bool TTimeWindowRule::IsSatisfied(const TSet<TString>& areasIds) const {
        switch (Arrangement) {
            case EArrangement::Inside: {
                return areasIds.contains(AreaId);
            }
            case EArrangement::Outside: {
                return !areasIds.contains(AreaId);
            }
        }
    }

    NDrive::TScheme TTimeWindowRule::GetScheme(const TSet<TString>& areasIds) {
        NDrive::TScheme scheme;
        scheme.Add<TFSVariants>("area_id", "id полигона").SetVariants(areasIds);
        scheme.Add<TFSVariants>("arrangement", "Положение относительно полигона").SetVariants(GetEnumAllValues<TTimeWindowRule::EArrangement>());
        scheme.Add<TFSStructure>("time_window", "Окно выполнения задачи при выполнении условия").SetStructure(TTimeRestriction::GetScheme());
        return scheme;
    }

    TInstant TGeneralOptions::GetShiftStart() const {
        auto [shiftStart, _] = ShiftDescription.GetAdjustedShiftBoundaries(Reference);
        return shiftStart;
    }

    TString TGeneralOptions::GetShift() const {
        auto timeWindow = ShiftDescription.GetAdjustedShiftBoundaries(Reference);
        return Format(timeWindow);
    }
}

template<>
NJson::TJsonValue NJson::ToJson(const NServiceRouting::TRoutingTask& routingTask) {
    NJson::TJsonValue result;
    result["id"] = routingTask.GetRoutingTaskId();
    result["day_of_shift_start"] = routingTask.GetAnchor().ToString();
    result["service_tasks_durations"] = NJson::ToJson(NJson::Dictionary(routingTask.GetServiceTasksDurations()));
    result["timestamp"] = NJson::ToJson(routingTask.GetTimestamp());
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NServiceRouting::TRoutingTask& routingTask) {
    if (value.Has("timestamp")) {
        if (!NJson::TryFromJson(value["timestamp"], routingTask.MutableTimestamp())) {
            return false;
        }
    } else {
        routingTask.MutableTimestamp() = Now(); // if there's no timestamp field, then the state is of old version
                                                // and we manually make the task fresh
    }
    return NJson::ParseField(value["id"], routingTask.MutableRoutingTaskId(), true)
        && value["day_of_shift_start"].IsString() && TInstant::TryParseIso8601(value["day_of_shift_start"].GetString(), routingTask.MutableAnchor())
        && NJson::ParseField(value["service_tasks_durations"], NJson::Dictionary(routingTask.MutableServiceTasksDurations()));
}

DECLARE_FIELDS_JSON_SERIALIZER(NServiceRouting::TTimeWindowRule);

NDrive::TScheme NServiceRouting::TShiftDescription::GetScheme(const TSet<TString>& areaIds) {
    NDrive::TScheme scheme;
    scheme.Add<TFSStructure>("shift_boundaries", "Время смены").SetStructure(TTimeRestriction::GetScheme());
    scheme.Add<TFSStructure>("calculate_boundaries", "Время подсчёта маршрутов").SetStructure(TTimeRestriction::GetScheme());
    scheme.Add<TFSArray>("time_window_rules").SetElement(TTimeWindowRule::GetScheme(areaIds));
    return scheme;
}

template <>
NJson::TJsonValue NJson::ToJson(const NServiceRouting::TShiftDescription& shiftDescription) {
    NJson::TJsonValue result;
    result["shift_boundaries"] = NJson::ToJson(shiftDescription.GetShiftBoundaries());
    result["calculate_boundaries"] = NJson::ToJson(shiftDescription.GetCalculateBoundaries());
    result["time_window_rules"] = NJson::ToJson(shiftDescription.GetTimeWindowRules());
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NServiceRouting::TShiftDescription& shiftDescription) {
    return NJson::ParseField(value["shift_boundaries"], shiftDescription.MutableShiftBoundaries())
        && shiftDescription.MutableShiftBoundaries().Compile()
        && NJson::ParseField(value["calculate_boundaries"], shiftDescription.MutableCalculateBoundaries())
        && shiftDescription.MutableCalculateBoundaries().Compile()
        && NJson::ParseField(value["time_window_rules"], shiftDescription.MutableTimeWindowRules());
}

template<>
NJson::TJsonValue NJson::ToJson(const NServiceRouting::TServiceTaskStatus& taskStatus) {
    NJson::TJsonValue result;
    NJson::FieldsToJson(result, taskStatus.GetFields());
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NServiceRouting::TServiceTaskStatus& taskStatus) {
    return NJson::TryFieldsFromJson(value, taskStatus.GetFields());
}

NJson::TJsonValue NServiceRouting::TServiceTaskPenalties::BuildReport() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    if (HasPenaltyOutOfTimeFixed()) {
        result["out_of_time"]["fixed"] = GetPenaltyOutOfTimeFixedRef();
    }
    if (HasPenaltyOutOfTimeMinute()) {
        result["out_of_time"]["minute"] = GetPenaltyOutOfTimeMinuteRef();
    }
    if (HasPenaltyEarlyFixed()) {
        result["early"]["fixed"] = GetPenaltyEarlyFixedRef();
    }
    if (HasPenaltyEarlyMinute()) {
        result["early"]["minute"] = GetPenaltyEarlyMinuteRef();
    }
    if (HasPenaltyLateFixed()) {
        result["late"]["fixed"] = GetPenaltyLateFixedRef();
    }
    if (HasPenaltyLateMinute()) {
        result["late"]["minute"] = GetPenaltyLateMinuteRef();
    }
    if (HasPenaltyDeliveryDeadlineFixed()) {
        result["delivery_deadline"]["fixed"] = GetPenaltyDeliveryDeadlineFixedRef();
    }
    if (HasPenaltyDeliveryDeadlineMinute()) {
        result["delivery_deadline"]["minute"] = GetPenaltyDeliveryDeadlineMinuteRef();
    }
    if (HasPenaltyDrop()) {
        result["drop"] = GetPenaltyDropRef();
    }
    return result;
}

template<>
NJson::TJsonValue NJson::ToJson(const NServiceRouting::TServiceTaskPenalties& penalties) {
    NJson::TJsonValue result;
    NJson::FieldsToJson(result, penalties.GetFields());
    return result;
}

template<>
bool NJson::TryFromJson(const NJson::TJsonValue& value, NServiceRouting::TServiceTaskPenalties& penalties) {
    return NJson::TryFieldsFromJson(value, penalties.GetFields());
}
