#pragma once

#include <rtline/library/geometry/coord.h>
#include <rtline/library/time_restriction/time_restriction.h>
#include <rtline/util/types/accessor.h>
#include <rtline/util/types/field.h>

namespace NServiceRouting {
    using TUserId = TString;
    using TRoutingTaskId = TString;
    using TServiceTaskId = TString;
    using TTimeWindow = std::pair<TInstant, TInstant>;

    TString Format(TTimeWindow window);

    enum class ERoutingQuality {
        Low = 0 /* "low" */,
        Normal = 1 /* "normal" */,
        High = 2 /* "high" */,
    };

    class TUserInfo {
    public:
        TString GetGarageId() const {
            return "garage_" + Id;
        }

    public:
        R_FIELD(TUserId, Id);
        R_FIELD(TGeoCoord, Position);
        R_FIELD(TVector<TString>, AvailableZones);
        R_FIELD(TVector<TServiceTaskId>, ObligatoryTasksIds);
        R_FIELD(TSet<TServiceTaskId>, DroppedTasksIds);
        R_FIELD(TSet<TServiceTaskId>, ExplicitlyAvailableTasksIds);
        R_FIELD(TVector<TString>, AdditionalTagNames);
    };

    class TServiceTaskPenalties {
    public:
        NJson::TJsonValue BuildReport() const;

    public:
        R_OPTIONAL(ui32, PenaltyOutOfTimeFixed);
        R_OPTIONAL(ui32, PenaltyOutOfTimeMinute);
        R_OPTIONAL(ui32, PenaltyEarlyFixed);
        R_OPTIONAL(ui32, PenaltyEarlyMinute);
        R_OPTIONAL(ui32, PenaltyLateFixed);
        R_OPTIONAL(ui32, PenaltyLateMinute);
        R_OPTIONAL(ui32, PenaltyDeliveryDeadlineFixed);
        R_OPTIONAL(ui32, PenaltyDeliveryDeadlineMinute);
        R_OPTIONAL(ui32, PenaltyDrop);

    public:
        DECLARE_FIELDS(
            Field(PenaltyOutOfTimeFixed, "penalty.out_of_time.fixed"),
            Field(PenaltyOutOfTimeMinute, "penalty.out_of_time.minute"),
            Field(PenaltyEarlyFixed, "penalty.early.fixed"),
            Field(PenaltyEarlyMinute, "penalty.early.minute"),
            Field(PenaltyLateFixed, "penalty.late.fixed"),
            Field(PenaltyLateMinute, "penalty.late.minute"),
            Field(PenaltyDeliveryDeadlineFixed, "penalty.delivery_deadline.fixed"),
            Field(PenaltyDeliveryDeadlineMinute, "penalty.delivery_deadline.minute"),
            Field(PenaltyDrop, "penalty.drop")
        );
    };

    class TServiceTaskInfo {
    public:
        R_FIELD(TServiceTaskId, Id);
        R_FIELD(TGeoCoord, Position);
        R_FIELD(TDuration, ServiceDuration);
        R_FIELD(TString, Zone);
        R_FIELD(bool, IsDroppedBySomeone, false);
        R_FIELD(TTimeWindow, TimeWindow);
        R_FIELD(TString, TagName);
        R_FIELD(TString, CarNumber);
        R_FIELD(TServiceTaskPenalties, Penalties);
    };

    class TRoutingOptions {
    public:
        using TTagNameToPenalties = TMap<TString, TServiceTaskPenalties>;
        using TTagNameToSLA = TMap<TString, TMaybe<TDuration>>;

    public:
        R_FIELD(ERoutingQuality, RoutingQuality, ERoutingQuality::Normal);
        R_FIELD(bool, UseAbsoluteTimeInResponse, true);
        R_FIELD(bool, AvoidTolls, true);
        R_FIELD(i32, TimeZone, 3);
        R_FIELD(bool, RestartOnDrop, false);
        R_OPTIONAL(double, CostPerKilometer);
        R_OPTIONAL(double, CostPerHour);
        R_OPTIONAL(double, GlobalProximityFactor);
        R_FIELD(bool, HardWindow, false);
        R_OPTIONAL(ui32, ShiftPenaltyOutOfTimeFixed);
        R_OPTIONAL(ui32, ShiftPenaltyOutOfTimeMinute);
        R_FIELD(bool, UseCarLocation, false);
        R_FIELD(bool, OneTaskPerCar, false);
        R_FIELD(bool, TasksHardWindow, false);
        R_FIELD(TSet<TString>, ServiceTasksTagNames);
        R_FIELD(TServiceTaskPenalties, DefaultPenalties);
        R_FIELD(TTagNameToPenalties, TagNameToPenalties);
        R_FIELD(TTagNameToSLA, TagNameToSLA);
    };

    class TEnqueueError : public yexception {
    public:
        enum class EStatus {
            InvalidPost = 400 /* "invalid_post" */,
            NoAPIKey = 401 /* "no_api_key" */,
            TooManyRequests = 429 /* "too_many_requests" */,
            ServerError = 500 /* "server_error" */,
            InsufficientResources = 503 /* "insufficient_resources" */,
        };

    public:
        const char* what() const noexcept override {
            return Message.c_str();
        }

    public:
        R_FIELD(EStatus, Status, EStatus::InvalidPost);
        R_FIELD(TString, Message);
    };

    class TRetrieveError : public yexception {
    public:
        enum class EStatus {
            IsRunning = 201 /* "is_running" */,
            InQueue = 202 /* "in_queue" */,
            NotFound = 410 /* "not_found" */,
            ServerError = 500 /* "server_error" */,
        };

    public:
        const char* what() const noexcept override {
            return Message.c_str();
        }

    public:
        R_FIELD(EStatus, Status, EStatus::IsRunning);
        R_FIELD(TString, Message);
    };

    using TServiceTasksDurations = TMap<TServiceTaskId, ui64>; // stored in seconds

    class TRoutingTask {
    public:
        R_FIELD(TRoutingTaskId, RoutingTaskId);
        R_FIELD(TInstant, Anchor);
        R_FIELD(TServiceTasksDurations, ServiceTasksDurations);
        R_FIELD(TInstant, Timestamp);
    };

    class TTimeWindowRule {
    public:
        enum class EArrangement {
            Inside = 0 /* "inside" */,
            Outside = 1 /* "outside" */,
        };

    public:
        static NDrive::TScheme GetScheme(const TSet<TString>& areasIds);

        bool IsSatisfied(const TSet<TString>& areasIds) const;

    private:
        R_FIELD(TString, AreaId);
        R_FIELD(EArrangement, Arrangement);
        R_FIELD(TTimeRestriction, TimeWindow);

    public:
        DECLARE_FIELDS(
            Field(AreaId, "area_id"),
            Field(NJson::Stringify(Arrangement), "arrangement"),
            Field(TimeWindow, "time_window")
        );
    };

    class TShiftDescription {
    public:
        static NDrive::TScheme GetScheme(const TSet<TString>& areaIds);
        NJson::TJsonValue SerializeToJson() const;
        bool DeserializeFromJson(const NJson::TJsonValue& value);
        bool IsActual(TInstant timestamp) const;
        TTimeWindow GetShiftBoundaries(TInstant reference) const;
        TTimeWindow GetAdjustedShiftBoundaries(TInstant reference) const;
        TTimeWindow GetRulesBasedTimeWindow(TInstant reference, const TSet<TString>& areasIdsInPoint) const;
        TTimeWindow GetAdjustedRulesBasedTimeWindow(TInstant reference, const TSet<TString>& areasIdsInPoint) const;

    public:
        R_FIELD(TTimeRestriction, ShiftBoundaries);
        R_FIELD(TTimeRestriction, CalculateBoundaries);
        R_FIELD(TVector<TTimeWindowRule>, TimeWindowRules);
    };

    class TGeneralOptions {
    public:
        TInstant GetShiftStart() const;
        TString GetShift() const;

    public:
        R_FIELD(TShiftDescription, ShiftDescription);
        R_FIELD(TInstant, Reference);
    };

    class TServiceTaskStatus {
    public:
        R_FIELD(TString, TaskId);
        R_FIELD(TString, CarId);
        R_FIELD(TString, TagName);
        R_OPTIONAL(TString, DropReason);

    public:
        DECLARE_FIELDS(
            Field(TaskId, "task_id"),
            Field(CarId, "car_id"),
            Field(TagName, "tag_name"),
            Field(DropReason, "drop_reason")
        )
    };
};
