#include "process.h"
#include "state.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/library/cpp/user_events_api/client.h>
#include <drive/library/cpp/user_events_api/realtime_user_data.h>

#include <library/cpp/http/simple/http_client.h>

#include <util/system/env.h>

namespace {
    bool IsBusyForServicing(const TTaggedObject& object, const TSet<NServiceRouting::TUserId>& serviceWorkersIds) {
        auto chargableDBTags = object.GetTagsByClass<TChargableTag>();
        for (const auto& chargableDBTag : chargableDBTags) {
            if (chargableDBTag->GetName() == TChargableTag::Servicing) {
                continue;
            }
            if (chargableDBTag->HasPerformer() && !serviceWorkersIds.contains(chargableDBTag->GetPerformer())) {
                return true;
            }
        }
        return false;
    }

    TMap<NServiceRouting::TUserId, TSet<NServiceRouting::TServiceTaskId>> GetWorkerToDroppedTasks(const NDrive::IServer& server, const TDriveAPI& api, const TSet<NServiceRouting::TUserId>& workersIds, NDrive::TEntitySession& session) {
        TMap<NServiceRouting::TUserId, TSet<NServiceRouting::TServiceTaskId>> res;
        auto now = Now();
        auto dropTasksHistorySize = server.GetSettings().GetValueDef<TDuration>("routing.drop_tasks_history_size", TDuration::Days(1));
        auto optionalEvents = api.GetTagsManager().GetDeviceTags().GetEvents({ now - dropTasksHistorySize, now }, session, TTagEventsManager::TQueryOptions()
            .SetUserIds(MakeSet(workersIds)).SetActions({EObjectHistoryAction::DropTagPerformer})
        );
        Y_ENSURE(optionalEvents, "can't fetch history events");
        for (const auto& ev : *optionalEvents) {
            res[ev.GetHistoryUserId()].insert(ev.GetTagId());
        }
        return res;
    }

    void ExcludeDroppedTasks(TVector<NServiceRouting::TServiceTaskInfo>& serviceTasksInfos, TVector<NServiceRouting::TUserInfo>& usersInfos) {
        TSet<NServiceRouting::TServiceTaskId> droppedTasksIds;
        for (const auto& userInfo : usersInfos) {
            droppedTasksIds.insert(userInfo.GetDroppedTasksIds().begin(), userInfo.GetDroppedTasksIds().end());
        }
        for (auto& serviceTaskInfo : serviceTasksInfos) {
            serviceTaskInfo.SetIsDroppedBySomeone(droppedTasksIds.contains(serviceTaskInfo.GetId()));
        }
        for (auto& userInfo : usersInfos) {
            for (const auto& taskId : droppedTasksIds) {
                if (!userInfo.GetDroppedTasksIds().contains(taskId)) {
                    userInfo.MutableExplicitlyAvailableTasksIds().insert(taskId);
                }
            }
        }
    }

    TVector<TString> GetEnclosingAreasIds(const TGeoCoord coord, const TAreasDB& areasDB) {
        TVector<TString> enclosingAreasIds;
        auto actor = [&enclosingAreasIds](const TFullAreaInfo& areaInfo) -> bool {
            enclosingAreasIds.push_back(areaInfo.GetArea().GetIdentifier());
            return true;
        };
        areasDB.ProcessAreasInPoint(coord, actor);
        return enclosingAreasIds;
    }

    TMaybe<TString> FindFirstEnclosingArea(const TGeoCoord coord, const TSet<TString>& areasIds, const TAreasDB& areasDB) {
        auto enclosingAreasIds = GetEnclosingAreasIds(coord, areasDB);
        for (const auto& id : enclosingAreasIds) {
            if (areasIds.contains(id)) {
                return id;
            }
        }
        return Nothing();
    }

    NServiceRouting::TRetrieveError InvalidFieldError(const TString& field) {
        NServiceRouting::TRetrieveError error;
        error.SetStatus(NServiceRouting::TRetrieveError::EStatus::ServerError);
        error.SetMessage("absent or incorrect type of field: " + field);
        return error;
    }

    void EnrichRoutesWithCarsData(TMap<NServiceRouting::TUserId, TServiceRoute>& serviceRoutes, const TDriveAPI& api) {
        auto session = api.BuildTx<NSQL::ReadOnly>();
        TVector<NServiceRouting::TServiceTaskId> tasksIds;
        for (const auto& [userId, serviceRoute] : serviceRoutes) {
            for (const auto& task : serviceRoute.GetTasks()) {
                if (task.IsGarage()) {
                    continue;
                }
                tasksIds.push_back(task.GetTagId());
            }
        }
        TVector<TDBTag> dbTags;
        Y_ENSURE(api.GetTagsManager().GetDeviceTags().RestoreTags(MakeSet(tasksIds), dbTags, session), TStringBuilder() << "can't restore tags: " << session.GetStringReport());
        TMap<NServiceRouting::TServiceTaskId, TString> serviceTaskIdToCarId;
        TSet<TString> carsIds;
        for (const auto& dbTag : dbTags) {
            serviceTaskIdToCarId[dbTag.GetTagId()] = dbTag.GetObjectId();
            carsIds.insert(dbTag.GetObjectId());
        }
        auto cars = Yensured(api.GetCarsData())->FetchInfo(carsIds, session);

        for (auto& [userId, serviceRoute] : serviceRoutes) {
            for (auto& task : serviceRoute.MutableTasks()) {
                if (task.IsGarage()) {
                    continue;
                }
                const auto& carId = serviceTaskIdToCarId[task.GetTagId()];
                task.SetCarId(carId);
                if (auto it = cars.find(carId); it != cars.end()) {
                    task.SetCarNumber(it->second.GetNumber());
                }
            }
        }
    }

    TMap<NServiceRouting::TServiceTaskId, TInstant> GetTagsCreationTimes(const TVector<TDBTag>& dbTags, const TDeviceTagsManager& manager, NDrive::TEntitySession& session) {
        using namespace NServiceRouting;
        TSet<TServiceTaskId> tasksIds;
        for (const auto& dbTag : dbTags) {
            tasksIds.insert(dbTag.GetTagId());
        }
        auto maybeEvents = manager.GetEvents(TRange<TInstant>{}, session, TTagEventsManager::TQueryOptions()
            .SetTagIds(std::move(tasksIds)).SetActions({EObjectHistoryAction::Add})
        );
        TMap<TServiceTaskId, TInstant> result;
        if (maybeEvents) {
            for (const auto& ev : *maybeEvents) {
                if (ev) {
                    result[ev.GetTagId()] = ev.GetHistoryTimestamp();
                }
            }
        } else {
            NDrive::TEventLog::Log("RoutingMRVPError", NJson::TMapBuilder
                ("message", session.GetStringReport())
            );
        }
        return result;
    }

    TVector<TString> TakeBestTagIdPerCar(const TVector<TDBTag>& dbTags, const TSet<NServiceRouting::TServiceTaskId>& performedTasksIds, const TMap<NServiceRouting::TServiceTaskId, TInstant>& taskIdToCreationTime) {
        TMap<TString, TDBTag> carToTagRepr;
        for (const auto& dbTag : dbTags) {
            const auto& carId = dbTag.GetObjectId();
            auto it = carToTagRepr.find(carId);
            if (it == carToTagRepr.end() || performedTasksIds.contains(dbTag.GetTagId())) {
                carToTagRepr[carId] = dbTag;
            } else if (dbTag->OptionalTagPriority() > it->second->OptionalTagPriority()) {
                carToTagRepr[carId] = dbTag;
            } else if (dbTag->OptionalTagPriority() == it->second->OptionalTagPriority()) {
                auto curTimeIt = taskIdToCreationTime.find(dbTag.GetTagId());
                auto bestTimeIt = taskIdToCreationTime.find(it->second.GetTagId());
                if (curTimeIt != taskIdToCreationTime.end() && bestTimeIt != taskIdToCreationTime.end()
                && curTimeIt->second > bestTimeIt->second) {
                    carToTagRepr[carId] = dbTag;
                }
            }
        }
        TVector<TString> result;
        for (const auto& [_, dbTag] : carToTagRepr) {
            result.push_back(dbTag.GetTagId());
        }
        return result;
    }

    TString GetServiceZone(const TGeoCoord coord, const TAreasDB& areasDB) {
        TString serviceZone;
        auto serviceZoneFinder = [&serviceZone](const TFullAreaInfo& areaInfo) {
            if (areaInfo.GetArea().GetType() == TServiceZonesTag::ServiceZoneAreaType) {
                serviceZone = areaInfo.GetArea().GetIdentifier();
                return false;
            }
            return true;
        };
        areasDB.ProcessAreasInPoint(coord, serviceZoneFinder);
        return serviceZone;
    }

    TSet<TString> GetCarsIdsFromRoutes(const TMap<NServiceRouting::TUserId, TServiceRoute>& serviceRoutes) {
        TSet<TString> result;
        for (const auto& [_, serviceRoute] : serviceRoutes) {
            for (const auto& task : serviceRoute.GetTasks()) {
                if (task.GetCarId()) {
                    result.insert(task.GetCarId());
                }
            }
        }
        return result;
    }
}

namespace NServiceRouting {
    TServiceRouteCalculatorProcess::TFactory::TRegistrator<TServiceRouteCalculatorProcess> TServiceRouteCalculatorProcess::Registrator(TServiceRouteCalculatorProcess::GetTypeName());

    NJson::TJsonValue SerializeLocationsToJson(const TVector<TUserInfo>& usersInfos, const TVector<TServiceTaskInfo>& serviceTasksInfos, const TGeneralOptions& options, const TRoutingOptions& routingOptions) {
        NJson::TJsonValue result;
        for (const auto& serviceTaskInfo : serviceTasksInfos) {
            NJson::TJsonValue locationJson;
            locationJson["id"] = serviceTaskInfo.GetId();
            locationJson["point"] = serviceTaskInfo.GetPosition().SerializeLatLonToJson();
            locationJson["time_window"] = Format(serviceTaskInfo.GetTimeWindow());
            locationJson["service_duration_s"] = serviceTaskInfo.GetServiceDuration().Seconds();
            locationJson["hard_window"] = routingOptions.GetTasksHardWindow();
            locationJson["comments"] = NJson::TJsonValue{
                NJson::TMapBuilder
                    ("car_number", serviceTaskInfo.GetCarNumber())
                    ("tag_name", serviceTaskInfo.GetTagName())
            }.GetStringRobust();
            locationJson["required_tags"].AppendValue(serviceTaskInfo.GetZone());
            if (serviceTaskInfo.GetIsDroppedBySomeone()) {
                locationJson["required_tags"].AppendValue(serviceTaskInfo.GetId());
            }
            locationJson["required_tags"].AppendValue(serviceTaskInfo.GetTagName());
            locationJson["penalty"] = serviceTaskInfo.GetPenalties().BuildReport();
            result.AppendValue(std::move(locationJson));
        }
        for (const auto& userInfo : usersInfos) {
            NJson::TJsonValue locationJson;
            locationJson["id"] = userInfo.GetGarageId();
            locationJson["type"] = "garage";
            locationJson["point"] = userInfo.GetPosition().SerializeLatLonToJson();
            locationJson["time_window"] = options.GetShift();
            result.AppendValue(std::move(locationJson));
        }
        return result;
    }

    NJson::TJsonValue SerializeVehiclesToJson(const TVector<TUserInfo>& usersInfos, const TVector<TServiceTaskInfo>& serviceTasksInfos, const TGeneralOptions& generalOptions, const TRoutingOptions& routingOptions) {
        NJson::TJsonValue result;
        TSet<TServiceTaskId> tasksPool;
        for (const auto& serviceTask : serviceTasksInfos) {
            tasksPool.insert(serviceTask.GetId());
        }
        for (const auto& userInfo : usersInfos) {
            NJson::TJsonValue vehicleJson;
            vehicleJson["cost"]["fixed"] = 0;  // no penalty for using additional vehicles
            vehicleJson["cost"]["hour"] = routingOptions.OptionalCostPerHour().GetOrElse(100); // penalty to minimize total time
            vehicleJson["id"] = userInfo.GetId();
            vehicleJson["start_at"] = userInfo.GetGarageId();
            vehicleJson["visit_depot_at_start"] = false;
            vehicleJson["return_to_depot"] = false;
            vehicleJson["tags"] = NJson::ToJson(userInfo.GetAvailableZones());
            for (const auto& taskId : userInfo.GetExplicitlyAvailableTasksIds()) {
                vehicleJson["tags"].AppendValue(taskId); // for enabling access to tasks, dropped by someone
            }
            for (const auto& tagName : routingOptions.GetServiceTasksTagNames()) {
                vehicleJson["tags"].AppendValue(tagName); // for enabling access to tasks with common tag names
            }
            for (const auto& tagName : userInfo.GetAdditionalTagNames()) {
                vehicleJson["tags"].AppendValue(tagName); // for enabling access to tasks with custom tag names
            }
            vehicleJson["excluded_tags"] = NJson::ToJson(userInfo.GetDroppedTasksIds());
            for (const auto& serviceTaskId : userInfo.GetObligatoryTasksIds()) { // these tasks will be the first in the route
                if (tasksPool.contains(serviceTaskId)) {
                    vehicleJson["visited_locations"].AppendValue(NJson::TMapBuilder("id", serviceTaskId));
                }
            }
            if (routingOptions.HasCostPerKilometer()) {
                vehicleJson["cost"]["km"] = routingOptions.GetCostPerKilometerRef();
            }
            auto& shift = vehicleJson["shifts"].AppendValue(NJson::JSON_NULL);
            shift["time_window"] = generalOptions.GetShift();
            shift["id"] = "main";
            if (routingOptions.GetHardWindow()) {
                shift["hard_window"] = true;
            }
            if (routingOptions.HasShiftPenaltyOutOfTimeFixed()) {
                shift["penalty"]["out_of_time"]["fixed"] = routingOptions.GetShiftPenaltyOutOfTimeFixedRef();
            }
            if (routingOptions.HasShiftPenaltyOutOfTimeMinute()) {
                shift["penalty"]["out_of_time"]["minute"] = routingOptions.GetShiftPenaltyOutOfTimeMinuteRef();
            }
            result.AppendValue(std::move(vehicleJson));
        }
        return result;
    }

    NJson::TJsonValue SerializeRoutingOptionsToJson(const TGeneralOptions& /* generalOptions */, const TRoutingOptions& routingOptions) {
        NJson::TJsonValue result;
        result["quality"] = ToString(routingOptions.GetRoutingQuality());
        result["absolute_time"] = routingOptions.GetUseAbsoluteTimeInResponse();
        result["avoid_tolls"] = routingOptions.GetAvoidTolls();
        // make time zone equal to 0 for easiness of calculating offsets
        Y_UNUSED(routingOptions.GetTimeZone());
        result["time_zone"] = 0;
        result["date"] = "1970-01-01";
        result["restart_on_drop"] = routingOptions.GetRestartOnDrop();
        if (routingOptions.HasGlobalProximityFactor()) {
            result["global_proximity_factor"] = routingOptions.GetGlobalProximityFactorRef();
        }
        return result;
    }

    NJson::TJsonValue SerializeDepotToJson(const TGeneralOptions& options) {
        // the field is fake, but it is necessary in query
        NJson::TJsonValue result;
        result["point"]["lat"] = 55.8;
        result["point"]["lon"] = 37.4;
        result["id"] = "depot";
        result["time_window"] = options.GetShift();
        return result;
    }

    TString GetCurrentDay(TInstant timestamp, const int timezone) {
        if (timezone > 0) {
            timestamp += TDuration::Hours(timezone);
        } else {
            timestamp -= TDuration::Hours(std::abs(timezone));
        }
        return ToString(TInstant::Days(timestamp.Days())).substr(0, 10);
    }

    TExpected<TRoutingTask, TEnqueueError> EnqueueRoutingTask(const TVector<TUserInfo>& usersInfos, const TVector<TServiceTaskInfo>& serviceTasksInfos,
                                                                const TGeneralOptions& generalOptions, const TRoutingOptions& routingOptions, TDuration timeout) {
        NJson::TJsonValue post;
        post["locations"] = SerializeLocationsToJson(usersInfos, serviceTasksInfos, generalOptions, routingOptions);
        post["vehicles"] = SerializeVehiclesToJson(usersInfos, serviceTasksInfos, generalOptions, routingOptions);
        post["options"] = SerializeRoutingOptionsToJson(generalOptions, routingOptions);
        post["depot"] = SerializeDepotToJson(generalOptions);

        TString relativeUrl = "/vrs/api/v1/add/mvrp?apikey=" + GetEnv("ROUTING_API_KEY");
        TStringStream ss;
        TKeepAliveHttpClient::THttpCode code;
        try {
            code = TKeepAliveHttpClient("https://courier.yandex.ru", 443, timeout, timeout).DoPost(relativeUrl, ToString(post), &ss);
        } catch (const std::exception& e) {
            TEnqueueError error;
            error.SetStatus(TEnqueueError::EStatus::ServerError);
            error.SetMessage(TStringBuilder() << "errors while sending request: " << FormatExc(e));
            return MakeUnexpected(error);
        }
        NJson::TJsonValue value;
        if (!NJson::ReadJsonFastTree(ss.Str(), &value)) {
            TEnqueueError error;
            error.SetStatus(TEnqueueError::EStatus::ServerError);
            error.SetMessage(TStringBuilder() << "can't parse reply content: " << ss.Str());
            return MakeUnexpected(error);
        }
        NDrive::TEventLog::Log("RoutingMRVPAddRequest", NJson::TMapBuilder
                                        ("request", std::move(post))
                                        ("reply", value)
                                        ("code", code));
        if (code != 202) {
            TEnqueueError error;
            error.SetStatus(static_cast<TEnqueueError::EStatus>(code));
            if (value.Has("message")) {
                error.SetMessage(value["message"].GetStringRobust());
            }
            return MakeUnexpected(error);
        }
        TRoutingTask result;
        if (!NJson::ParseField(value["id"], result.MutableRoutingTaskId())) {
            TEnqueueError error;
            error.SetStatus(TEnqueueError::EStatus::ServerError);
            error.SetMessage("incorrect value in field \"id\"");
            return MakeUnexpected(error);
        }
        auto& serviceTasksDurations = result.MutableServiceTasksDurations();
        for(const auto& serviceTaskInfo : serviceTasksInfos) {
            serviceTasksDurations[serviceTaskInfo.GetId()] = serviceTaskInfo.GetServiceDuration().Seconds();
        }
        result.SetAnchor(TInstant{});
        return result;
    }

    TExpected<TMap<TUserId, TServiceRoute>, TRetrieveError> RetrieveRoutingTask(const TRoutingTask& routingTask, bool emptyNonUsedWorkersRouteTags) {
        TString relativeUrl = "/vrs/api/v1/result/mvrp/" + routingTask.GetRoutingTaskId();
        TStringStream ss;
        TKeepAliveHttpClient::THttpCode code;
        try {
            code = TKeepAliveHttpClient("https://courier.yandex.ru", 443).DoGet(relativeUrl, &ss);
        } catch (const std::exception& e) {
            TRetrieveError error;
            error.SetStatus(TRetrieveError::EStatus::ServerError);
            error.SetMessage(TStringBuilder() << "errors while sending request: " << FormatExc(e));
            return MakeUnexpected(error);
        }

        NJson::TJsonValue value;
        if (!NJson::ReadJsonFastTree(ss.Str(), &value)) {
            TRetrieveError error;
            error.SetStatus(TRetrieveError::EStatus::ServerError);
            error.SetMessage(TStringBuilder() << "can't parse reply content: " << ss.Str());
            return MakeUnexpected(error);
        }
        NDrive::TEventLog::Log("RoutingMRVPGetRequest", NJson::TMapBuilder
            ("reply", value)
            ("routing_task_id", routingTask.GetRoutingTaskId())
            ("code", code)
        );
        if (code != 200) {
            TRetrieveError error;
            error.SetStatus(static_cast<TRetrieveError::EStatus>(code));
            if (value["error"]["message"].IsString()) {
                error.SetMessage(value["error"]["message"].GetString());
            }
            return MakeUnexpected(error);
        }
        if (!value["result"]["routes"].IsArray()) {
            return MakeUnexpected(InvalidFieldError("routes"));
        }
        TMap<TUserId, TServiceRoute> serviceRoutes;
        for (const auto& routeInfo : value["result"]["routes"].GetArray()) {
            TUserId userId;
            if (!NJson::ParseField(routeInfo["vehicle_id"], userId, true)) {
                return MakeUnexpected(InvalidFieldError("vehicle_id"));
            }
            if (!routeInfo["route"].IsArray()) {
                return MakeUnexpected(InvalidFieldError("route"));
            }
            TServiceRoute serviceRoute;
            for (const auto& task : routeInfo["route"].GetArray()) {
                double arrivalTimeSeconds;
                double transitDurationSeconds;
                TString taskId;
                if (!NJson::ParseField(task["arrival_time_s"], arrivalTimeSeconds, true)) {
                    return MakeUnexpected(InvalidFieldError("arrival_time_s"));
                }
                if (!NJson::ParseField(task["transit_duration_s"], transitDurationSeconds, true)) {
                    return MakeUnexpected(InvalidFieldError("transit_duration_s"));
                }
                if (!NJson::ParseField(task["node"]["value"]["id"], taskId)) {
                    return MakeUnexpected(InvalidFieldError("id"));
                }
                serviceRoute.MutableTasks().emplace_back().SetTagId(taskId).SetStartTime(routingTask.GetAnchor() + TDuration::Seconds(static_cast<ui64>(arrivalTimeSeconds)));
                if (serviceRoute.GetTasks().size() >= 2) {
                    auto it = routingTask.GetServiceTasksDurations().find(taskId);
                    Y_ENSURE(it != routingTask.GetServiceTasksDurations().end());
                    ui64 taskDurationSeconds = it->second;
                    serviceRoute.MutableTasks().back().SetEndTime(routingTask.GetAnchor() + TDuration::Seconds(static_cast<ui64>(arrivalTimeSeconds + taskDurationSeconds)));
                } else {
                    serviceRoute.MutableTasks().back().SetEndTime(serviceRoute.MutableTasks().back().GetStartTime());
                }
            }
            serviceRoute.SetRoutingTaskId(routingTask.GetRoutingTaskId());
            serviceRoutes[userId] = std::move(serviceRoute);
        }
        if (emptyNonUsedWorkersRouteTags) {
            if (!value["result"]["vehicles"].IsArray()) {
                return MakeUnexpected(InvalidFieldError("result.vehicles"));
            }
            for (const auto& vehicle : value["result"]["vehicles"].GetArray()) {
                if (!vehicle["id"].IsString()) {
                    return MakeUnexpected(InvalidFieldError("result.vehicles.id"));
                }
                if (!serviceRoutes.contains(vehicle["id"].GetString())) {
                    serviceRoutes[vehicle["id"].GetString()] = TServiceRoute{};
                }
            }
        }
        return serviceRoutes;
    }

    TString TServiceRouteCalculatorProcess::GetTypeName() {
        return "service_route_calculator";
    }

    TString TServiceRouteCalculatorProcess::GetType() const {
        return GetTypeName();
    }

    NDrive::TScheme TServiceRouteCalculatorProcess::DoGetScheme(const IServerBase& server) const {
        NDrive::TScheme scheme = TBase::DoGetScheme(server);
        scheme.Add<TFSNumeric>("routes_recalculation_period", "Routes recalculation period (seconds)").SetRequired(true);
        scheme.Add<TFSNumeric>("worker_activeness_threshold", "Worker activeness threshold (seconds)").SetDefault(30).SetRequired(true);
        scheme.Add<TFSBoolean>("check_worker_activeness", "Filter workers by worker activeness threshold").SetDefault(false);
        auto& frServer = *Yensured(server.GetAs<NDrive::IServer>());
        scheme.Add<TFSVariants>("service_tasks_tags", "Service tasks' tags to build routes from").SetVariants(
            NContainer::Keys(frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car))
        ).SetMultiSelect(true);
        scheme.Add<TFSVariants>("service_workers_roles", "Service worker roles").SetVariants(
            NContainer::Keys(frServer.GetDriveAPI()->GetRolesManager()->GetRolesDB().GetCachedObjectsMap())
        ).SetMultiSelect(true);
        scheme.Add<TFSVariants>("forbidden_car_tags", "Forbidden tags for cars:").SetVariants(
            NContainer::Keys(frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car))
        ).SetMultiSelect(true);
        scheme.Add<TFSVariants>("excluded_areas", "Exclude cars in areas:").SetVariants(
            frServer.GetDriveAPI()->GetAreasDB()->GetAreaIds()
        ).SetMultiSelect(true);

        scheme.Add<TFSVariants>("routing_quality", "Routing quality").SetVariants(GetEnumAllValues<ERoutingQuality>());
        scheme.Add<TFSArray>("shifts", "Shifts").SetElement(TShiftDescription::GetScheme(frServer.GetDriveAPI()->GetAreasDB()->GetAreaIds()));
        scheme.Add<TFSBoolean>("use_utilized_zones_only", "Фильтровать задачи по назначенным зонам").SetDefault(false);
        scheme.Add<TFSBoolean>("restart_on_drop", "Restart on drop").SetDefault(false);
        scheme.Add<TFSNumeric>("cost_per_kilometer", "Cost per kilometer");
        scheme.Add<TFSNumeric>("cost_per_hour", "Cost per hour");
        scheme.Add<TFSNumeric>("global_proximity_factor", "Global proximity factor");
        scheme.Add<TFSString>("routing_task_max_age", "Routing task max age");
        scheme.Add<TFSBoolean>("fix_performer_tasks", "Fix performer tasks");
        scheme.Add<TFSBoolean>("hard_window", "Shifts hard window");
        scheme.Add<TFSNumeric>("shift.penalty.out_of_time.fixed", "shift.penalty.out_of_time.fixed");
        scheme.Add<TFSNumeric>("shift.penalty.out_of_time.minute", "shift.penalty.out_of_time.minute");
        scheme.Add<TFSBoolean>("use_car_location", "Use car location").SetDefault(false);
        scheme.Add<TFSBoolean>("one_task_per_car", "Leave one task per car").SetDefault(false);
        scheme.Add<TFSBoolean>("tasks_hard_window", "Tasks hard window");
        scheme.Add<TFSBoolean>("empty_non_used_workers_route_tags", "Empty non-used workers' route tags").SetDefault(false);
        scheme.Add<TFSBoolean>("mark_cars_in_routing", "Mark cars in routing").SetDefault(false);
        scheme.Add<TFSString>("additional_tag_names_lifetime", "Additional tag names lifetime");

        {
            NDrive::TScheme tasksPenaltiesScheme;
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.drop", "penalty.drop");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.out_of_time.fixed", "penalty.out_of_time.fixed");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.out_of_time.minute", "penalty.out_of_time.minute");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.early.fixed", "penalty.early.fixed");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.early.minute", "penalty.early.minute");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.late.fixed", "penalty.late.fixed");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.late.minute", "penalty.late.minute");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.delivery_deadline.fixed", "penalty.delivery_deadline.fixed");
            tasksPenaltiesScheme.Add<TFSNumeric>("penalty.delivery_deadline.minute", "penalty.delivery_deadline.minute");
            NDrive::TScheme tasksPenaltiesFullScheme;
            tasksPenaltiesFullScheme.Add<TFSVariants>("tag_name", "Tag name").SetVariants(
                NContainer::Keys(frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car))
            );
            tasksPenaltiesFullScheme.Add<TFSString>("sla", "tag SLA");
            tasksPenaltiesFullScheme.Add<TFSStructure>("penalties", "Penalties").SetStructure(tasksPenaltiesScheme);

            auto guard = scheme.StartTabGuard("tags_settings");
            scheme.Add<TFSStructure>("default_penalties").SetStructure(tasksPenaltiesScheme);
            scheme.Add<TFSArray>("tags_penalties").SetElement(tasksPenaltiesFullScheme);
        }
        return scheme;
    }

    bool TServiceRouteCalculatorProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
        if (!TBase::DoDeserializeFromJson(value)) {
            return false;
        }
        ui64 periodSeconds;
        if (!NJson::ParseField(value["routes_recalculation_period"], periodSeconds, true)) {
            return false;
        }
        RoutesRecalculationPeriod = TDuration::Seconds(periodSeconds);

        ui64 workerActivenessThresholdSeconds;
        if (!NJson::ParseField(value["worker_activeness_threshold"], workerActivenessThresholdSeconds, true)) {
            return false;
        }
        WorkerActivenessThreshold = TDuration::Seconds(workerActivenessThresholdSeconds);
        return NJson::ParseField(value["service_tasks_tags"], RoutingOptions.MutableServiceTasksTagNames())
            && NJson::ParseField(value["service_workers_roles"], ServiceWorkersRolesNames)
            && NJson::ParseField(value["check_worker_activeness"], CheckWorkerActiveness)
            && NJson::ParseField(value["routing_quality"], NJson::Stringify(RoutingOptions.MutableRoutingQuality()))
            && NJson::ParseField(value["shifts"], Shifts)
            && NJson::ParseField(value["forbidden_car_tags"], ForbiddenCarTagsNames)
            && NJson::ParseField(value["use_utilized_zones_only"], UseUtilizedZonesOnly)
            && NJson::ParseField(value["restart_on_drop"], RoutingOptions.MutableRestartOnDrop())
            && NJson::ParseField(value["cost_per_kilometer"], RoutingOptions.OptionalCostPerKilometer())
            && NJson::ParseField(value["cost_per_hour"], RoutingOptions.OptionalCostPerHour())
            && NJson::ParseField(value["global_proximity_factor"], RoutingOptions.OptionalGlobalProximityFactor())
            && NJson::ParseField(value["hard_window"], RoutingOptions.MutableHardWindow())
            && NJson::ParseField(value["default_penalties"], RoutingOptions.MutableDefaultPenalties())
            && NJson::ParseField(value["shift.penalty.out_of_time.fixed"], RoutingOptions.OptionalShiftPenaltyOutOfTimeFixed())
            && NJson::ParseField(value["shift.penalty.out_of_time.minute"], RoutingOptions.OptionalShiftPenaltyOutOfTimeMinute())
            && NJson::ParseField(value["use_car_location"], RoutingOptions.MutableUseCarLocation())
            && NJson::ParseField(value["one_task_per_car"], RoutingOptions.MutableOneTaskPerCar())
            && NJson::ParseField(value["tasks_hard_window"], RoutingOptions.MutableTasksHardWindow())
            && NJson::ParseField(value["routing_task_max_age"], RoutingTaskMaxAge)
            && NJson::ParseField(value["excluded_areas"], ExcludedAreas)
            && NJson::ParseField(value["fix_performer_tasks"], FixPerformerTasks)
            && NJson::ParseField(value["empty_non_used_workers_route_tags"], EmptyNonUsedWorkersRouteTags)
            && NJson::ParseField(value["mark_cars_in_routing"], MarkCarsInRouting)
            && NJson::ParseField(value["additional_tag_names_lifetime"], AdditionalTagNamesLifetime)
            && NJson::ParseField(value["tags_penalties"], NJson::KeyValue(RoutingOptions.MutableTagNameToPenalties(), "tag_name", "penalties"))
            && NJson::ParseField(value["tags_penalties"], NJson::KeyValue(RoutingOptions.MutableTagNameToSLA(), "tag_name", "sla"));
    }

    NJson::TJsonValue TServiceRouteCalculatorProcess::DoSerializeToJson() const {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        result["routes_recalculation_period"] = RoutesRecalculationPeriod.Seconds();
        result["worker_activeness_threshold"] = WorkerActivenessThreshold.Seconds();
        result["service_tasks_tags"] = NJson::ToJson(RoutingOptions.GetServiceTasksTagNames());
        result["service_workers_roles"] = NJson::ToJson(ServiceWorkersRolesNames);
        result["check_worker_activeness"] = NJson::ToJson(CheckWorkerActiveness);
        result["routing_quality"] = NJson::ToJson(ToString(RoutingOptions.GetRoutingQuality()));
        result["shifts"] = NJson::ToJson(Shifts);
        result["forbidden_car_tags"] = NJson::ToJson(ForbiddenCarTagsNames);
        result["use_utilized_zones_only"] = NJson::ToJson(UseUtilizedZonesOnly);
        result["restart_on_drop"] = NJson::ToJson(RoutingOptions.GetRestartOnDrop());
        result["cost_per_kilometer"] = NJson::ToJson(RoutingOptions.OptionalCostPerKilometer());
        result["cost_per_hour"] = NJson::ToJson(RoutingOptions.OptionalCostPerHour());
        result["global_proximity_factor"] = NJson::ToJson(RoutingOptions.OptionalGlobalProximityFactor());
        result["hard_window"] = NJson::ToJson(RoutingOptions.GetHardWindow());
        result["default_penalties"] = NJson::ToJson(RoutingOptions.GetDefaultPenalties());
        result["shift.penalty.out_of_time.fixed"] = NJson::ToJson(RoutingOptions.OptionalShiftPenaltyOutOfTimeFixed());
        result["shift.penalty.out_of_time.minute"] = NJson::ToJson(RoutingOptions.OptionalShiftPenaltyOutOfTimeMinute());
        result["use_car_location"] = NJson::ToJson(RoutingOptions.GetUseCarLocation());
        result["one_task_per_car"] = NJson::ToJson(RoutingOptions.GetOneTaskPerCar());
        result["tasks_hard_window"] = NJson::ToJson(RoutingOptions.GetTasksHardWindow());
        result["routing_task_max_age"] = NJson::ToJson(RoutingTaskMaxAge);
        result["excluded_areas"] = NJson::ToJson(ExcludedAreas);
        result["fix_performer_tasks"] = NJson::ToJson(FixPerformerTasks);
        result["empty_non_used_workers_route_tags"] = NJson::ToJson(EmptyNonUsedWorkersRouteTags);
        result["mark_cars_in_routing"] = NJson::ToJson(MarkCarsInRouting);
        if (AdditionalTagNamesLifetime) {
            result["additional_tag_names_lifetime"] = NJson::ToJson(NJson::Hr(*AdditionalTagNamesLifetime));
        }
        const auto& tagNameToSLA = RoutingOptions.GetTagNameToSLA();
        for (auto& [tagName, penalties] : RoutingOptions.GetTagNameToPenalties()) {
            NJson::TJsonValue tagNameSettings;
            tagNameSettings["penalties"] = NJson::ToJson(penalties);
            tagNameSettings["tag_name"] = NJson::ToJson(tagName);
            if (auto it = tagNameToSLA.find(tagName); it != tagNameToSLA.end()) {
                tagNameSettings["sla"] = NJson::ToJson(NJson::Hr(it->second));
            }
            result["tags_penalties"].AppendValue(std::move(tagNameSettings));
        }
        return result;
    }

    TVector<TMaybe<NDrive::TLocation>> TServiceRouteCalculatorProcess::GetWorkersCarsPositions(const TExecutionContext& context, const TVector<TUserId>& workersIds) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        auto now = Now();
        TVector<TMaybe<NDrive::TLocation>> result;
        TMap<TUserId, TString> workerIdToCarId;
        auto sessionsBuilder = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
        if (!sessionsBuilder) {
            ERROR_LOG << GetRobotId() << ": sessionsBuilder is not configured" << Endl;
            result.assign(workersIds.size(), Nothing());
            return result;
        }
        auto workersIdsSet = MakeSet(workersIds);
        auto actualSessions = sessionsBuilder->GetSessionsActual(now - TDuration::Seconds(1), now);
        for (const auto& [userId, userSessions] : actualSessions) {
            if (!workersIdsSet.contains(userId)) {
                continue;
            }
            for (const auto& userSession : userSessions) {
                if (userSession->GetClosed()) {
                    continue;
                }
                workerIdToCarId[userId] = userSession->GetObjectId();
                break;
            }
        }
        const auto& snapshotsManager = server.GetSnapshotsManager();
        for (const auto& workerId : workersIds) {
            auto it = workerIdToCarId.find(workerId);
            if (it == workerIdToCarId.end()) {
                result.push_back(Nothing());
                continue;
            }
            const auto& carId = it->second;
            auto snapshot = snapshotsManager.GetSnapshot(carId);
            result.push_back(snapshot.GetLocation());
        }
        return result;
    }

    TVector<TMaybe<TGeoCoord>> TServiceRouteCalculatorProcess::GetWorkersPositions(const TExecutionContext& context, const TVector<TUserId>& workersIds) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        TVector<NThreading::TFuture<NDrive::TRealtimeUserData>> realtimeUsersInfos;
        realtimeUsersInfos.reserve(workersIds.size());
        auto now = Now();
        {
            const auto& userEventsApi = *Yensured(server.GetUserEventsApi());
            for (const auto& id : workersIds) {
                realtimeUsersInfos.push_back(userEventsApi.GetRealtimeUserInfo(id, true, false));
            }
        }
        TVector<TMaybe<TGeoCoord>> result(workersIds.size(), Nothing());
        TVector<TMaybe<NDrive::TLocation>> workersCarsLocations;

        if (RoutingOptions.GetUseCarLocation()) {
            workersCarsLocations = GetWorkersCarsPositions(context, workersIds);
        } else {
            workersCarsLocations.assign(workersIds.size(), Nothing());
        }
        for (size_t i = 0; i < workersIds.size(); ++i) {
            TMaybe<NDrive::TRealtimeUserData> userData;
            try {
                userData = realtimeUsersInfos[i].ExtractValueSync();
            } catch (const std::exception& e) {
                ERROR_LOG << GetRobotId() << ": can't fetch realtime data for user " << workersIds[i] << "; " << FormatExc(e) << Endl;
            }
            TMaybe<std::pair<TGeoCoord, TInstant>> freshPosition;
            if (workersCarsLocations[i] && userData) {
                if (workersCarsLocations[i]->Timestamp < userData->GetLastLocationUpdate()) {
                    freshPosition = {userData->GetLocation(), userData->GetLastLocationUpdate()};
                } else {
                    freshPosition = {workersCarsLocations[i]->GetCoord(), workersCarsLocations[i]->Timestamp};
                }
            } else if (workersCarsLocations[i]) {
                freshPosition = {workersCarsLocations[i]->GetCoord(), workersCarsLocations[i]->Timestamp};
            } else if (userData) {
                freshPosition = {userData->GetLocation(), userData->GetLastLocationUpdate()};
            }
            if (freshPosition && (!CheckWorkerActiveness || now - WorkerActivenessThreshold < freshPosition->second)) {
                result[i] = freshPosition->first;
            }
        }
        return result;
    }

    TVector<TUserInfo> TServiceRouteCalculatorProcess::FetchActiveWorkers(const TExecutionContext& context) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        const TDriveAPI& api = *Yensured(server.GetDriveAPI());
        auto session = api.BuildTx<NSQL::ReadOnly>();
        const auto& usersData = *Yensured(api.GetUsersData());

        TVector<TUserId> workersIds;
        Y_ENSURE(usersData.GetRoles().GetUsersWithRoles(ServiceWorkersRolesNames, workersIds, true), "can't fetch workers");

        auto workersPositions = GetWorkersPositions(context, workersIds);
        Y_ENSURE(workersPositions.size() == workersIds.size());
        TString tagName = server.GetSettings().GetValue<TString>(TServiceZonesTag::AvailableZonesTagNameSettings).GetOrElse(TServiceZonesTag::DefaultAvailableZonesTagName);
        TMap<TString, TDBTag> workerToServiceZonesDBTag;
        {
            TVector<TDBTag> dbTags;
            Y_ENSURE(api.GetTagsManager().GetUserTags().RestoreTags(MakeSet(workersIds), {tagName}, dbTags, session), "can't fetch service tags: " << session.GetStringReport());
            for (auto& dbTag : dbTags) {
                workerToServiceZonesDBTag[dbTag.GetObjectId()] = std::move(dbTag);
            }
        }

        TMap<TUserId, TVector<TServiceTaskId>> workerToPerformedTagIds;
        if (FixPerformerTasks) {
            TSet<TString> availableTagNames = RoutingOptions.GetServiceTasksTagNames();
            for (const auto& [_, dbTag] : workerToServiceZonesDBTag) {
                auto* tag = dbTag.GetTagAs<TServiceZonesTag>();
                if (tag) {
                    const auto& additionalTagNames = tag->GetAdditionalTagNames();
                    availableTagNames.insert(additionalTagNames.begin(), additionalTagNames.end());
                }
            }
            TVector<TDBTag> dbTags;
            Y_ENSURE(api.GetTagsManager().GetDeviceTags().RestorePerformerTags(MakeVector(availableTagNames), workersIds, dbTags, session), "can't fetch service tags: " << session.GetStringReport());
            for (auto& dbTag : dbTags) {
                workerToPerformedTagIds[dbTag->GetPerformer()].push_back(dbTag.GetTagId());
            }
        }

        TMap<TUserId, TSet<TServiceTaskId>> workerToDroppedTasks;
        bool reportDroppedTasks = server.GetSettings().GetValueDef<bool>("routing.report_dropped_tasks", true);
        if (!reportDroppedTasks) {
            workerToDroppedTasks = GetWorkerToDroppedTasks(server, api, MakeSet(workersIds), session);
        }

        TVector<TUserInfo> activeWorkersInfo;
        for (size_t i = 0; i < workersIds.size(); ++i) {
            const auto& maybeLocation = workersPositions[i];
            if (!maybeLocation) {
                continue;
            }
            const auto& location = *maybeLocation;
            TUserInfo userInfo;
            userInfo.SetId(workersIds[i]);
            if (std::abs(location.X) < 1e-6 && std::abs(location.Y) < 1e-6) {
                ERROR_LOG << GetRobotId() << ": user " << workersIds[i] << " has coordinates that are zeros" << Endl;
                continue;
            }
            if (auto it = workerToServiceZonesDBTag.find(workersIds[i]); it != workerToServiceZonesDBTag.end()) {
                auto* tag = it->second.MutableTagAs<TServiceZonesTag>();
                if (tag) {
                    userInfo.SetAvailableZones(std::move(tag->MutableAvailableZones()));
                    userInfo.SetAdditionalTagNames(std::move(tag->MutableAdditionalTagNames()));
                }
            }
            if (auto it = workerToPerformedTagIds.find(workersIds[i]); it != workerToPerformedTagIds.end()) {
                userInfo.SetObligatoryTasksIds(std::move(it->second));
            }
            if (auto it = workerToDroppedTasks.find(workersIds[i]); it != workerToDroppedTasks.end()) {
                userInfo.SetDroppedTasksIds(std::move(it->second));
            }
            userInfo.SetPosition(location);
            activeWorkersInfo.push_back(std::move(userInfo));
        }
        return activeWorkersInfo;
    }

    TDuration TServiceRouteCalculatorProcess::GetServiceTaskDurationByTagName() const {
        return TDuration::Minutes(30);
    }

    TVector<TServiceTaskInfo> TServiceRouteCalculatorProcess::FetchTasks(const TExecutionContext& context, const TVector<TUserInfo>& activeWorkersInfos, const TGeneralOptions& generalOptions) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        const TDriveAPI& api = *Yensured(server.GetDriveAPI());
        auto session = api.BuildTx<NSQL::ReadOnly>();
        const auto& areasDB = *Yensured(api.GetAreasDB());
        TVector<TDBTag> dbTags;
        TSet<TString> allowedTagNames = RoutingOptions.GetServiceTasksTagNames();
        for (const auto& worker : activeWorkersInfos) {
            const auto& additionalTagNames = worker.GetAdditionalTagNames();
            allowedTagNames.insert(additionalTagNames.begin(), additionalTagNames.end());
        }
        Y_ENSURE(api.GetTagsManager().GetDeviceTags().RestoreTags({}, MakeVector(allowedTagNames), dbTags, session), "can't fetch service tags:" << session.GetStringReport());
        TVector<TDBTag> forbiddenCarDBTags;
        if (!ForbiddenCarTagsNames.empty()) {
            Y_ENSURE(api.GetTagsManager().GetDeviceTags().RestoreTags({}, MakeVector(ForbiddenCarTagsNames), forbiddenCarDBTags, session), "can't fetch forbidden car tags:" << session.GetStringReport());
        }
        TSet<TString> forbiddenCars;
        for (const auto& dbTag : forbiddenCarDBTags) {
            forbiddenCars.insert(dbTag.GetObjectId());
        }
        TSet<TString> carsToFetch;
        for (const auto& dbTag : dbTags) {
            if (!forbiddenCars.contains(dbTag.GetObjectId())) {
                carsToFetch.insert(dbTag.GetObjectId());
            }
        }
        TTaggedObjectsSnapshot objectsSnapshot;
        Y_ENSURE(api.GetTagsManager().GetDeviceTags().RestoreObjects(carsToFetch, objectsSnapshot, session), "can't restore cars: " << session.GetStringReport());

        auto carIdToCarInfo = Yensured(api.GetCarsData())->GetCachedObjectsMap(carsToFetch);

        TVector<TServiceTaskInfo> result;
        TSet<TUserId> activeWorkersIds;
        TSet<TString> performedTasksIds;
        TMap<TString, TSet<TString>> serviceZoneToAvailableTagNames;
        for (const auto& worker : activeWorkersInfos) {
            activeWorkersIds.insert(worker.GetId());
            const auto& availableZones = worker.GetAvailableZones();
            const auto& performedTasks = worker.GetObligatoryTasksIds();
            performedTasksIds.insert(performedTasks.begin(), performedTasks.end());
            for (const auto& zone : availableZones) {
                const auto& commonTagNames = RoutingOptions.GetServiceTasksTagNames();
                serviceZoneToAvailableTagNames[zone].insert(commonTagNames.begin(), commonTagNames.end());
                const auto& additionalTagNames = worker.GetAdditionalTagNames();
                serviceZoneToAvailableTagNames[zone].insert(additionalTagNames.begin(), additionalTagNames.end());
            }
        }
        TMaybe<TSet<TString>> bestTagIds;
        auto tagIdToCreationTime = GetTagsCreationTimes(dbTags, api.GetTagsManager().GetDeviceTags(), session);
        if (RoutingOptions.GetOneTaskPerCar()) {
            bestTagIds = MakeSet(TakeBestTagIdPerCar(dbTags, performedTasksIds, tagIdToCreationTime));
        }
        TVector<TServiceTaskStatus> serviceTasksStatuses;
        for (const auto& dbTag : dbTags) {
            TServiceTaskStatus taskStatus;
            taskStatus.SetTaskId(dbTag.GetTagId());
            taskStatus.SetCarId(dbTag.GetObjectId());
            taskStatus.SetTagName(dbTag->GetName());
            auto dropTask = [&taskStatus, &serviceTasksStatuses](TStringBuf dropReason) {
                taskStatus.SetDropReason(dropReason);
                serviceTasksStatuses.push_back(taskStatus);
            };
            if (bestTagIds && !bestTagIds->contains(dbTag.GetTagId())) {
                dropTask("tag was not chosen as best per car");
                continue;
            }
            if (forbiddenCars.contains(dbTag.GetObjectId())) {
                dropTask("car is forbidden");
                continue;
            }
            const TTaggedObject* obj = objectsSnapshot.Get(dbTag.GetObjectId());
            if (!obj) {
                dropTask("can't restore car");
                continue;
            }
            if (IsBusyForServicing(*obj, activeWorkersIds)) {
                dropTask("is busy for servicing");
                continue;
            }
            TRTDeviceSnapshot snapshot = server.GetSnapshotsManager().GetSnapshot(dbTag.GetObjectId());
            auto maybeLocation = snapshot.GetRawLocation();
            if (!maybeLocation) {
                dropTask("can't find raw location of car");
                continue;
            }
            TString serviceZone = GetServiceZone(maybeLocation->GetCoord(), areasDB);
            if (!serviceZone) {
                dropTask("empty service zone");
                continue;
            }
            if (UseUtilizedZonesOnly && !serviceZoneToAvailableTagNames[serviceZone].contains(dbTag->GetName())) {
                dropTask(TStringBuilder() << "zone " << serviceZone << " doesn't contain tag " << dbTag->GetName() << " as available");
                continue;
            }
            if (auto zone = FindFirstEnclosingArea(maybeLocation->GetCoord(), ExcludedAreas, areasDB)) {
                dropTask(TStringBuilder() << "enclosed by excluded area: " << *zone);
                continue;
            }

            TServiceTaskInfo taskInfo;
            taskInfo.SetId(dbTag.GetTagId());
            taskInfo.SetPosition(maybeLocation->GetCoord());
            taskInfo.SetServiceDuration(GetServiceTaskDurationByTagName());
            taskInfo.SetZone(serviceZone);
            taskInfo.SetTagName(dbTag->GetName());
            if (auto it = RoutingOptions.GetTagNameToPenalties().find(dbTag->GetName()); it != RoutingOptions.GetTagNameToPenalties().end()) {
                taskInfo.SetPenalties(it->second);
            } else {
                taskInfo.SetPenalties(RoutingOptions.GetDefaultPenalties());
            }
            auto enclosingAreasIds = GetEnclosingAreasIds(maybeLocation->GetCoord(), areasDB);
            auto timeWindow = generalOptions.GetShiftDescription().GetRulesBasedTimeWindow(generalOptions.GetShiftStart(), MakeSet(enclosingAreasIds));
            if (auto itTime = tagIdToCreationTime.find(dbTag.GetTagId()); itTime != tagIdToCreationTime.end()) {
                timeWindow.first = itTime->second;
            }
            if (auto it = RoutingOptions.GetTagNameToSLA().find(dbTag->GetName()); it != RoutingOptions.GetTagNameToSLA().end() && it->second) {
                timeWindow.second = std::min(timeWindow.second, timeWindow.first + *(it->second));
            }
            taskInfo.SetTimeWindow(timeWindow);
            if (auto it = carIdToCarInfo.find(dbTag.GetObjectId()); it != carIdToCarInfo.end()) {
                taskInfo.SetCarNumber(it->second.GetNumber());
            }
            result.push_back(std::move(taskInfo));
        }
        NDrive::TEventLog::Log("RoutingMRVPFetchTasks", NJson::TMapBuilder("tasks_statuses", NJson::ToJson(serviceTasksStatuses)));
        return result;
    }

    TMap<TUserId, TInstant> TServiceRouteCalculatorProcess::UpdateAdditionalTagNamesState(TInstant now, TMap<TString, TInstant> lastSetAdditionalTagNames, const NDrive::IServer& server) const {
        const auto& api = *Yensured(server.GetDriveAPI());
        const auto& userTagsManager = api.GetTagsManager().GetUserTags();
        auto session = userTagsManager.BuildSession();
        TString tagName = server.GetSettings().GetValue<TString>(TServiceZonesTag::AvailableZonesTagNameSettings).GetOrElse(TServiceZonesTag::DefaultAvailableZonesTagName);
        TVector<TDBTag> dbTags;
        Y_ENSURE(api.GetTagsManager().GetUserTags().RestoreTags({}, {tagName}, dbTags, session), "can't fetch service tags: " << session.GetStringReport());

        for (auto& dbTag : dbTags) {
            auto* tag = dbTag.MutableTagAs<TServiceZonesTag>();
            if (!tag) {
                continue;
            }
            if (!tag->GetAdditionalTagNames().empty()) {
                if (auto it = lastSetAdditionalTagNames.find(dbTag.GetObjectId()); it != lastSetAdditionalTagNames.end()) {
                    if (AdditionalTagNamesLifetime && it->second + *AdditionalTagNamesLifetime < now) {
                        tag->MutableAdditionalTagNames().clear();
                        Y_ENSURE(userTagsManager.AddTag(dbTag.GetData(), GetRobotUserId(), it->first, &server, session), "can't update tag " << TServiceRouteTag::ServiceRouteTagName << ": " << session.GetStringReport());
                        lastSetAdditionalTagNames.erase(it);
                    }
                } else {
                    lastSetAdditionalTagNames[dbTag.GetObjectId()] = now;
                }
            }
        }
        Y_ENSURE(session.Commit(), TStringBuilder() << "can't commit session: " << session.GetStringReport());
        return lastSetAdditionalTagNames;
    }

    void TServiceRouteCalculatorProcess::AssignServiceTasks(const TExecutionContext& context, const TMap<TUserId, TServiceRoute>& routes) const {
        const auto& server = context.GetServerAs<NDrive::IServer>();
        const TDriveAPI& api = *Yensured(server.GetDriveAPI());
        const auto& userTagsManager = api.GetTagsManager().GetUserTags();
        auto session = userTagsManager.BuildSession();
        const auto& tagMeta = api.GetTagsManager().GetTagsMeta();
        for (const auto& [userId, serviceRoute] : routes) {
            auto tag = tagMeta.CreateTag(TServiceRouteTag::ServiceRouteTagName);
            Y_ENSURE(tag, "cannot create tag " << TServiceRouteTag::ServiceRouteTagName);
            auto* serviceRouteTag = dynamic_cast<TServiceRouteTag*>(tag.Get());
            Y_ENSURE(serviceRouteTag, "cannot cast tag " << TServiceRouteTag::ServiceRouteTagName << " as TServiceRouteTag");
            serviceRouteTag->SetRoute(serviceRoute); // copy by value
            Y_ENSURE(userTagsManager.AddTag(tag, GetRobotUserId(), userId, &server, session), "can't add tag " << TServiceRouteTag::ServiceRouteTagName << ": " << session.GetStringReport());
        }
        Y_ENSURE(session.Commit(), "cannot Commit: " << session.GetStringReport());
    }

    void TServiceRouteCalculatorProcess::UpdateCarsRelationsToRouting(const NDrive::IServer& server, const TSet<TString>& carsInRoutingIds, NDrive::TEntitySession& session) const {
        const auto& api = *Yensured(server.GetDriveAPI());
        const auto& deviceTagsManager = api.GetTagsManager().GetDeviceTags();
        const auto& carInRoutingTagName = server.GetSettings().GetValueDef<TString>("routing.car_in_routing_tag_name", "in_routing_tag");
        TVector<TDBTag> previousCarsInRoutingDBTags;
        Y_ENSURE(deviceTagsManager.RestoreTags({}, { carInRoutingTagName }, previousCarsInRoutingDBTags, session), "can't fetch car in routing tags: " << session.GetStringReport());
        TSet<TString> previousCarsInRoutingIds;
        for (const auto& dbTag : previousCarsInRoutingDBTags) {
            previousCarsInRoutingIds.insert(dbTag.GetObjectId());
        }
        for (auto& dbTag : previousCarsInRoutingDBTags) {
            if (!carsInRoutingIds.contains(dbTag.GetObjectId())) {
                // car is no longer in any route
                Y_ENSURE(deviceTagsManager.RemoveTagSimple(dbTag, GetRobotUserId(), session, false), "can't remove tag: " << session.GetStringReport());
            }
        }
        for (const auto& carId : carsInRoutingIds) {
            if (!previousCarsInRoutingIds.contains(carId)) {
                // car appeared in one of the routes
                auto tag = Yensured(api.GetTagsManager().GetTagsMeta().CreateTag(carInRoutingTagName));
                Y_ENSURE(deviceTagsManager.AddTag(tag, GetRobotUserId(), carId, &server, session), "can't add tag " << carInRoutingTagName << ": " << session.GetStringReport());
            }
        }
    }

    TMaybe<TShiftDescription> GetActualShiftDescription(TInstant now, const TVector<TShiftDescription>& shifts) {
        auto it = std::find_if(shifts.begin(), shifts.end(), [now](const TShiftDescription& shift) {
            return shift.IsActual(now);
        });
        return it != shifts.end() ? MakeMaybe(*it) : Nothing();
    }

    TExpectedState TServiceRouteCalculatorProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
        auto curState = dynamic_cast<const TServiceRouteCalculatorProcessState*>(state.Get());
        if (!curState && state) {
            ERROR_LOG << GetRobotId() << ": broken state " << state->GetType() << ' ' << state->GetReport().GetStringRobust() << Endl;
        }
        TVector<TRoutingTask> unfinishedTasks;
        const auto& server = context.GetServerAs<NDrive::IServer>();
        const auto& api = *Yensured(server.GetDriveAPI());
        if (curState) {
            TVector<TRoutingTask> routingTasks = curState->GetExecutingTasks();
            for (const auto& routingTask : routingTasks) {
                auto expectedRoutingTaskResult = RetrieveRoutingTask(routingTask, EmptyNonUsedWorkersRouteTags);
                if (expectedRoutingTaskResult) {
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName() + "-retrieve", "success", 1);
                    auto& routingTaskResult = *expectedRoutingTaskResult;
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "workers_number", routingTaskResult.size());
                    auto nonEmptyRoutesCount = 0;
                    for (const auto& [_, route] : routingTaskResult) {
                        if (!route.GetTasks().empty()) {
                            ++nonEmptyRoutesCount;
                        }
                    }
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "routes_number", nonEmptyRoutesCount);
                    EnrichRoutesWithCarsData(routingTaskResult, api);
                    AssignServiceTasks(context, routingTaskResult);
                    if (MarkCarsInRouting) {
                        const auto& carsIdsInRouting = GetCarsIdsFromRoutes(routingTaskResult);
                        // TODO(cezarnik): create TEntitySession only once per execution
                        auto session = api.BuildTx<NSQL::Writable>();
                        UpdateCarsRelationsToRouting(server, carsIdsInRouting, session);
                        Y_ENSURE(session.Commit(), "can't commit session: " << session.GetStringReport());
                        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "cars_in_routing", carsIdsInRouting.size());
                    }
                } else {
                    auto error = expectedRoutingTaskResult.GetError();
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName() + "-retrieve", "error-" + ToString(error.GetStatus()), 1);
                    if (error.GetStatus() != TRetrieveError::EStatus::IsRunning
                     && error.GetStatus() != TRetrieveError::EStatus::InQueue) {
                        NDrive::TEventLog::Log("RoutingMRVPError", NJson::TMapBuilder
                            ("id", routingTask.GetRoutingTaskId())
                            ("type", "retrieve")
                            ("code", static_cast<ui32>(error.GetStatus()))
                            ("message", error.GetMessage())
                        );
                    }

                    bool shouldAdd = !RoutingTaskMaxAge || Now() - routingTask.GetTimestamp() < *RoutingTaskMaxAge;
                    if (shouldAdd) {
                        unfinishedTasks.push_back(routingTask);
                    }
                }
            }
        }
        TInstant lastRoutesCalculationTime = curState ? curState->GetLastRoutesCalculationTime() : TInstant::Zero();
        auto newState = MakeAtomicShared<TServiceRouteCalculatorProcessState>();
        newState->SetLastRoutesCalculationTime(lastRoutesCalculationTime);
        TInstant now = Now();
        newState->SetLastSetAdditionalTags(UpdateAdditionalTagNamesState(now, curState ? curState->GetLastSetAdditionalTags() : TMap<TString, TInstant>{}, server));
        if (lastRoutesCalculationTime + RoutesRecalculationPeriod <= now) {
            auto maybeShift = GetActualShiftDescription(now, Shifts);
            if (maybeShift) {
                TGeneralOptions generalOptions;
                generalOptions.SetShiftDescription(*maybeShift);
                generalOptions.SetReference(now);
                auto workers = FetchActiveWorkers(context);
                auto serviceTasks = FetchTasks(context, workers, generalOptions);
                ExcludeDroppedTasks(serviceTasks, workers);
                TExpected<TRoutingTask, TEnqueueError> expectedRoutingTask = EnqueueRoutingTask(workers, serviceTasks, generalOptions, RoutingOptions, EnqueueHandlerTimeout);
                if (expectedRoutingTask) {
                    unfinishedTasks.push_back(*expectedRoutingTask);
                    unfinishedTasks.back().SetTimestamp(now);
                    newState->SetLastRoutesCalculationTime(now);
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName() + "-enqueue", "success", 1);
                } else {
                    auto error = expectedRoutingTask.GetError();
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName() + "-enqueue", "error-" + ToString(error.GetStatus()), 1);
                    ERROR_LOG << GetRobotId() << ": can't enqueue routing task; error_type=" << ToString(error.GetStatus()) << " message=" << error.GetMessage() << "\n";
                    NDrive::TEventLog::Log("RoutingMRVPError", NJson::TMapBuilder
                        ("type", "enqueue")
                        ("code", static_cast<ui32>(error.GetStatus()))
                        ("message", error.GetMessage())
                    );
                }
            } else {
                INFO_LOG << GetRobotId() << ": skipping " << now << " due to time restrictions" << Endl;
            }
        }
        newState->SetExecutingTasks(unfinishedTasks);
        return newState;
    }
};
