#include "process.h"

#include <drive/backend/data/servicing.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>

TExpectedState TServicingAutostartProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& deviceTagManager = server.GetDriveDatabase().GetTagsManager().GetDeviceTags();

    TDBTags tags;
    {
        auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly>();
        auto optionalTags = deviceTagManager.RestoreTags(TVector<TString>{}, TagNames, tx);
        R_ENSURE(optionalTags, {}, "cannot RestoreTags", tx);
        for (auto&& tag : *optionalTags) {
            auto servicingTag = tag.GetTagAs<TServicingTag>();
            if (!servicingTag) {
                WARNING_LOG << GetRobotId() << ": cannot cast tag " << tag.GetTagId() << " to ServicingTag" << Endl;
                continue;
            }
            if (!servicingTag->IsEligible(server)) {
                continue;
            }
            tags.push_back(std::move(tag));
        }
    }
    auto robotPermissions = server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), {});
    if (!robotPermissions) {
        return MakeUnexpected<TString>("cannot GetUserPermissions");
    }
    for (auto&& tag : tags) try {
        auto tx = deviceTagManager.BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        auto optionalTag = deviceTagManager.RestoreTag(tag.GetTagId(), tx);
        R_ENSURE(optionalTag, {}, "cannot RestoreTags", tx);

        auto servicingTag = optionalTag->GetTagAs<TServicingTag>();
        if (!servicingTag) {
            WARNING_LOG << GetRobotId() << ": cannot cast tag " << tag.GetTagId() << " to ServicingTag" << Endl;
            continue;
        }
        if (!servicingTag->IsEligible(server)) {
            continue;
        }
        R_ENSURE(servicingTag->Start(tag, *robotPermissions, server, tx), {}, "cannot Start servicing", tx);
        R_ENSURE(tx.Commit(), {}, "cannot Commit", tx);
    } catch (...) {
        ERROR_LOG << GetRobotId() << ": cannot process tag " << tag.GetTagId() << ": " << CurrentExceptionInfo(true).GetStringRobust() << Endl;
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TServicingAutostartProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto tagNames = api
        ? api->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TServicingTag::Type() })
        : TSet<TString>();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("tag_names").SetVariants(tagNames).SetMultiSelect(true);
    return scheme;
}

bool TServicingAutostartProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["tag_names"], TagNames) &&
        true;
}

NJson::TJsonValue TServicingAutostartProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["tag_names"] = NJson::ToJson(TagNames);
    return result;
}

TServicingAutostartProcess::TFactory::TRegistrator<TServicingAutostartProcess> TServicingAutostartProcess::Registrator(TServicingAutostartProcess::GetTypeName());
