#include "process.h"

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/sessions/manager/billing.h>

#include <drive/telematics/api/sensor/interface.h>

#include <library/cpp/iterator/iterate_keys.h>

TExpectedState TSessionCloseProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
    ui32 limit = 500;

    TDBTags orphans;
    TSessionManager::TConstSessions sessions;
    {
        auto tx = tagsManager.GetTraceTags().BuildTx<NSQL::ReadOnly>();
        auto optionalTags = tagsManager.GetTraceTags().RestoreTags(TVector<TString>{}, { TChargableSessionTag::Type() }, tx);
        R_ENSURE(optionalTags, {}, "cannot RestoreTags", tx);

        TMap<TString, TDBTag> sessionTags;
        ui32 queue = 0;
        for (auto&& tag : *optionalTags) {
            auto impl = tag.GetTagAs<TChargableSessionTag>();
            R_ENSURE(impl, HTTP_INTERNAL_SERVER_ERROR, "cannot cast to ChargableSessionTag " + tag.GetTagId(), tx);
            if (impl->GetSessionTagEntityType() == NEntityTagsManager::EEntityType::Car) {
                auto expectedTag = tagsManager.GetDeviceTags().GetTagFromCache(ToString(impl->GetSessionTagId()));
                if (expectedTag) {
                    continue;
                }
            }
            if (sessionTags.size() < limit) {
                sessionTags.emplace(tag.GetObjectId(), tag);
            } else {
                ++queue;
            }
        }
        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "queue", queue);

        if (sessionTags) {
            auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("session_close_process", &server);
            auto optionalCompiledSessions = server.GetDriveDatabase().GetCompiledSessionManager().Get<TMinimalCompiledRiding>(MakeVector(NContainer::Keys(sessionTags)), tx, ydbTx);
            R_ENSURE(optionalCompiledSessions, {}, "cannot Get MinimalCompiledRiding ", tx);
            for (const auto& session : *optionalCompiledSessions) {
                auto tag = sessionTags.extract(session.GetSessionId());
                if (tag) {
                    orphans.push_back(tag.mapped());
                }
                INFO_LOG << GetRobotId() << ": discovered already finished session " << session.GetSessionId() << Endl;
            }
            for (const auto& [id, tag] : sessionTags) {
                auto optionalSession = server.GetDriveDatabase().GetSessionManager().GetSession(tag.GetObjectId(), tx);
                R_ENSURE(optionalSession, {}, "cannot GetSession " << tag.GetObjectId(), tx);
                auto session = std::move(*optionalSession);
                auto threshold = Now() - FinishThreshold;
                if (session && session->GetClosed() && session->GetLastTS() < threshold) {
                    INFO_LOG << GetRobotId() << ": discovered unfinished session " << tag.GetObjectId() << Endl;
                    sessions.push_back(std::move(session));
                    continue;
                }
            }
        }
    }

    for (auto&& tag : orphans) {
        if (DryRun) {
            NOTICE_LOG << GetRobotId() << ": planning to remove orphan " << tag.GetTagId() << Endl;
            continue;
        }

        auto tx = tagsManager.GetTraceTags().BuildTx<NSQL::Writable>();
        auto removed = tagsManager.GetTraceTags().RemoveTag(tag, GetRobotUserId(), &server, tx);
        if (!removed || !tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": an exception occurred while processing orphan " << tag.GetTagId() << ": " << tx.GetStringReport() << Endl;
        }
    }

    for (auto&& session : sessions) try {
        Y_ENSURE(session);
        const auto& userId = session->GetUserId();
        const auto permissions = server.GetDriveDatabase().GetUserPermissions(userId, {});
        R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << userId);
        if (DryRun) {
            NOTICE_LOG << GetRobotId() << ": planning to finish session " << session->GetSessionId() << Endl;
            continue;
        }
        TChargableTag::CloseSession(session, *permissions, server, /*strict=*/false);
    } catch (...) {
        ERROR_LOG << GetRobotId() << ": an exception occurred while processing " << Yensured(session)->GetSessionId() << ": " << CurrentExceptionInfo().GetStringRobust() << Endl;
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TSessionCloseProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSDuration>("finish_threshold").SetDefault(FinishThreshold);
    return scheme;
}

bool TSessionCloseProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["finish_threshold"], FinishThreshold) &&
        true;
}

NJson::TJsonValue TSessionCloseProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["finish_threshold"] = NJson::ToJson(FinishThreshold);
    return result;
}

TExpectedState TSessionZeroEvolveProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& deviceTagManager = api->GetTagsManager().GetDeviceTags();
    const auto sessionBuilder = deviceTagManager.GetSessionsBuilder("billing");
    const auto sessions = Yensured(sessionBuilder)->GetSessionsActual();
    for (auto& session : sessions) {
        if (!session) {
            continue;
        }
        if (session->GetClosed()) {
            continue;
        }
        const auto& sessionId = session->GetSessionId();
        auto billingSession = std::dynamic_pointer_cast<TBillingSession>(session);
        if (!billingSession) {
            ERROR_LOG << GetRobotId() << ": cannot cast " << sessionId << " to BillingSession" << Endl;
            continue;
        }
        auto lastEvent = billingSession->GetLastEvent();
        if (!lastEvent) {
            ERROR_LOG << GetRobotId() << ": cannot GetLastEvent for " << sessionId << Endl;
            continue;
        }
        auto now = Now();
        if (now < lastEvent->GetHistoryTimestamp() + Threshold) {
            continue;
        }
        if (lastEvent.GetRef()->GetName() != TChargableTag::Riding) {
            continue;
        }
        auto tx = deviceTagManager.BuildSession(/*readOnly=*/false, /*repeatableRead=*/true);
        auto optionalEvents = deviceTagManager.GetEventsByTag(lastEvent->GetTagId(), tx, 0, {}, IEntityTagsManager::TQueryOptions(1, true));
        if (!optionalEvents) {
            ERROR_LOG << GetRobotId() << ": cannot GetEventsByTag for " << sessionId << ": " << tx.GetStringReport() << Endl;
            continue;
        }
        if (optionalEvents->empty()) {
            ERROR_LOG << GetRobotId() << ": empty events for " << sessionId << ": tag_id=" << lastEvent->GetTagId() << Endl;
            continue;
        }
        if (optionalEvents->front().GetHistoryEventId() != lastEvent->GetHistoryEventId()) {
            WARNING_LOG << GetRobotId() << ": session " << sessionId << " changed" << Endl;
            continue;
        }
        const auto& userId = session->GetUserId();
        auto permissions = api->GetUserPermissions(userId);
        if (!permissions) {
            ERROR_LOG << GetRobotId() << ": cannot GetUserPermissions for " << userId << Endl;
            continue;
        }
        tx.SetOriginatorId(GetRobotUserId());
        if (!TChargableTag::DirectEvolve(session, TChargableTag::Riding, *permissions, server, tx, nullptr)) {
            ERROR_LOG << GetRobotId() << ": cannot DirectEvolve for " << sessionId << ": " << tx.GetStringReport() << Endl;
            continue;
        }
        if (DryRun) {
            bool rolledBack = tx.Rollback();
            Y_UNUSED(rolledBack);
            NOTICE_LOG << GetRobotId() << ": dry run for " << sessionId << ": " << tx.GetStringReport() << Endl;
            continue;
        }
        auto hash = FnvHash<ui32>(sessionId);
        ui32 prec = 10000;
        if (hash % prec > Fraction * prec) {
            INFO_LOG << GetRobotId() << ": skip " << sessionId << Endl;
            continue;
        }
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit for " << sessionId << ": " << tx.GetStringReport() << Endl;
            continue;
        }
        INFO_LOG << GetRobotId() << ": evolved for " << sessionId << Endl;
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TSessionZeroEvolveProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSNumeric>("fraction").SetMin(0).SetMax(1).SetPrecision(3);
    scheme.Add<TFSDuration>("threshold").SetDefault(Threshold);
    return scheme;
}

bool TSessionZeroEvolveProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["fraction"], Fraction) &&
        NJson::ParseField(value["threshold"], Threshold);
}

NJson::TJsonValue TSessionZeroEvolveProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["fraction"] = Fraction;
    result["threshold"] = NJson::ToJson(NJson::Hr(Threshold));
    return result;
}

THashMap<TString, bool> TSessionCloseWithLagTelematicsProcess::GetSensorByIMEIs(const auto& sensorApi, TConstArrayRef<TString> carIMEIs) const {
    NDrive::ISensorApi::TSensorsQueryOptions queryOptions;
    queryOptions.MutableFetchOptions().SetTimeout(WaitForSensorsApi);
    auto asyncSensors = sensorApi->GetSensors(carIMEIs, {CAN_ENGINE_IS_ON}, queryOptions);
    if (!asyncSensors.Wait(WaitForSensorsApi)) {
        ERROR_LOG << GetRobotId() << ": could not wait for sensors 'CAN_ENGINE_IS_ON' request" << Endl;
    }

    THashMap<TString, bool> result;
    const auto& imeisToSensor = asyncSensors.GetValue();
    for (const auto& imei : carIMEIs) {
        if (imeisToSensor.count(imei)) {
            auto sensor = imeisToSensor.at(imei);
            if (sensor.size() != 1) {
                INFO_LOG << GetRobotId() << ": sensor CAN_ENGINE_IS_ON is missing in Sensor API" << Endl;
            } else {
                NDrive::TSensor engineOnSensor = sensor[0];
                if (engineOnSensor.Timestamp > TInstant::Now() - FreshnessCanEngineIsOnSensor) {
                    result[imei] =  NDrive::TSensor::TryConvertTo<bool>(engineOnSensor).GetOrElse(false);
                    continue;
                }
            }
        } else {
            INFO_LOG << GetRobotId() << ": sensor CAN_ENGINE_IS_ON is missing in Sensor API for imei: " << imei  << Endl;
        }
        result[imei] = false;
    }
    return result;
}

TExpectedState TSessionCloseWithLagTelematicsProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& settings = server.GetSettings();
    const auto api = Yensured(server.GetDriveAPI());
    const auto* sensorApi = server.GetSensorApi();
    const auto& deviceTagManager = api->GetTagsManager().GetDeviceTags();

    const auto sessionBuilder = deviceTagManager.GetSessionsBuilder("billing");
    const auto sessions = Yensured(sessionBuilder)->GetSessionsActual();

    ui32 success = 0;
    ui32 exception = 0;
    ui32 error = 0;

    auto engineSessionsOfferName = settings.GetValue<TString>("telematics.engine_session.offer_name").GetOrElse("engine_session");

    TVector<TString> IMEIs;
    for (auto& session : sessions) {
        auto billingSession = dynamic_cast<const TBillingSession*>(session.Get());
        if (!billingSession) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get session " << Endl;
            break;
        }
        TString carId = billingSession->GetObjectId();
        auto carIMEI = api->GetIMEI(carId);
        if (carIMEI) {
            IMEIs.push_back(carIMEI);
        }
    }
    auto sensorByIMEIs = GetSensorByIMEIs(sensorApi, IMEIs);
    R_ENSURE(!sensorByIMEIs.empty(), HTTP_INTERNAL_SERVER_ERROR, "cannot get sensor by imeis");

    for (auto& session : sessions) try {
        auto billingSession = dynamic_cast<const TBillingSession*>(session.Get());
        if (!billingSession) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get session " << Endl;
            error++;
            continue;
        }
        TString carId = billingSession->GetObjectId();
        TString sessionId = session->GetSessionId();

        auto offer = billingSession->GetCurrentOffer();
        if(!offer) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get session offer, carId: " << carId << Endl;
            error++;
            continue;
        }
        if (offer->GetName() != engineSessionsOfferName) {
            NOTICE_LOG << GetRobotId() << ": skipping,  offer name is not " << engineSessionsOfferName << " name is " << offer->GetName() << Endl;
            continue;
        }
        const auto snapshot = server.GetSnapshotsManager().GetSnapshotPtr(carId);
        if (!snapshot) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get snapshot by carId: " << carId << Endl;
            error++;
            continue;
        }

        auto carIMEI = api->GetIMEI(carId);
        if (!carIMEI) {
            ERROR_LOG << GetRobotId() << ": IMEI is missing for carId: " << carId << Endl;
            error++;
            continue;
        }

        if (sensorByIMEIs[carIMEI]) {
            auto heartbeat = snapshot->GetHeartbeat();
            if (heartbeat && heartbeat->Timestamp > TInstant::Now() - FinishThreshold) {
                continue;
            } else {
                auto heartbeatAsync = sensorApi->GetHeartbeat(carIMEI);
                heartbeatAsync.Wait(WaitForSensorsApi);
                Y_ENSURE(heartbeatAsync.HasValue(), "No information received from the SensorApi of the car: " << carIMEI);
                auto heartbeatValue = heartbeatAsync.GetValue();
                if (heartbeatValue->Timestamp > TInstant::Now() - FinishThreshold) {
                    continue;
                }
            }
        } else {
            WARNING_LOG << GetRobotId() << ": sensor CAN_ENGINE_IS_ON for carId: " << carId << " is too old" << Endl;
        }

        const auto& userId = session->GetUserId();
        auto permissions = api->GetUserPermissions(userId);
        R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << userId);
        auto tx = deviceTagManager.BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        tx.SetOriginatorId(GetRobotUserId());
        if (!TChargableTag::DirectEvolve(session, TChargableTag::Reservation, *permissions, server, tx, nullptr)) {
            ERROR_LOG << GetRobotId() << ": cannot DirectEvolve for " << sessionId << ": " << tx.GetStringReport() << Endl;
            error++;
            continue;
        }

        if (DryRun) {
            bool rolledBack = tx.Rollback();
            Y_UNUSED(rolledBack);
            NOTICE_LOG << GetRobotId() << ": dry run for " << sessionId << ": " << tx.GetStringReport() << Endl;
            success++;
            continue;
        }

        tx.Committed().Subscribe([sessionId, carId](const NThreading::TFuture<void>& c) {
            if (c.HasValue()) {
                NDrive::TEventLog::Log("FinishSessionWithTelematicsLag", NJson::TMapBuilder
                    ("session_id", sessionId)
                    ("car_id", carId)
                );
            }
        });
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit for " << sessionId << ": " << tx.GetStringReport() << Endl;
            error++;
            continue;
        }
        INFO_LOG << GetRobotId() << ": evolved to 'Reservation' for " << sessionId << Endl;
        success++;
    } catch (...) {
        ERROR_LOG << GetRobotId() << ": an exception occurred while processing " << Yensured(session)->GetSessionId() << ": " << CurrentExceptionInfo().GetStringRobust() << Endl;
        exception++;
    }
    SuccessClosedSessionsSignal.Signal(success);
    ExceptionClosedSessionsSignal.Signal(exception);
    ErrorClosedSessionsSignal.Signal(error);
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TSessionCloseWithLagTelematicsProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSDuration>("finish_threshold").SetDefault(FinishThreshold);
    scheme.Add<TFSDuration>("freshness_sensor_can_engine_is_on").SetDefault(FreshnessCanEngineIsOnSensor);
    scheme.Add<TFSDuration>("wait_for_heartbeat_respond").SetDefault(WaitForSensorsApi);
    return scheme;
}

bool TSessionCloseWithLagTelematicsProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["finish_threshold"], FinishThreshold) &&
        NJson::ParseField(value["freshness_sensor_can_engine_is_on"], FreshnessCanEngineIsOnSensor) &&
        NJson::ParseField(value["wait_for_heartbeat_respond"], WaitForSensorsApi);
}

NJson::TJsonValue TSessionCloseWithLagTelematicsProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["finish_threshold"] = NJson::ToJson(NJson::Hr(FinishThreshold));
    result["freshness_sensor_can_engine_is_on"] = NJson::ToJson(NJson::Hr(FreshnessCanEngineIsOnSensor));
    result["wait_for_heartbeat_respond"] = NJson::ToJson(NJson::Hr(WaitForSensorsApi));
    return result;
}

TExpectedState TGPSSessionCloseProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto& settings = server.GetSettings();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& deviceTagManager = api->GetTagsManager().GetDeviceTags();

    const auto sessionBuilder = deviceTagManager.GetSessionsBuilder("billing");
    const auto sessionsByObject = Yensured(sessionBuilder)->GetSessionsActualByObjects();

    ui32 success = 0;
    ui32 exception = 0;
    ui32 error = 0;

    auto gpsSessionsOfferName = settings.GetValue<TString>(SettingPath + ".offer_name").GetOrElse("gps_session");
    auto gpsSessionsTagName = settings.GetValue<TString>(SettingPath + ".tag").GetOrElse("enable_gps_sessions");

    TSet<TString> carIds;
    TVector<TTaggedObject> taggedCars;
    {
        auto tx = deviceTagManager.BuildTx<NSQL::ReadOnly | NSQL::RepeatableRead>();
        R_ENSURE(api->GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds(MakeSet<TString>(IterateKeys(sessionsByObject)), taggedCars), {}, "cannot GetObjectsFromCacheByIds", tx);
    }

    for (const auto& taggedCar : taggedCars) {
        if (taggedCar.HasTag(gpsSessionsTagName)) {
            carIds.insert(taggedCar.GetId());
        }
    }

    for (auto& [carId, session] : Yensured(sessionBuilder)->GetSessionsActualByObjects(&carIds)) try {
        auto billingSession = dynamic_cast<const TBillingSession*>(session.Get());
        if (!billingSession) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get session " << Endl;
            error++;
            continue;
        }
        TString sessionId = session->GetSessionId();

        auto offer = billingSession->GetCurrentOffer();
        if(!offer) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get session offer, carId: " << carId << Endl;
            error++;
            continue;
        }
        if (offer->GetName() != gpsSessionsOfferName) {
            NOTICE_LOG << GetRobotId() << ": skipping,  offer name is not " << gpsSessionsOfferName << " name is " << offer->GetName() << Endl;
            continue;
        }
        const auto snapshot = server.GetSnapshotsManager().GetSnapshotPtr(carId);
        if (!snapshot) {
            ERROR_LOG << GetRobotId() << ": skipping when trying to get snapshot by carId: " << carId << Endl;
            error++;
            continue;
        }

        auto heartbeat = snapshot->GetHeartbeat();
        if (heartbeat && heartbeat->Timestamp > StartInstant - FinishThreshold) {
            continue;
        }

        const auto& userId = session->GetUserId();
        auto permissions = api->GetUserPermissions(userId);
        R_ENSURE(permissions, HTTP_INTERNAL_SERVER_ERROR, "cannot GetUserPermissions for " << userId);
        auto tx = deviceTagManager.BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
        {
            tx.SetOriginatorId(GetRobotUserId());
            if (!TChargableTag::DirectEvolve(session, TChargableTag::Reservation, *permissions, server, tx, nullptr)) {
                ERROR_LOG << GetRobotId() << ": cannot DirectEvolve for " << sessionId << ": " << tx.GetStringReport() << Endl;
                error++;
                continue;
            }
        }

        if (DryRun) {
            bool rolledBack = tx.Rollback();
            Y_UNUSED(rolledBack);
            NOTICE_LOG << GetRobotId() << ": dry run for " << sessionId << ": " << tx.GetStringReport() << Endl;
            success++;
            continue;
        }

        tx.Committed().Subscribe([sessionId, carId = carId](const NThreading::TFuture<void>& c) {
            if (c.HasValue()) {
                NDrive::TEventLog::Log("FinishGPSSession", NJson::TMapBuilder
                    ("session_id", sessionId)
                    ("car_id", carId)
                );
            }
        });
        if (!tx.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot Commit for " << sessionId << ": " << tx.GetStringReport() << Endl;
            error++;
            continue;
        }
        INFO_LOG << GetRobotId() << ": evolved to 'Reservation' for " << sessionId << Endl;
        success++;
    } catch (...) {
        ERROR_LOG << GetRobotId() << ": an exception occurred while processing " << Yensured(session)->GetSessionId() << ": " << CurrentExceptionInfo().GetStringRobust() << Endl;
        exception++;
    }
    SuccessClosedSessionsSignal.Signal(success);
    ExceptionClosedSessionsSignal.Signal(exception);
    ErrorClosedSessionsSignal.Signal(error);
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TGPSSessionCloseProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSDuration>("finish_threshold").SetDefault(FinishThreshold);
    scheme.Add<TFSString>("setting_path").SetDefault(SettingPath).SetRequired(true);
    return scheme;
}

bool TGPSSessionCloseProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["finish_threshold"], FinishThreshold) &&
        NJson::ParseField(value["setting_path"], SettingPath);
}

NJson::TJsonValue TGPSSessionCloseProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["finish_threshold"] = NJson::ToJson(NJson::Hr(FinishThreshold));
    result["setting_path"] = SettingPath;
    return result;
}

TSessionCloseProcess::TFactory::TRegistrator<TSessionCloseProcess> TSessionCloseProcess::Registrator(TSessionCloseProcess::GetTypeName());
TSessionZeroEvolveProcess::TFactory::TRegistrator<TSessionZeroEvolveProcess> TSessionZeroEvolveProcess::Registrator(TSessionZeroEvolveProcess::GetTypeName());
TSessionCloseWithLagTelematicsProcess::TFactory::TRegistrator<TSessionCloseWithLagTelematicsProcess> TSessionCloseWithLagTelematicsProcess::Registrator(TSessionCloseWithLagTelematicsProcess::GetTypeName());
TGPSSessionCloseProcess::TFactory::TRegistrator<TGPSSessionCloseProcess> TGPSSessionCloseProcess::Registrator(TGPSSessionCloseProcess::GetTypeName());

