#pragma once

#include <rtline/library/unistat/signals.h>

#include <drive/backend/rt_background/common/config.h>

class TSessionCloseProcess: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "session_close";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TDuration FinishThreshold = TDuration::Minutes(5);
    bool DryRun = true;

private:
    static TFactory::TRegistrator<TSessionCloseProcess> Registrator;
};

class TSessionZeroEvolveProcess: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "session_zero_evolve";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TDuration Threshold = TDuration::Days(1);
    double Fraction = 0;
    bool DryRun = true;

private:
    static TFactory::TRegistrator<TSessionZeroEvolveProcess> Registrator;
};

class TSessionCloseWithLagTelematicsProcess: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

    TUnistatSignal<double> SuccessClosedSessionsSignal;
    TUnistatSignal<double> ExceptionClosedSessionsSignal;
    TUnistatSignal<double> ErrorClosedSessionsSignal;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "session_close_with_lag_telematics";
    }

    TSessionCloseWithLagTelematicsProcess()
        : TBase()
        , SuccessClosedSessionsSignal({"session-close-with-lag-telematics-signal-success"}, false)
        , ExceptionClosedSessionsSignal({"session-close-with-lag-telematics-signal-exception"}, false)
        , ErrorClosedSessionsSignal({"session-close-with-lag-telematics-signal-error"}, false)
    {
    }

    THashMap<TString, bool>  GetSensorByIMEIs(const auto& sensorApi, TConstArrayRef<TString> carIMEI) const;

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TDuration FinishThreshold = TDuration::Minutes(60);
    TDuration FreshnessCanEngineIsOnSensor = TDuration::Hours(1);
    TDuration WaitForSensorsApi = TDuration::Seconds(10);
    bool DryRun = true;

private:
    static TFactory::TRegistrator<TSessionCloseWithLagTelematicsProcess> Registrator;
};

class TGPSSessionCloseProcess: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

    TUnistatSignal<double> SuccessClosedSessionsSignal;
    TUnistatSignal<double> ExceptionClosedSessionsSignal;
    TUnistatSignal<double> ErrorClosedSessionsSignal;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "gps_session_close";
    }

    TGPSSessionCloseProcess()
        : TBase()
        , SuccessClosedSessionsSignal({"gps-session-close-signal-success"}, false)
        , ExceptionClosedSessionsSignal({"gps-session-close-signal-exception"}, false)
        , ErrorClosedSessionsSignal({"gps-session-close-signal-error"}, false)
    {
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TDuration FinishThreshold = TDuration::Minutes(5);
    TString SettingPath = "telematics.gps_session";
    bool DryRun = true;

private:
    static TFactory::TRegistrator<TGPSSessionCloseProcess> Registrator;
};
