#include "process.h"

#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/processors/common_app/fetcher.h>
#include <drive/backend/processors/leasing/session_process.h>

TExpectedState TCloseSignalqSessionsProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& deviceTagsManager = api->GetTagsManager().GetDeviceTags();
    const auto& settings = server.GetSettings();
    const auto& snapshotsManager = server.GetSnapshotsManager();

    const auto enableSignalqSessionsTagName = settings.GetValue<TString>("signalq.signalq_session.tag");
    R_ENSURE(enableSignalqSessionsTagName && *enableSignalqSessionsTagName, {}, "no signalq.signalq_session.tag in gvars");
    const auto enableEngineSessionsTagName = settings.GetValue<TString>("telematics.engine_session.tag");
    R_ENSURE(enableEngineSessionsTagName && *enableEngineSessionsTagName, {}, "no telematics.engine_session.tag in gvars");
    const auto enableTaxiSessionsTagName = settings.GetValue<TString>("taxi.taxi_session.tag");
    R_ENSURE(enableTaxiSessionsTagName && *enableTaxiSessionsTagName, {}, "no taxi.taxi_session.tag in gvars");

    const auto filter = TTagsFilter::BuildFromString(*enableSignalqSessionsTagName);
    const auto fetchedCarIds = deviceTagsManager.PrefilterObjects(filter, nullptr, TInstant::Now());
    R_ENSURE(fetchedCarIds, {}, "failed to get cars with TTagsFilter: " << filter.ToString());
    auto fetchedCarsInfo = api->GetCarsData()->FetchInfo(*fetchedCarIds, TInstant::Now());
    const auto sessionBuilder = deviceTagsManager.GetHistoryManager().GetSessionsBuilder("billing", Now());
    R_ENSURE(sessionBuilder, {}, "cannot get sessionbuilder");

    TVector<TString> carIds;
    carIds.reserve(fetchedCarsInfo.size());
    for (auto&& fetchedCarInfo: fetchedCarsInfo) {
        auto& carId = fetchedCarInfo.second.MutableId();
        const auto cachedObject = deviceTagsManager.GetObject(carId);
        if (cachedObject->HasTag(*enableEngineSessionsTagName) || cachedObject->HasTag(*enableTaxiSessionsTagName)) {
            continue;
        }
        carIds.push_back(std::move(carId));
    }

    for (const auto& carId: carIds) {
        const auto snapshot = snapshotsManager.GetSnapshot(carId);
        const auto lastCarEvent = snapshot.GetLastSignalqEventPtr();
        const auto signalqStatusPtr = snapshot.GetSignalqStatusPtr();
        const auto now = Now();
        const bool signalqStatusAtExpired = !signalqStatusPtr || signalqStatusPtr->GetStatusAt() + SignalqSessionsStatusTimeout < now;
        const bool signalqEventAtExpired = !lastCarEvent || lastCarEvent->GetType() == "shutdown" || lastCarEvent->GetAt() + SignalqSessionsEventTimeout < now;
        if (signalqStatusAtExpired && signalqEventAtExpired) {
            if (!signalqStatusPtr) {
                WARNING_LOG << "carId " << carId << " with signal device has no device status" << Endl;
            }
            if (!lastCarEvent) {
                WARNING_LOG << "carId " << carId << " with signal device has no signals" << Endl;
            }
            const auto session = sessionBuilder->GetLastObjectSession(carId);
            if (session && session->GetCurrentState() == ISession::ECurrentState::Started) {
                INFO_LOG << GetRobotId() << ": last session for carId: " << carId << " is 'Started', sessionId: " << session->GetSessionId() << Endl;
                const auto carUserId = session->GetUserId();
                NDrivematics::CloseSignalqSession(session, carUserId, server, DryRun);
            } else {
                DEBUG_LOG << "Cannot find actual session for carId: " << carId << Endl;
            }
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TCloseSignalqSessionsProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSString>("name_signalq_session_tag").SetDefault(server.GetSettings().template GetValue<TString>("signalq.signalq_session.tag").GetOrElse("Empty GVar signalq.signalq_session.tag")).SetReadOnly(true);
    scheme.Add<TFSString>("name_engine_session_tag").SetDefault(server.GetSettings().template GetValue<TString>("telematics.engine_session.tag").GetOrElse("Empty GVar telematics.engine_session.tag")).SetReadOnly(true);
    scheme.Add<TFSString>("name_taxi_session_tag").SetDefault(server.GetSettings().template GetValue<TString>("taxi.taxi_session.tag").GetOrElse("Empty GVar taxi.taxi_session.tag")).SetReadOnly(true);
    scheme.Add<TFSDuration>("signalq_session_status_timeout").SetDefault(SignalqSessionsStatusTimeout);
    scheme.Add<TFSDuration>("signalq_session_event_timeout").SetDefault(server.GetSettings().template GetValue<TDuration>("signalq.signalq_events.actual_ride_gap"));
    return scheme;
}

bool TCloseSignalqSessionsProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["signalq_session_status_timeout"], SignalqSessionsStatusTimeout) &&
        NJson::ParseField(value["signalq_session_event_timeout"], SignalqSessionsEventTimeout);
}

NJson::TJsonValue TCloseSignalqSessionsProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["signalq_session_status_timeout"] = NJson::ToJson(NJson::Hr(SignalqSessionsStatusTimeout));
    result["signalq_session_event_timeout"] = NJson::ToJson(NJson::Hr(SignalqSessionsEventTimeout));
    return result;
}

TCloseSignalqSessionsProcess::TFactory::TRegistrator<TCloseSignalqSessionsProcess> TCloseSignalqSessionsProcess::Registrator(TCloseSignalqSessionsProcess::GetTypeName());
