#include "process.h"

#include <drive/backend/cars/hardware.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/cppclient/client.h>
#include <drive/backend/cppclient/definitions.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/processors/leasing/get_signals.h>
#include <drive/backend/signalq/signals/tag.h>

#include <drive/library/cpp/auth/tvm.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/client.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/definitions.h>
#include <drive/library/cpp/taxi/signalq_drivematics_api/utils.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTSignalqEventsSupportNotification> TRTSignalqEventsSupportNotification::Registrator(TRTSignalqEventsSupportNotification::GetTypeName());
TRTSignalqEventsSupportNotificationState::TFactory::TRegistrator<TRTSignalqEventsSupportNotificationState> TRTSignalqEventsSupportNotificationState::Registrator(TRTSignalqEventsSupportNotification::GetTypeName());

TString TRTSignalqEventsSupportNotificationState::GetType() const {
    return TRTSignalqEventsSupportNotification::GetTypeName();
}

namespace {

    using TSerialNumberEventIdToEventPresignedUrlsMap = NDrive::NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse::TSerialNumberEventIdToEventPresignedUrlsMap;
    using TSerialNumberUnixTimestampToDataMap = NDrive::NSignalq::TV1EventRaiseUrlsGenerateResponse::TSerialNumberUnixTimestampToDataMap;

    class TRequestDataItem {
        R_FIELD(TString, SerialNumber);
        R_FIELD(TString, EventId);
        R_FIELD(TString, Vin);
        R_FIELD(NDrive::NSignalq::TApiSignalqNotifySupportRequestEvent, Event);
    };

    TVector<TString> GetWhitelistSignalqSignalsTagNames(const ISettings& settings, const TSet<TString>& whitelistEventTypes) {
        const auto optionalSignalqSignalsNamesString = settings.GetValue<TString>("signalq.signalq_events.signals_description");
        R_ENSURE(optionalSignalqSignalsNamesString && *optionalSignalqSignalsNamesString, {}, "no signalq.signalq_events.signals_description in gvars");
        TVector<NJson::TJsonValue> signalqSignalsDescriptions;
        NJson::TJsonValue signalqSignalsNamesJson;
        R_ENSURE(
            NJson::ReadJsonFastTree(*optionalSignalqSignalsNamesString, &signalqSignalsNamesJson),
            HTTP_INTERNAL_SERVER_ERROR,
            "can't parse signalq signals names"
        );
        R_ENSURE(
            NJson::ParseField(signalqSignalsNamesJson["descriptions"], signalqSignalsDescriptions),
            HTTP_INTERNAL_SERVER_ERROR,
            "can't convert json into TVector<NJson::TJsonValue>"
        );
        TVector<TString> result;
        result.reserve(whitelistEventTypes.size());
        for (const auto& signalqSignalDesciption: signalqSignalsDescriptions) {
            TString signalqEventType;
            R_ENSURE(
                NJson::ParseField(signalqSignalDesciption["type"], signalqEventType),
                HTTP_INTERNAL_SERVER_ERROR,
                "can't parse type"
            );
            if (whitelistEventTypes.contains(signalqEventType)) {
                TString tagName;
                R_ENSURE(
                    NJson::ParseField(signalqSignalDesciption["name"], tagName),
                    HTTP_INTERNAL_SERVER_ERROR,
                    "can't parse signalq signal tag name"
                );
                result.push_back(std::move(tagName));
            }
        }
        return result;
    }

    bool CheckSignalqEventHasRequiredS3Paths(const NDrive::NSignalq::TEvent& signalqEvent, TMap<TString, TSet<TString>>& eventTypesToMediaTypes) {
        const auto& requiredMediaTypes = eventTypesToMediaTypes[signalqEvent.GetType()];
        for (const auto& mediaType: requiredMediaTypes) {
            if (mediaType == "photo" && !signalqEvent.HasS3PhotoPath()) {
                return false;
            }
            if (mediaType == "external_photo" && !signalqEvent.HasS3ExternalPhotoPath()) {
                return false;
            }
            if (mediaType == "video" && !signalqEvent.HasS3VideoPath()) {
                return false;
            }
            if (mediaType == "external_video" && !signalqEvent.HasS3ExternalVideoPath()) {
                return false;
            }
        }
        return true;
    }

    void ResheduleNotUploadedEvents(const TOptionalTagHistoryEvents& optionalEvents, const TSet<TString>& eventsToResheduleIds, const TString& userId, const TTraceTagsManager& traceTagsManager, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        TVector<TDBTag> tagsToUpdate;
        for (const auto& event: *optionalEvents) {
            const auto signalqTagPtr = event.GetTagAs<TSignalqEventTraceTag>();
            if (signalqTagPtr && eventsToResheduleIds.contains(signalqTagPtr->GetEvent().GetId())) {
                TDBTag tagToUpdate = event.Clone(server.GetDriveAPI()->GetTagsHistoryContext());
                tagsToUpdate.push_back(std::move(tagToUpdate));
            }
        }
        const auto tagsToUpdateSize = tagsToUpdate.size();
        if (traceTagsManager.UpdateTagsData(std::move(tagsToUpdate), userId, tx)) {
            INFO_LOG << "Successfully updated " << tagsToUpdateSize << " tags" << Endl;
        } else {
            ERROR_LOG << "Failed to update " << tagsToUpdateSize << " tags" << Endl;
        }
    }

}

TExpectedState TRTSignalqEventsSupportNotification::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    auto state = dynamic_cast<const TRTSignalqEventsSupportNotificationState*>(stateExt.Get());
    auto newState = MakeHolder<TRTSignalqEventsSupportNotificationState>();
    const auto api = Yensured(server.GetDriveAPI());
    const auto& traceTagsManager = api->GetTagsManager().GetTraceTags();
    const auto& deviceTagsManager = api->GetTagsManager().GetDeviceTags();
    THistoryRidesContext sessionsContext(server);
    const auto& settings = server.GetSettings();
    const auto signalqDrivematicsApiClient = server.GetTaxiSignalqDrivematicsApiClient();
    R_ENSURE(signalqDrivematicsApiClient, {}, "No signalq-drivematics-api client");
    const auto carsharingBackendClient = NDrive::TCarsharingBackendClient(CarsharingBackendUri, new NDrive::TTvmAuth(server.GetTvmClient(DriveTvmId), DriveTvmId));

    TSet<TString> whitelistEventTypes = MakeSet(NContainer::Keys(EventTypesToMediaTypes));
    auto tx = traceTagsManager.BuildTx<NSQL::Writable>();
    TTagEventsManager::TQueryOptions traceTagsHistoryQuery;
    const ui64 historyIdCursor = state ? state->GetLastEventId() : 0;
    newState->SetLastEventId(historyIdCursor);
    traceTagsHistoryQuery
        .SetTags(GetWhitelistSignalqSignalsTagNames(settings, whitelistEventTypes))
        .SetActions({ EObjectHistoryAction::Add, EObjectHistoryAction::UpdateData })
        .SetOrderBy({"history_event_id"})
        .SetLimit(Limit);
    const auto now = Now();
    const auto optionalEvents = traceTagsManager.GetEvents(historyIdCursor + 1,{ now - MaxEventAtDelay, now }, tx, traceTagsHistoryQuery);
    R_ENSURE(optionalEvents, {}, "can't get events", tx);
    INFO_LOG << "Fetched " << optionalEvents->size() << " events with history_event_ids: " << Endl;
    if (optionalEvents->empty()) {
        return newState;
    }

    std::multimap<TString, const TSignalqEventTraceTag*> sessionIdsToSignalqTagPtrs;
    for (const auto& event: *optionalEvents) {
        INFO_LOG << event.GetHistoryEventId() << Endl;
        const auto signalqTagPtr = event.GetTagAs<TSignalqEventTraceTag>();
        if (!signalqTagPtr) {
            WARNING_LOG << "Tag " << event->GetName() << " is not SignalqEventTraceTag" << Endl;
            continue;
        }
        const auto& signalqEvent = signalqTagPtr->GetEvent();
        const bool isEventExpired = signalqTagPtr->GetEvent().GetAt() + MaxEventAtDelay < Now();
        if (!isEventExpired && CheckSignalqEventHasRequiredS3Paths(signalqEvent, EventTypesToMediaTypes)) {
            sessionIdsToSignalqTagPtrs.emplace(event.GetObjectId(), signalqTagPtr);
        }
    }

    NDrive::NSignalq::TV1EventsMediaPresignedUrlsGenerateRequest eventMediaPresignedUrlsGenerateRequestBody;
    const auto linksExpiresAt = Now() + MediaUrlTtl;
    eventMediaPresignedUrlsGenerateRequestBody.SetLinksExpiresAt(linksExpiresAt);
    eventMediaPresignedUrlsGenerateRequestBody.SetOnlyUploaded(NotifyAboutOnlyFullUploadedEvents);
    TVector<NDrive::NSignalq::TV1EventWithMedia> eventMediaPresignedUrlsParameteres;
    NDrive::NSignalq::TV1EventRaiseUrlsGenerateRequest eventRaiseUrlsGenerateRequestBody;
    TVector<NDrive::NSignalq::TUrlParameteresItem> eventRaiseUrlsParameteres;
    TVector<TRequestDataItem> requestDataItems;
    auto ydbTx = api->BuildYdbTx<NSQL::ReadOnly>("rtsignalq_events_support_notification", &server);
    R_ENSURE(sessionsContext.InitializeSessions(MakeVector(NContainer::Keys(sessionIdsToSignalqTagPtrs)), tx, ydbTx), HTTP_INTERNAL_SERVER_ERROR, "cannot initialize HistoryRidesContext");
    const auto fetchedSessions = sessionsContext.GetSessions(Now(), sessionIdsToSignalqTagPtrs.size());
    INFO_LOG << "Fetched " << fetchedSessions.size() << " sessions" << Endl;
    for (const auto& [sessionId, signalqTagPtr]: sessionIdsToSignalqTagPtrs) {
        const auto& sessionIdRef = sessionId; // need to avoid compiler's bug: can't capture structured binding variable
        const auto sessionPtr = std::find_if(fetchedSessions.begin(), fetchedSessions.end(), [&sessionIdRef](const THistoryRideObject& elem) {
            return elem.GetSessionId() == sessionIdRef;
        });
        if (sessionPtr == fetchedSessions.end()) {
            ERROR_LOG << "Failed to fetch session with id " << sessionId << Endl;
            continue;
        }

        const auto& carId = sessionPtr->GetObjectId();
        const auto cachedObject = deviceTagsManager.GetObject(carId);
        if (!cachedObject) {
            ERROR_LOG << "cannot get cachedObject for carId " << carId << Endl;
            continue;
        }

        bool carHasOneOfRequiredTags = false;
        for (const auto& tagName: CarTagNames) {
            if (cachedObject->HasTag(tagName)) {
                carHasOneOfRequiredTags = true;
                break;
            }
        }
        if (!carHasOneOfRequiredTags) {
            continue;
        }

        TCarGenericAttachment attachment;
        if (!server.GetDriveAPI()->GetCarAttachmentAssignments().TryGetAttachmentOfType(carId, EDocumentAttachmentType::CarRegistryDocument, attachment)) {
            WARNING_LOG << "CarId " << carId << " has no CarRegistryDocument attached" << Endl;
            continue;
        }
        const auto carRegistryDocumentPtr = dynamic_cast<const TCarRegistryDocument*>(attachment.Get());
        if (!carRegistryDocumentPtr) {
            ERROR_LOG << "Cannot cast carId " << carId << "'s attachment " << attachment.GetId() << " to TCarRegistryDocument" << Endl;
            continue;
        }
        auto vin = StripString(carRegistryDocumentPtr->GetVin());
        if (vin.Empty()) {
            NOTICE_LOG << "Car " << carId << " has empty VIN" << Endl;
            continue;
        }

        const auto& signalqEvent = signalqTagPtr->GetEvent();
        const auto eventAt = signalqEvent.GetAt();
        const auto& serialNumber = signalqTagPtr->GetSerialNumber();
        NDrive::NSignalq::TUrlParameteresItem urlParameteresItem;
        urlParameteresItem.SetSerialNumber(serialNumber);
        urlParameteresItem.SetUnixTimestamp(eventAt.Seconds());
        eventRaiseUrlsParameteres.push_back(std::move(urlParameteresItem));

        TRequestDataItem requestDataItem;
        const auto& eventId = signalqEvent.GetId();
        const auto& eventType = signalqEvent.GetType();
        requestDataItem.SetVin(std::move(vin));
        requestDataItem.SetEventId(eventId);
        requestDataItem.SetSerialNumber(serialNumber);

        NDrive::NSignalq::TApiSignalqNotifySupportRequestEvent apiSignalqNotifySupportRequestEvent;
        apiSignalqNotifySupportRequestEvent.SetAt(eventAt.Seconds());
        apiSignalqNotifySupportRequestEvent.SetType(eventType);
        requestDataItem.SetEvent(std::move(apiSignalqNotifySupportRequestEvent));

        requestDataItems.push_back(std::move(requestDataItem));

        NDrive::NSignalq::TV1EventWithMedia v1EventWithMedia;
        v1EventWithMedia.SetSerialNumberEventId(NDrive::NSignalq::MakeSerialNumberEventId(serialNumber, eventId));
        if (EventTypesToMediaTypes[eventType].contains("photo")) {
            v1EventWithMedia.SetS3PhotoPath(signalqEvent.GetS3PhotoPathRef());
        }
        if (EventTypesToMediaTypes[eventType].contains("external_photo")) {
            v1EventWithMedia.SetS3ExternalPhotoPath(signalqEvent.GetS3ExternalPhotoPathRef());
        }
        if (EventTypesToMediaTypes[eventType].contains("video")) {
            v1EventWithMedia.SetS3VideoPath(signalqEvent.GetS3VideoPathRef());
        }
        if (EventTypesToMediaTypes[eventType].contains("external_video")) {
            v1EventWithMedia.SetS3ExternalVideoPath(signalqEvent.GetS3ExternalVideoPathRef());
        }
        eventMediaPresignedUrlsParameteres.push_back(std::move(v1EventWithMedia));
    }
    if (eventMediaPresignedUrlsParameteres.empty()) {
        newState->SetLastEventId(optionalEvents->back().GetHistoryEventId());
        return newState;
    }
    eventMediaPresignedUrlsGenerateRequestBody.SetEvents(std::move(eventMediaPresignedUrlsParameteres));
    eventRaiseUrlsGenerateRequestBody.SetUrlParameteres(std::move(eventRaiseUrlsParameteres));

    auto signalqPresignedUrlsFuture = NThreading::MakeFuture(TSerialNumberEventIdToEventPresignedUrlsMap());
    auto signalqEventRaiseUrlsFuture = NThreading::MakeFuture(TSerialNumberUnixTimestampToDataMap());

    auto requestEventsCount = eventMediaPresignedUrlsGenerateRequestBody.GetEvents().size();
    INFO_LOG << "Requesting media urls for " << requestEventsCount << " events" << Endl;
    auto start = Now();
    const auto signalqPresignedUrlsResponse = signalqDrivematicsApiClient->GetEventsMediaPresignedUrls(eventMediaPresignedUrlsGenerateRequestBody);
    signalqPresignedUrlsFuture = signalqPresignedUrlsResponse.Apply([start, requestEventsCount, linksExpiresAt](const NThreading::TFuture<NSignalq::TV1EventsMediaPresignedUrlsGenerateResponse>& response) {
        if (response.HasValue()) {
            const auto finish = Now();
            const auto duration = finish - start;
            INFO_LOG << "Fetched signalq events presigned urls, duration: " << duration << Endl;
            const auto eventsCount = response.GetValue().GetEvents().size();
            if (eventsCount != requestEventsCount) {
                ERROR_LOG << "Fetched signalq events in count: " << eventsCount << " differs than expected: " << requestEventsCount << Endl;
            } else {
                INFO_LOG << "Fetched events with presigned urls in count: " << eventsCount << Endl;
            }
            return response.GetValue().BuildSerialNumberEventIdToEventPresignedUrlsMap(linksExpiresAt);
        } else {
            ythrow yexception() << "Cant fetch signalq presigned urls: no response, exception: " << NThreading::GetExceptionMessage(response) << Endl;
        }
    });

    requestEventsCount = eventRaiseUrlsGenerateRequestBody.GetUrlParameteres().size();
    INFO_LOG << "Requesting event raise urls for " << requestEventsCount << " events" << Endl;
    start = Now();
    const auto signalqEventRaiseUrlsResponse = signalqDrivematicsApiClient->GetSignalqEventRaiseUrls(eventRaiseUrlsGenerateRequestBody);
    signalqEventRaiseUrlsFuture = signalqEventRaiseUrlsResponse.Apply([start, requestEventsCount](const NThreading::TFuture<NSignalq::TV1EventRaiseUrlsGenerateResponse>& response) {
        if (response.HasValue()) {
            const auto finish = Now();
            const auto duration = finish - start;
            INFO_LOG << "Fetched signalq event raise urls, duration: " << duration << Endl;
            const auto eventsCount = response.GetValue().GetItems().size();
            if (eventsCount != requestEventsCount) {
                ERROR_LOG << "Fetched signalq event raise urls in count: " << eventsCount << " differs than expected: " << requestEventsCount << Endl;
            } else {
                INFO_LOG << "Fetched signalq event raise urls in count: " << eventsCount << Endl;
            }
            return response.GetValue().BuildSerialNumberUnixTimestampToDataMap();
        } else {
            ythrow yexception() << "Cant fetch signalq event raise urls: no response, exception: " << NThreading::GetExceptionMessage(response) << Endl;
        }
    });
    signalqPresignedUrlsFuture.Wait();
    R_ENSURE(signalqPresignedUrlsFuture.HasValue(), {}, "can't get presigned urls", tx);
    auto signalqPresignedUrlsMap = signalqPresignedUrlsFuture.ExtractValue();

    signalqEventRaiseUrlsFuture.Wait();
    R_ENSURE(signalqEventRaiseUrlsFuture.HasValue(), {}, "can't get event raise urls", tx);
    auto signalqEventRaiseUrlsMap = signalqEventRaiseUrlsFuture.ExtractValue();

    NDrive::NSignalq::TApiSignalqNotifySupportRequest notifySupportRequestBody;
    TVector<NDrive::NSignalq::TApiSignalqNotifySupportRequestItem> notifySupportParameteres;
    TSet<TString> eventsToResheduleIds;
    for (auto& requestDataItem: requestDataItems) {
        NDrive::NSignalq::TApiSignalqNotifySupportRequestItem notifySupportParameteresItem;
        notifySupportParameteresItem.SetVin(std::move(requestDataItem.MutableVin()));
        auto& event = requestDataItem.MutableEvent();
        const auto& eventType = event.GetType();
        auto& presignedUrls = signalqPresignedUrlsMap[NDrive::NSignalq::MakeSerialNumberEventId(requestDataItem.GetSerialNumber(), requestDataItem.GetEventId())];
        const auto& requiredMediaTypes = EventTypesToMediaTypes[eventType];
        bool isNotUploadedEvent = false;
        for (const auto& mediaType: requiredMediaTypes) {
            if (mediaType == "photo") {
                if (!presignedUrls.HasS3PhotoPresignedUrl()) {
                    isNotUploadedEvent = true;
                    break;
                }
                event.SetPhotoUrl(presignedUrls.GetS3PhotoPresignedUrlRef());
            } else if (mediaType == "external_photo") {
                if (!presignedUrls.HasS3ExternalPhotoPresignedUrl()) {
                    isNotUploadedEvent = true;
                    break;
                }
                event.SetExternalPhotoUrl(presignedUrls.GetS3ExternalPhotoPresignedUrlRef());
            } else if (mediaType == "video") {
                if (!presignedUrls.HasS3VideoPresignedUrl()) {
                    isNotUploadedEvent = true;
                    break;
                }
                event.SetVideoUrl(presignedUrls.GetS3VideoPresignedUrlRef());
            } else if (mediaType == "external_video") {
                if (!presignedUrls.HasS3ExternalVideoPresignedUrl()) {
                    isNotUploadedEvent = true;
                    break;
                }
                event.SetExternalVideoUrl(presignedUrls.GetS3ExternalVideoPresignedUrlRef());
            }
        }
        if (isNotUploadedEvent) {
            eventsToResheduleIds.insert(std::move(requestDataItem.MutableEventId()));
            continue;
        }
        auto& eventRaiseUrl = signalqEventRaiseUrlsMap[NDrive::NSignalq::MakeSerialNumberUnixTimestamp(requestDataItem.GetSerialNumber(), requestDataItem.GetEvent().GetAt())];
        event.SetAlarmRaiseUrl(eventRaiseUrl);
        notifySupportParameteresItem.SetEvent(std::move(event));
        notifySupportParameteres.push_back(std::move(notifySupportParameteresItem));
    }

    INFO_LOG << "Notifying support about " << notifySupportParameteres.size() << " signals" << Endl;
    if (DryRun) {
        INFO_LOG << "Dry run. Returning" << Endl;
        newState->SetLastEventId(optionalEvents->back().GetHistoryEventId());
        return newState;
    }
    auto notifySupportFuture = NThreading::MakeFuture(ui32());
    if (!notifySupportParameteres.empty()) {
        notifySupportRequestBody.SetItems(std::move(notifySupportParameteres));
        start = Now();
        const auto notifySupportResponse = carsharingBackendClient.NotifySupport(notifySupportRequestBody);
        auto notifySupportFuture = notifySupportResponse.Apply([start](const NThreading::TFuture<NSignalq::TApiSignalqNotifySupportResponse>& response) {
            if (response.HasValue()) {
                const auto finish = Now();
                const auto duration = finish - start;
                INFO_LOG << "Send events to support, duration: " << duration << Endl;
                const auto successfullySentSignalsAmount = response.GetValue().GetAddedTagsAmount();
                INFO_LOG << "Successfully notified support about " << successfullySentSignalsAmount << " signals" << Endl;
                return successfullySentSignalsAmount;
            } else {
                ythrow yexception() << "Cant notify support: no response, exception: " << NThreading::GetExceptionMessage(response) << Endl;
            }
        });
    }
    ResheduleNotUploadedEvents(optionalEvents, eventsToResheduleIds, GetRobotUserId(), traceTagsManager, server, tx);
    if (notifySupportFuture.Initialized()) {
        notifySupportFuture.Wait();
    }
    if (!tx.Commit()) {
        return MakeUnexpected<TString>(GetRobotId() + ": Commit fails. " + tx.GetStringReport());
    }
    newState->SetLastEventId(optionalEvents->back().GetHistoryEventId());
    return newState;
}

NDrive::TScheme TRTSignalqEventsSupportNotification::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run").SetDefault(DryRun);
    scheme.Add<TFSNumeric>("limit").SetDefault(Limit);
    scheme.Add<TFSDuration>("max_event_at_delay").SetDefault(MaxEventAtDelay);
    scheme.Add<TFSJson>("event_types_to_media_types", "Отображение типов сигналов, о которых оповещаем саппортов, в типы медиа");
    scheme.Add<TFSArray>("car_tag_names", "Оповещаем, если на машине есть хотя бы один тег из списка").SetElement<TFSString>();
    scheme.Add<TFSDuration>("media_url_ttl", "Время жизни ссылок на фото и видео").SetDefault(MediaUrlTtl);
    scheme.Add<TFSBoolean>("notify_about_only_full_uploaded_events", "Оповещаем, только если у сигнала загрузились все необходимые фото и видео").SetDefault(NotifyAboutOnlyFullUploadedEvents);
    scheme.Add<TFSString>("carsharing_backend_uri").SetDefault(CarsharingBackendUri);
    scheme.Add<TFSNumeric>("drive_tvm_id").SetDefault(DriveTvmId);
    return scheme;
}

bool TRTSignalqEventsSupportNotification::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }

    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["limit"], Limit) &&
        NJson::ParseField(value["max_event_at_delay"], MaxEventAtDelay) &&
        NJson::ParseField(value["event_types_to_media_types"], NJson::Dictionary(EventTypesToMediaTypes)) &&
        NJson::ParseField(value["car_tag_names"], CarTagNames) &&
        NJson::ParseField(value["media_url_ttl"], MediaUrlTtl) &&
        NJson::ParseField(value["notify_about_only_full_uploaded_events"], NotifyAboutOnlyFullUploadedEvents) &&
        NJson::ParseField(value["carsharing_backend_uri"], CarsharingBackendUri) &&
        NJson::ParseField(value["drive_tvm_id"], DriveTvmId);
}

NJson::TJsonValue TRTSignalqEventsSupportNotification::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["limit"] = Limit;
    result["max_event_at_delay"] = NJson::ToJson(NJson::Hr(MaxEventAtDelay));
    result["media_url_ttl"] = NJson::ToJson(NJson::Hr(MediaUrlTtl));
    result["notify_about_only_full_uploaded_events"] = NotifyAboutOnlyFullUploadedEvents;
    result["event_types_to_media_types"] = NJson::ToJson(NJson::Dictionary(EventTypesToMediaTypes));
    result["carsharing_backend_uri"] = CarsharingBackendUri;
    result["drive_tvm_id"] = DriveTvmId;

    NJson::TJsonValue tagNamesJson = NJson::JSON_ARRAY;
    for (auto&& tagName : CarTagNames) {
        tagNamesJson.AppendValue(tagName);
    }
    result["car_tag_names"] = std::move(tagNamesJson);

    return result;
}
