#include "config.h"

#include <drive/backend/database/drive_api.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

TCarSLACheckerConfig::TFactory::TRegistrator<TCarSLACheckerConfig> TCarSLACheckerConfig::Registrator(TCarSLACheckerConfig::GetTypeName());
TUserSLACheckerConfig::TFactory::TRegistrator<TUserSLACheckerConfig> TUserSLACheckerConfig::Registrator(TUserSLACheckerConfig::GetTypeName());

static const TString RemoveTagConstant = "$remove$";
static const TString NativeActionConstant = "$native$";

template <class TBaseRegularBackgroundProcess>
TExpectedState ISLACheckerConfig<TBaseRegularBackgroundProcess>::Process(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const IRTBackgroundProcess::TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    TVector<TDBTag> tags;
    const IEntityTagsManager& manager = GetEntityTagsManager(server);
    if (!manager.RestoreTags(GetFiltredObjects(context), GetTagNames(), tags, session)) {
        return MakeUnexpected<TString>("cannot RestoreTags: " + session.GetStringReport());
    }
    TSet<TString> objects;
    const TInstant now = Now();

    TMap<TString, TVector<TDBTag>> problemsByType;

    for (auto&& i : tags) {
        if (i->HasSLAInstant() && i->GetSLAInstantUnsafe() < now) {
            problemsByType[i->GetName()].emplace_back(i);
            objects.emplace(i.GetObjectId());
        }
    }

    NDrive::INotifier::TPtr notifier = server.GetNotifier(GetNotifierName());
    if (problemsByType.empty()) {
        if (notifier && GetNotifyEmpty()) {
            notifier->Notify(NDrive::INotifier::TMessage("Все теги типа " + JoinSeq(",", GetTagNames()) + " выполняются оперативно (в пределах указанного в теге интервала)"));
        }
        return new IRTBackgroundProcessState();
    }

    auto robotUserPermissions = server.GetDriveAPI()->GetUserPermissions(TBaseRegularBackgroundProcess::GetRobotUserId(), {});
    for (auto&& tagType : problemsByType) {
        TSet<TString> evolved;
        TString evolvedName;
        const auto doEvolve = [this, &server, &manager, &robotUserPermissions, &evolved, &evolvedName](const TString& tagName, const TVector<TDBTag>& tags)->TString {
            if (!robotUserPermissions) {
                return "incorrect user_id: " + TBaseRegularBackgroundProcess::GetRobotUserId();
            }

            auto td = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
            if (!td) {
                return "Нет метаданных для тега " + tagName;
            }
            if (!td->HasSLAEvolution()) {
                return "";
            }
            evolvedName = td->GetSLAEvolutionUnsafe();
            auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (evolvedName == NativeActionConstant) {
                for (auto&& i : tags) {
                    if (!i->OnSLAExpired(i, *robotUserPermissions, &manager, &server, session)) {
                        return session.GetStringReport();
                    }
                    evolved.emplace(i.GetTagId());
                }
            } else if (evolvedName != RemoveTagConstant) {
                auto tdNew = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(td->GetSLAEvolutionUnsafe());
                if (!tdNew) {
                    return "Нет метаданных для тега эволюции " + td->GetSLAEvolutionUnsafe();
                }
                ITag::TPtr tag = ITag::TFactory::Construct(tdNew->GetType());
                if (!tag) {
                    return "Некорректный тип тега эволюции " + td->GetSLAEvolutionUnsafe() + " / " + tdNew->GetType();
                }
                tag->SetName(evolvedName);
                if (tdNew->HasSLADuration()) {
                    tag->SetSLAInstant(Now() + tdNew->GetSLADurationUnsafe());
                }
                if (td->HasSLAEvolutionPriority()) {
                    tag->SetTagPriority(td->GetSLAEvolutionPriorityUnsafe());
                }
                for (auto&& i : tags) {
                    if (!GetModifyPerformed() && !!i->GetPerformer()) {
                        continue;
                    }
                    if (!manager.EvolveTag(i, tag, *robotUserPermissions, &server, session)) {
                        return session.GetStringReport();
                    }
                    evolved.emplace(i.GetTagId());
                }
            } else {
                for (auto&& i : tags) {
                    if (!manager.RemoveTag(i, TBaseRegularBackgroundProcess::GetRobotUserId(), &server, session, GetModifyPerformed())) {
                        return session.GetStringReport();
                    }
                    evolved.emplace(i.GetTagId());
                }
            }
            if (!GetDryRun() && !session.Commit()) {
                return session.GetStringReport();
            }
            return "";
        };
        const TString failOnAdd = doEvolve(tagType.first, tagType.second);

        TString header = "У тегов типа " + tagType.first + " обнаружены просроченные " + ::ToString(tagType.second.size()) +" объектов: \n";
        if (failOnAdd) {
            header += " " + failOnAdd;
            NDrive::INotifier::Notify(notifier, header);
            continue;
        }
        if (evolvedName) {
            if (evolvedName == RemoveTagConstant) {
                header += "удаление\n";
            } else if (evolvedName == NativeActionConstant) {
                header += "нативное действие\n";
            } else {
                header += "эволюция в " + evolvedName + "\n";
            }
        }
        TVector<TString> overdueTags;
        const auto pred = [](const TDBTag& tagLeft, const TDBTag& tagRight) -> bool {
            return tagLeft->GetSLAInstantUnsafe() < tagRight->GetSLAInstantUnsafe();
        };
        std::sort(tagType.second.begin(), tagType.second.end(), pred);
        for (auto&& i : tagType.second) {
            overdueTags.emplace_back(GetObjectInfo(i.GetObjectId(), server));
            overdueTags.back() += " (" + server.GetLocalization()->FormatDuration(ELocalization::Rus, now - i->GetSLAInstantUnsafe()) + ")";
            if (evolvedName) {
                if (evolvedName == RemoveTagConstant) {
                    overdueTags.back() += " удален";
                } else if (evolvedName == NativeActionConstant) {
                    header += " вызван OnSLAExpired";
                } else {
                    if (!evolved.contains(i.GetTagId())) {
                        overdueTags.back() += " не модифицирован";
                    } else {
                        overdueTags.back() += " модифицирован";
                    }
                }
            }
        }
        NDrive::INotifier::MultiLinesNotify(notifier, header, overdueTags);
    }

    for (auto&& i : problemsByType) {
        TUnistatSignalsCache::SignalAddX("frontend-sla-failed-tags", i.first, 1);
    }

    return new IRTBackgroundProcessState();
}
template<class TBaseRegularBackgroundProcess>
NDrive::TScheme ISLACheckerConfig<TBaseRegularBackgroundProcess>::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("tag_names", "Теги");
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("notify_empty", "Нотифицировать об отсутствии изменений").SetDefault(false);
    scheme.Add<TFSBoolean>("dry_run", "Только нотификации").SetDefault(false);
    scheme.Add<TFSBoolean>("modify_performed", "Модифицировать исполняемые").SetDefault(false);
    return scheme;
}

template<class TBaseRegularBackgroundProcess>
bool ISLACheckerConfig<TBaseRegularBackgroundProcess>::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    if (!NotifierName) {
        return false;
    }

    TString tagNames;
    JREAD_STRING(jsonInfo, "tag_names", tagNames);
    JREAD_BOOL_OPT(jsonInfo, "notify_empty", NotifyEmpty);
    JREAD_BOOL_OPT(jsonInfo, "dry_run", DryRun);
    JREAD_BOOL_OPT(jsonInfo, "modify_performed", ModifyPerformed);

    StringSplitter(tagNames).SplitBySet(", ").SkipEmpty().Collect(&TagNames);
    return true;
}

template<class TBaseRegularBackgroundProcess>
NJson::TJsonValue ISLACheckerConfig<TBaseRegularBackgroundProcess>::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "notifier", NotifierName);
    JWRITE(result, "notify_empty", NotifyEmpty);
    JWRITE(result, "dry_run", DryRun);
    JWRITE(result, "modify_performed", ModifyPerformed);
    JWRITE(result, "tag_names", JoinSeq(",", TagNames));
    return result;
}

template class ISLACheckerConfig<IRTCarsProcess>;
template class ISLACheckerConfig<IRTRegularBackgroundProcess>;
