#pragma once
#include <drive/backend/rt_background/car_counter/config.h>

#include <rtline/util/types/accessor.h>

template<class TBaseRegularBackgroundProcess>
class ISLACheckerConfig: public TBaseRegularBackgroundProcess {
private:
    using TBase = TBaseRegularBackgroundProcess;;
    R_READONLY(TVector<TString>, TagNames);
    R_READONLY(TString, NotifierName);
    R_READONLY(bool, NotifyEmpty, false);
    R_READONLY(bool, DryRun, false);
    R_READONLY(bool, ModifyPerformed, false);

public:
    using TBase::TBase;
    TExpectedState Process(TAtomicSharedPtr<IRTBackgroundProcessState> state, const IRTBackgroundProcess::TExecutionContext& context) const;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    virtual const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const = 0;
    virtual const TSet<TString>& GetFiltredObjects(const IRTBackgroundProcess::TExecutionContext& context) const = 0;
    virtual TString GetObjectInfo(const TString& objectId, const NDrive::IServer& server) const = 0;
};

class TCarSLACheckerConfig : public ISLACheckerConfig<IRTCarsProcess> {
    static TFactory::TRegistrator<TCarSLACheckerConfig> Registrator;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "sla_watcher";
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& /*server*/, TTagsModificationContext& context) const override {
        return Process(state, context);
    }

private:
    virtual const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const override {
        return server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    }

    virtual const TSet<TString>& GetFiltredObjects(const IRTBackgroundProcess::TExecutionContext& context) const override {
        auto carContext = dynamic_cast<const TCarsProcessContext*>(&context);
        if (carContext) {
            return carContext->GetFilteredCarIds();
        }
        return Default<TSet<TString>>();
    }

    virtual TString GetObjectInfo(const TString& objectId, const NDrive::IServer& server) const override {
        auto fetchResult = server.GetDriveAPI()->GetCarsData()->GetObject(objectId);
        if (fetchResult) {
            return fetchResult->GetHRReport();
        }
        return objectId;
    }
};

class TUserSLACheckerConfig : public ISLACheckerConfig<IRTRegularBackgroundProcess> {
    static TFactory::TRegistrator<TUserSLACheckerConfig> Registrator;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "user_sla_watcher";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const IRTBackgroundProcess::TExecutionContext& context) const override {
        return Process(state, context);
    }

private:
    virtual const IEntityTagsManager& GetEntityTagsManager(const NDrive::IServer& server) const override {
        return server.GetDriveAPI()->GetTagsManager().GetUserTags();
    }

    virtual const TSet<TString>& GetFiltredObjects(const IRTBackgroundProcess::TExecutionContext& /*context*/) const override {
        return Default<TSet<TString>>();
    }

    virtual TString GetObjectInfo(const TString& objectId, const NDrive::IServer& server) const override {
        auto fetchResult = server.GetDriveAPI()->GetUsersData()->GetCachedObject(objectId);
        if (fetchResult) {
            return fetchResult->GetHRReport();
        }
        return objectId;
    }
};
