#include "process.h"

#include <drive/library/cpp/yt/node/cast.h>

#include <mapreduce/yt/interface/client.h>

#include <rtline/library/json/parse.h>

TExpectedState TSql2UnistatProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    TDatabasePtr database;
    if (!DBName) {
        if (!server.GetDriveAPI()) {
            return MakeUnexpected<TString>("Drive API is missing");
        }
        database = server.GetDatabase(server.GetDriveAPI()->GetDatabaseName());
    } else {
        database = server.GetDatabase(DBName);
    }
    if (!database) {
        return MakeUnexpected<TString>("database " + DBName + " is missing");
    }
    TRecordsSet records;
    {
        auto transaction = database->CreateTransaction(/*readonly=*/true);
        auto queryResult = transaction->Exec(Query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return MakeUnexpected<TString>("query execution failed: " + transaction->GetErrors().GetStringReport());
        }
    }

    for (auto&& record : records) {
        const TString& signal = record.Get("signal");
        const TString& valueStr = record.Get("value");
        if (valueStr.empty() || signal.empty()) {
            return MakeUnexpected<TString>("empty signal/value: " + signal + "/" + valueStr);
        }
        ui64 value = 0;
        if (!TryFromString(valueStr, value)) {
            return MakeUnexpected<TString>("cannot parse ui64 from string: " + valueStr);
        }
        TUnistatSignalsCache::SignalAdd(GetRTProcessName(),signal, value);
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TSql2UnistatProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("db_name", "Database name").SetVariants(server.GetDatabaseNames());
    scheme.Add<TFSText>("query", "SQL query").SetRequired(true);
    return scheme;
}

bool TSql2UnistatProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    if (GetPeriod() < TDuration::Seconds(10)) {
        return false;
    }
    return
        NJson::ParseField(value["db_name"], DBName) &&
        NJson::ParseField(value["query"], Query, true);
}

NJson::TJsonValue TSql2UnistatProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["db_name"] = DBName;
    result["query"] = Query;
    return result;
}

TSql2UnistatProcess::TFactory::TRegistrator<TSql2UnistatProcess> TSql2UnistatProcess::Registrator(TSql2UnistatProcess::GetTypeName());
