#include "process.h"

#include <drive/library/cpp/yt/node/cast.h>

#include <mapreduce/yt/interface/client.h>

#include <rtline/library/json/parse.h>

TExpectedState TSql2YtProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    TDatabasePtr database;
    if (!DBName) {
        if (!server.GetDriveAPI()) {
            return MakeUnexpected<TString>("Drive API is missing");
        }
        database = server.GetDatabase(server.GetDriveAPI()->GetDatabaseName());
    } else {
        database = server.GetDatabase(DBName);
    }
    if (!database) {
        return MakeUnexpected<TString>("database " + DBName + " is missing");
    }

    TInstant timestamp = Now();
    TRecordsSet records;
    {
        auto transaction = database->CreateTransaction(/*readonly=*/true);
        auto queryResult = transaction->Exec(Query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return MakeUnexpected<TString>("query execution failed: " + transaction->GetErrors().GetStringReport());
        }
    }

    TVector<NJson::TJsonValue> objects;
    for (auto&& record : records) {
        auto object = NJson::TJsonValue(NJson::JSON_MAP);
        auto parser = [&object, this](const TString& k, const TString& v) {
            NJson::TJsonValue value;
            if (StringFields.contains(k)) {
                value = v;
            } else if (v) {
                if (!NJson::ReadJsonFastTree(v, &value)) {
                    value = v;
                }
            } else {
                value = NJson::JSON_NULL;
            }
            object[k] = std::move(value);
        };
        record.Scan(parser);

        object["_timestamp"] = timestamp.MicroSeconds();
        objects.push_back(std::move(object));
    }

    NYT::IClientPtr client = NYT::CreateClient(YtCluster);
    NYT::TYPath path = YtDirectory + '/' + GetTableName(timestamp);
    NYT::TYPath current = YtDirectory + '/' + "current";
    {
        NYT::ITransactionPtr tx = client->StartTransaction();
        if (CreateYtDirectory && !tx->Exists(YtDirectory)) {
            tx->Create(YtDirectory, NYT::NT_MAP);
        }
        auto richPath = NYT::TRichYPath(path);
        if (HasYtSchema()) {
            richPath.Schema(GetYtSchema());
        }
        auto writer = tx->CreateTableWriter<NYT::TNode>(richPath);
        for (auto&& object : objects) {
            auto node = NYT::ToNode(object);
            if (HasYtSchema()) {
                node = Schematize(std::move(node));
            }
            writer->AddRow(node);
        }
        writer->Finish();
        tx->Copy(path, current, NYT::TCopyOptions().Force(true));
        tx->Commit();
    }
    if (TimeToLive) {
        auto expirationTime = timestamp + TimeToLive;
        client->Set(path + '/' + "@expiration_time", expirationTime.ToString());
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TSql2YtProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    TYtProcessTraits::FillScheme(scheme);
    scheme.Add<TFSBoolean>("create_yt_directory", "Create YT directory if necessary");
    scheme.Add<TFSArray>("string_fields", "String fields").SetElement<TFSString>();
    scheme.Add<TFSDuration>("ttl", "Time to live");
    scheme.Add<TFSText>("query", "SQL query").SetRequired(true);
    scheme.Add<TFSVariants>("db_name", "Database name").SetVariants(server.GetDatabaseNames());
    scheme.Add<TFSString>("yt_cluster", "YT cluster").SetDefault("hahn").SetRequired(true);
    scheme.Add<TFSString>("yt_directory", "YT directory").SetRequired(true);
    return scheme;
}

bool TSql2YtProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    if (!TYtProcessTraits::Deserialize(value)) {
        return false;
    }
    return
        NJson::ParseField(value["create_yt_directory"], CreateYtDirectory) &&
        NJson::ParseField(value["string_fields"], StringFields) &&
        NJson::ParseField(value["ttl"], TimeToLive) &&
        NJson::ParseField(value["query"], Query, true) &&
        NJson::ParseField(value["yt_cluster"], YtCluster, true) &&
        NJson::ParseField(value["yt_directory"], YtDirectory, true) &&
        NJson::ParseField(value["db_name"], DBName, false);
}

NJson::TJsonValue TSql2YtProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TYtProcessTraits::Serialize(result);
    result["create_yt_directory"] = CreateYtDirectory;
    result["string_fields"] = NJson::ToJson(StringFields);
    result["ttl"] = TimeToLive.ToString();
    result["query"] = Query;
    result["yt_cluster"] = YtCluster;
    result["yt_directory"] = YtDirectory;
    result["db_name"] = DBName;
    return result;
}

TString TSql2YtProcess::GetTableName(TInstant now) const {
    if (GetPeriod() >= TDuration::Days(1)) {
        return now.FormatLocalTime("%Y-%m-%d");
    } else {
        return now.ToStringLocalUpToSeconds();
    }
}

TSql2YtProcess::TFactory::TRegistrator<TSql2YtProcess> TSql2YtProcess::Registrator(TSql2YtProcess::GetTypeName());
