#include "process.h"

#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/device_snapshot/manager.h>

#include <rtline/library/geometry/polyline.h>

TExpectedState TSubscriptionCheckProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    TDBTags dbTags;
    auto tagNames = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TUserSubscriptionTag::TypeName });
    {
        auto readOnlyTx = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto optionalTags = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>(), MakeVector(tagNames), readOnlyTx);
        if (!optionalTags) {
            return MakeUnexpected<TString>(GetRobotId() + ": Cannot get user tags " + readOnlyTx.GetStringReport());
        }
        dbTags = std::move(*optionalTags);

        if (ForbiddenPushTagName) {
            auto tags = server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>(), { ForbiddenPushTagName }, readOnlyTx);
            if (!tags) {
                return MakeUnexpected<TString>(TStringBuilder() << GetRobotId() << ": Cannot get user tags for forbidden push. " << readOnlyTx.GetStringReport());
            }
            for (const auto& tag : *tags) {
                UsersWithForbiddenPush.insert(tag.GetObjectId());
            }
        }
    }

    for (auto& dbTag : dbTags) {
        auto result = ProcessSubscriptionTag(dbTag, server);
        Y_UNUSED(result);
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TSubscriptionCheckProcess::TFactory::TRegistrator<TSubscriptionCheckProcess> TSubscriptionCheckProcess::Registrator(TSubscriptionCheckProcess::GetTypeName());

bool TSubscriptionCheckProcess::ProcessSubscriptionTag(TDBTag& dbTag, const NDrive::IServer& server) const {
    auto tag = dbTag.MutableTagAs<TUserSubscriptionTag>();
    if (!tag) {
        return true;
    }
    if (tag->GetStatus() == TUserSubscriptionTag::ESubscriptionStatus::Undefined) {
        ALERT_LOG << "SubscriptionCheckProcessError: subscription tag has undefined status. Tag id" << dbTag.GetTagId();
        return false;
    }

    auto tx = server.GetDriveAPI()->template BuildTx<NSQL::Writable | NSQL::RepeatableRead | NSQL::Deferred>();
    switch (tag->GetStatus()) {
    case TUserSubscriptionTag::ESubscriptionStatus::Pending: {
        if (!ProcessPendingTag(dbTag, *tag, server, tx)) {
            return false;
        }
        break;
    }
    case TUserSubscriptionTag::ESubscriptionStatus::Active: {
        if (!ProcessActiveTag(dbTag, *tag, server, tx)) {
            return false;
        }
        break;
    }
    default :
        break;
    }
    if (!tx.Commit()) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("tag_id", dbTag.GetTagId())
            ("error", "process subscription tag: cannot commit tx")
            ("tx_error", tx.GetStringReport())
        );
        return false;
    }
    return true;
}

bool TSubscriptionCheckProcess::ProcessPendingTag(TDBTag& dbTag, TUserSubscriptionTag& tag, const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    auto paymentStatus = tag.CheckPayment(dbTag, dbTag.GetObjectId(), &server, tx);

    if (!paymentStatus) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("tag_id", dbTag.GetTagId())
            ("error", "process pending tag: cannot get payment status for subscription tag")
            ("tx_error", tx.GetStringReport())
        );
        return false;
    }
    switch (*paymentStatus) {
    case TUserSubscriptionTag::EPaymentStatus::Completed: {
        auto tagDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag.GetName());
        auto subscriptionDescription = Yensured(tagDescription)->GetAs<TUserSubscriptionTag::TDescription>();
        tag.SetSLAInstant(TInstant::Now() + subscriptionDescription->GetSLADurationRef());
        tag.SetStatus(TUserSubscriptionTag::ESubscriptionStatus::Active);
        return UpdateTag(dbTag, tag, server, tx);
    }

    case TUserSubscriptionTag::EPaymentStatus::Unknown: {
        ALERT_LOG << GetRobotId() << ": process pending tag. Payment status is unknown. Tag id: " << dbTag.GetTagId() << Endl;
        return false;
    }

    case TUserSubscriptionTag::EPaymentStatus::Failed: {
        if (!AddPushTagOnFailedPayment(server, tx, dbTag.GetObjectId(),  FailedPaymentPushTagName)) {
            return false;
        }
        if (tag.GetSLAInstant() < TInstant::Now()) {
            return RemoveTag(dbTag, server, tx);
        }
        break;
    }
    default :
        break;
    }
    return true;
}

bool TSubscriptionCheckProcess::ProcessActiveTag(TDBTag& dbTag, TUserSubscriptionTag& tag, const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    auto currentTime = TInstant::Now();
    auto deadline = currentTime + TimeShiftForPayment;

    if (tag.GetSLAInstant() > deadline) {
        return true;
    }
    auto paymentStatus = tag.CheckPayment(dbTag, dbTag.GetObjectId(), &server, tx);
    if (!paymentStatus) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("tag_id", dbTag.GetTagId())
            ("error", "process active tag : cannot get payment status for subscription tag")
            ("tx_error", tx.GetStringReport())
        );
        return false;
    }

    switch (*paymentStatus) {
    case TUserSubscriptionTag::EPaymentStatus::Failed: {
        if (!AddPushTagOnFailedPayment(server, tx, dbTag.GetObjectId(), FailedAutoRenevalPushTagName)) {
            return false;
        }
        if (tag.GetSLAInstant() < TInstant::Now()) {
            return RemoveTag(dbTag, server, tx);
        }
        if (tag.GetAutoPayment()) {
            return ApplyPayment(dbTag, tag, server, tx);
        }
        break;
    }
    case TUserSubscriptionTag::EPaymentStatus::Completed: {
        auto tagDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tag.GetName());
        auto subscriptionDescription = Yensured(tagDescription)->GetAs<TUserSubscriptionTag::TDescription>();
        tag.SetSLAInstant(tag.GetSLAInstant() + subscriptionDescription->GetSLADurationRef());
        tag.SetStatus(TUserSubscriptionTag::ESubscriptionStatus::Active);
        return UpdateTag(dbTag, tag, server, tx);
        break;
    }
    case TUserSubscriptionTag::EPaymentStatus::Unknown: {
        if (tag.GetAutoPayment()) {
            return ApplyPayment(dbTag, tag, server, tx);
        }

        if (tag.GetSLAInstant() < TInstant::Now()) {
            return RemoveTag(dbTag, server, tx);
        }
        break;
    }
    default :
        break;
    }
    return true;
}

NDrive::TScheme TSubscriptionCheckProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSDuration>("time_shift_for_payment", "Временной инвервал автоматической оплаты");
    scheme.Add<TFSString>("failed_payment_push_tag_name", "Имя тега пуша упавшего платежа");
    scheme.Add<TFSString>("failed_auto_reneval_push_tag_name", "Имя тега пуша упавшего автопродления");
    scheme.Add<TFSString>("forbidden_push_tag_name", "Имя тега запрещающего повторную отправку пуша");
    return scheme;
}

bool TSubscriptionCheckProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return NJson::ParseField(value, "time_shift_for_payment", TimeShiftForPayment, false) &&
        NJson::ParseField(value, "failed_payment_push_tag_name", FailedPaymentPushTagName, false) &&
        NJson::ParseField(value, "failed_auto_reneval_push_tag_name", FailedAutoRenevalPushTagName, false) &&
        NJson::ParseField(value, "forbidden_push_tag_name", ForbiddenPushTagName, false);
}

NJson::TJsonValue TSubscriptionCheckProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "time_shift_for_payment", TimeShiftForPayment);
    NJson::InsertField(result, "failed_payment_push_tag_name", FailedPaymentPushTagName);
    NJson::InsertField(result, "failed_auto_reneval_push_tag_name", FailedAutoRenevalPushTagName);
    NJson::InsertField(result, "forbidden_push_tag_name", ForbiddenPushTagName);
    return result;
}

bool TSubscriptionCheckProcess::UpdateTag(TDBTag& dbTag, TUserSubscriptionTag& tag, const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(dbTag, &tag, GetRobotUserId(), &server, tx)) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("tag_id", dbTag.GetTagId())
            ("error", "cannot update tag data")
            ("tx_error", tx.GetStringReport())
        );
        return false;
    }
    return true;
}

bool TSubscriptionCheckProcess::RemoveTag(TDBTag& dbTag, const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTag(dbTag, GetRobotUserId(), &server, tx)) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("tag_id", dbTag.GetTagId())
            ("error", "cannot remove tag")
            ("tx_error", tx.GetStringReport())
        );
        return false;
    }
    return true;
}


bool TSubscriptionCheckProcess::ApplyPayment(TDBTag& dbTag, TUserSubscriptionTag& tag, const NDrive::IServer& server, NDrive::TEntitySession& tx) const {
    if (!tag.ApplyPayment(dbTag, dbTag.GetObjectId(), &server, tx)) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("tag_id", dbTag.GetTagId())
            ("error", "apply payment failed")
            ("tx_error", tx.GetStringReport())
        );
        return false;
    }
    return true;
}

bool TSubscriptionCheckProcess::AddPushTagOnFailedPayment(const NDrive::IServer& server, NDrive::TEntitySession& tx, const TString& userId, const TString& pushTagName) const {
    if (!pushTagName) {
        return true;
    }

    // push have been sent before
    if (ForbiddenPushTagName && UsersWithForbiddenPush.contains(userId)) {
        return true;
    }

    auto pushTagMaybe = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(pushTagName);
    if (!pushTagMaybe) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("error", "canot create push tag")
        );
        return false;
    }
    auto pushTag = std::dynamic_pointer_cast<TUserPushTag>(pushTagMaybe);
    if (!pushTag) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("error", "cannot cast push tag")
        );
        return false;
    }
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(pushTag, GetRobotId(), userId, &server, tx)) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("user_id", userId)
            ("error", "cannot add user push tag")
        );
        return false;
    }

    auto forbiddenPushTag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(ForbiddenPushTagName);
    if (!forbiddenPushTag) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("error", "canot create tag for forbiding push tags")
        );
        return false;
    }
    auto forbiddenTag = std::dynamic_pointer_cast<TSimpleUserTag>(forbiddenPushTag);
    if (!forbiddenTag) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("error", "cannot cast forbidden push tag")
        );
        return false;
    }
    if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(forbiddenTag, GetRobotId(), userId, &server, tx)) {
        NDrive::TEventLog::Log("SubscriptionCheckProcessError", NJson::TMapBuilder
            ("user_id", userId)
            ("error", "cannot add user forbidden push tag")
        );
        return false;
    }
    return true;
}
