#include "internal_call_center.h"


TInternalCallCenterRecordsProcessor::TFactory::TRegistrator<TInternalCallCenterRecordsProcessor> TInternalCallCenterRecordsProcessor::Registrator(TInternalCallCenterRecordsProcessor::GetTypeName());
TRTInstantWatcherState::TFactory::TRegistrator<TInternalCallCenterRecordsState> TInternalCallCenterRecordsState::Registrator(TInternalCallCenterRecordsProcessor::GetTypeName());


TString TInternalCallCenterRecordsState::GetType() const {
    return TInternalCallCenterRecordsProcessor::GetTypeName();
}


NDrive::TScheme TInternalCallCenterRecordsProcessor::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("need_check", "Проверять наличие трека в mds").SetDefault(NeedCheck);
    scheme.Add<TFSNumeric>("max_calls_count", "Максимальное количество звонков в обработке за раз").SetDefault(MaxCallsCount);
    scheme.Add<TFSNumeric>("max_events_count", "Максимальное количество событий в обработке за раз").SetDefault(MaxEventsCount);
    scheme.Add<TFSNumeric>("check_events_count", "Сколько событий проверять на то, что уже обработаны роботом").SetDefault(CheckEventsCount);
    scheme.Add<TFSNumeric>("max_calls_in_request", "Загружать за раз").SetDefault(MaxCallsInRequest);
    scheme.Add<TFSNumeric>("skip_errors_count", "Сколько пропускать ошибок за итерацию").SetDefault(SkipErrorsCount);
    scheme.Add<TFSNumeric>("retry_count", "Количество повторных попыток при большом количестве ошибок (0 - пропуска итерации не будет)").SetDefault(RetryCount);
    scheme.Add<TFSStructure>("event_time_filter", "Диапазон времени для обработки").SetStructure(TDateTimeFilterConfig::GetScheme());
    scheme.Add<TFSString>("bucket_name", "В какой бакет лить").SetRequired(true);
    return scheme;
}

NJson::TJsonValue TInternalCallCenterRecordsProcessor::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "need_check", NeedCheck);
    NJson::InsertField(result, "max_calls_count", MaxCallsCount);
    NJson::InsertField(result, "max_events_count", MaxEventsCount);
    NJson::InsertField(result, "check_events_count", CheckEventsCount);
    NJson::InsertField(result, "max_calls_in_request", MaxCallsInRequest);
    NJson::InsertField(result, "skip_errors_count", SkipErrorsCount);
    NJson::InsertField(result, "retry_count", RetryCount);
    NJson::InsertNonNull(result, "event_time_filter", TimeFilterConfig);
    NJson::InsertField(result, "bucket_name", BucketName);
    return result;
}

bool TInternalCallCenterRecordsProcessor::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBase::DoDeserializeFromJson(jsonInfo)
        && NJson::ParseField(jsonInfo["need_check"], NeedCheck)
        && NJson::ParseField(jsonInfo["max_calls_count"], MaxCallsCount)
        && NJson::ParseField(jsonInfo["max_events_count"], MaxEventsCount)
        && NJson::ParseField(jsonInfo["check_events_count"], CheckEventsCount)
        && NJson::ParseField(jsonInfo["max_calls_in_request"], MaxCallsInRequest)
        && NJson::ParseField(jsonInfo["skip_errors_count"], SkipErrorsCount)
        && NJson::ParseField(jsonInfo["retry_count"], RetryCount)
        && NJson::ParseField(jsonInfo["event_time_filter"], TimeFilterConfig)
        && NJson::ParseField(jsonInfo["bucket_name"], BucketName, /* required = */ true);
}

TExpectedState TInternalCallCenterRecordsProcessor::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server) {
        ERROR_LOG << GetRobotId() << ": Fail to get server" << Endl;
        return nullptr;
    }
    const auto* supportManagerPtr = server->GetSupportCenterManager();
    if (!supportManagerPtr) {
        ERROR_LOG << GetRobotId() << ": Support center manager is not configured" << Endl;
        return nullptr;
    }
    if (!supportManagerPtr->GetMDSClient()) {
        ERROR_LOG << GetRobotId() << ": Support mds is not configured" << Endl;
        return nullptr;
    }
    auto bucket = supportManagerPtr->GetMDSClient()->GetBucket(BucketName);
    if (!bucket) {
        ERROR_LOG << GetRobotId() << ": Support mds bucket is unknown " << BucketName << Endl;
        return nullptr;
    }
    const auto& eventsReader = supportManagerPtr->GetCiptCallEventsReader();

    TCiptCallEvent::TId lastLastEventId = 0;
    if (auto lastState = dynamic_cast<const TInternalCallCenterRecordsState*>(state.Get())) {
        lastLastEventId = lastState->GetLastEventId();
    }

    auto allowedTimeRange = TimeFilterConfig.GetFilter();
    if (lastLastEventId) {
        auto session = eventsReader.BuildSession(/* readOnly = */ true);
        auto event = eventsReader.Fetch(lastLastEventId, session);
        if (!event) {
            return MakeUnexpected<TString>(GetRobotId() + ": Fail to get call events " + session.GetStringReport());
        }
        if (allowedTimeRange.From.GetOrElse(TInstant::Zero()) > event->GetEventTS()) {
            lastLastEventId = 0;
        }
    }

    TSet<TString> toSkip;
    if (lastLastEventId) {
        TRange<TCiptCallEvent::TId> eventIdsRange(TMaybe<TCiptCallEvent::TId>(), lastLastEventId);
        auto session = eventsReader.BuildSession(/* readOnly = */ true);
        auto events = eventsReader.GetObjects(session, {}, eventIdsRange, TCiptCallEvent::ServicedActions, CheckEventsCount, /* descending = */ true);
        if (!events) {
            return MakeUnexpected<TString>(GetRobotId() + ": Fail to get call events before hadling period" + session.GetStringReport());
        }
        Transform(events->begin(), events->end(), std::inserter(toSkip, toSkip.begin()), [](const auto& event) { return event.GetCallId(); });
    }

    TMaybe<TVector<TCiptCallEvent>> callEvents;
    {
        TRange<TCiptCallEvent::TId> eventIdsRange;
        if (lastLastEventId) {
            eventIdsRange.From = lastLastEventId;
        }
        auto session = eventsReader.BuildSession(/* readOnly = */ true);
        callEvents = eventsReader.GetObjects(session, lastLastEventId ? TRange<TInstant>() : allowedTimeRange, eventIdsRange, TCiptCallEvent::ServicedActions, MaxEventsCount, /* descending = */ false);
        if (!callEvents) {
            return MakeUnexpected<TString>(GetRobotId() + ": Fail to get call events " + session.GetStringReport());
        }
    }

    ui64 callsCount = 0;
    TSet<TString> callIds;
    TCiptCallEvent::TId futureEventId = lastLastEventId;
    TCiptCallEvent::TId processedEventId = lastLastEventId;
    for (auto&& callEvent : *callEvents) {
        if (callsCount >= MaxCallsCount) {
            break;
        }
        if (!toSkip.insert(callEvent.GetCallId()).second) {
            WARNING_LOG << GetRobotId() << ": Skip call event: " << callEvent.GetCallId();
            continue;
        }
        callsCount++;
        futureEventId = callEvent.GetId();
        callIds.insert(callEvent.GetCallId());
        if (callIds.size() >= MaxCallsInRequest) {
            if (!ProcessCalls(*supportManagerPtr, *bucket, callIds)) {
                break;
            }
            processedEventId = futureEventId;
        }
    }
    if (!callIds.empty() && ProcessCalls(*supportManagerPtr, *bucket, callIds)) {
        processedEventId = futureEventId;
    }

    auto newState = MakeAtomicShared<TInternalCallCenterRecordsState>();
    newState->SetLastEventId(processedEventId);
    return newState;
}

bool TInternalCallCenterRecordsProcessor::ProcessCalls(const ISupportCenterManager& manager, const NS3::TBucket& bucket, TSet<TString>& callIds) const {
    const ui32 retryCount = RetryCount ? RetryCount : 1;
    for (ui32 i(0); i < retryCount; ++i) {
        TFutureResults futures;
        for (auto&& call : callIds) {
            futures.EmplaceResult(call, manager.ProcessInternalCall(bucket, call, false));
        }
        if (!ProcessFutureResults(manager, futures, callIds)) {
            if (RetryCount) {
                continue;
            }
            callIds.clear();
            return false;
        }
    }
    callIds.clear(); // clear after failed retries
    return true;
}

bool TInternalCallCenterRecordsProcessor::ProcessFutureResults(const ISupportCenterManager& manager, TFutureResults& results, TSet<TString>& callIds) const {
    auto waiter = NThreading::WaitAll(results.GetResults());
    if (!waiter.Wait(manager.GetInternalCallCenterClient().GetRequestTimeout() + manager.GetMDSClient()->GetConfig().GetRequestTimeout())) {
        NDrive::TEventLog::Log("InternalCallCenterProcessorError", NJson::TMapBuilder
            ("error", "Iteration timeouted")
            ("call_ids", NJson::ToJson(callIds))
        );
        if (!RetryCount) {
            return false;
        }
    }

    ui64 errorsCount = 0;
    for (ui32 i = 0; i < results.GetResults().size(); ++i) {
        auto ptr = results.GetResultPtr(i);
        if (!ptr) {
            NDrive::TEventLog::Log("InternalCallCenterProcessorError", NJson::TMapBuilder
                ("error", "Wrong data counter")
                ("call_ids", NJson::ToJson(callIds))
            );
            return false;
        }
        auto& future = *ptr;
        if (!future.HasValue()) {
            NDrive::TEventLog::Log("InternalCallCenterProcessorError", NJson::TMapBuilder
                ("error", "Error in call process: " + NThreading::GetExceptionMessage(future))
                ("error_type", "unknown")
                ("call_id", results.GetCallId(i))
            );
            errorsCount++;
            continue;
        }
        auto result = future.ExtractValue();
        if (result.first == TSupportCenterManager::EProcessCallResult::Ok) {
            NDrive::TEventLog::Log("InternalCallCenterProcessorSuccess", NJson::TMapBuilder
                ("call_id", result.second)
            );
            callIds.erase(result.second);
            continue;
        }
        NDrive::TEventLog::Log("InternalCallCenterProcessorError", NJson::TMapBuilder
            ("error", "Error in call process: " + result.second)
            ("error_type", ToString(result.first))
            ("call_id", result.second)
        );
        errorsCount++;
    }

    if (errorsCount > SkipErrorsCount) {
        NDrive::TEventLog::Log("InternalCallCenterProcessorError", NJson::TMapBuilder
            ("error", "Errors count exceeded limit")
            ("error_type", "exceeded_limit")
            ("error_limit", SkipErrorsCount)
            ("error_count", errorsCount)
            ("call_ids", NJson::ToJson(callIds))
        );
        return false;
    }
    return true;
}
