#include "processor.h"

#include <drive/backend/abstract/settings.h>
#include <drive/backend/billing/accounts/trust.h>
#include <drive/backend/registrar/manager.h>
#include <drive/backend/saas/api.h>

#include <drive/library/cpp/mds/client.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <rtline/library/storage/abstract.h>

TSupportRequestsDistributor::TFactory::TRegistrator<TSupportRequestsDistributor> TSupportRequestsDistributor::Registrator("support_requests_distributor");

TExpectedState TSupportRequestsDistributor::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    TSupportRequestsDistributonManager processor(server, NotifierName);
    processor.ResolveSupportRequests(Tags);

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TSupportRequestsDistributor::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("notifier_name", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSArray>("tags", "Теги обращений").SetElement<TFSString>();

    return scheme;
}

NJson::TJsonValue TSupportRequestsDistributor::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "notifier_name", NotifierName);

    NJson::TJsonValue tagsJson = NJson::JSON_ARRAY;
    for (auto&& tag : Tags) {
        tagsJson.AppendValue(tag);
    }
    result["tags"] = std::move(tagsJson);

    return result;
}

bool TSupportRequestsDistributor::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "notifier_name", NotifierName);
    if (!jsonInfo.Has("tags") || !jsonInfo["tags"].IsArray()) {
        return false;
    }

    Tags.clear();
    auto tagsJsonArray = jsonInfo["tags"].GetArray();
    for (auto&& tagItem : tagsJsonArray) {
        Tags.push_back(tagItem.GetString());
    }

    return true;
}


TWebPhoneCallDataSyncProcessor::TFactory::TRegistrator<TWebPhoneCallDataSyncProcessor> TWebPhoneCallDataSyncProcessor::Registrator(TWebPhoneCallDataSyncProcessor::GetTypeName());

NDrive::TScheme TWebPhoneCallDataSyncProcessor::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("max_calls_count", "Максимальное количество при выборе из базы").SetDefault(MaxCallsCount);
    return scheme;
}

NJson::TJsonValue TWebPhoneCallDataSyncProcessor::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "max_calls_count", MaxCallsCount);
    return result;
}

bool TWebPhoneCallDataSyncProcessor::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBase::DoDeserializeFromJson(jsonInfo)
        && NJson::ParseField(jsonInfo["max_calls_count"], MaxCallsCount);
}

TExpectedState TWebPhoneCallDataSyncProcessor::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server) {
        ERROR_LOG << GetRobotId() << ": Fail to get server" << Endl;
        return nullptr;
    }
    const auto* supportManagerPtr = server->GetSupportCenterManager();
    if (!supportManagerPtr) {
        ERROR_LOG << GetRobotId() << ": Support center manager is not configured" << Endl;
        return nullptr;
    }
    if (!server->GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << GetRobotId() << ": MDS client is not configured" << Endl;
        return nullptr;
    }
    if (!server->GetDriveAPI()->GetMDSClient().GetBucket(supportManagerPtr->GetWebPhoneCallManager().GetBucketName())) {
        ERROR_LOG << GetRobotId() << ": MDS bucket is not found" << Endl;
        return nullptr;
    }
    TMaybe<TVector<TWebPhoneCall>> calls;
    {
        TSet<TString> syncStatuses;
        Transform(TWebPhoneCall::NeedSyncStatuses.cbegin(), TWebPhoneCall::NeedSyncStatuses.cend(), std::inserter(syncStatuses, syncStatuses.begin()), [](auto status) { return ToString(status); });
        auto supportSession = supportManagerPtr->BuildSession(/* readOnly = */ true);
        auto options = NSQL::TQueryOptions().SetGenericCondition("status", syncStatuses).SetLimit(MaxCallsCount);
        calls = supportManagerPtr->GetWebPhoneCallManager().GetObjects(supportSession, options);
        if (!calls) {
            return MakeUnexpected<TString>(GetRobotId() + ": Cannot get calls " + supportSession.GetStringReport());
        }
    }
    TMap<TString, ui32> errorsMap;
    for (auto&& [_, name] : GetEnumNames<TWebPhoneCallManager::ECallDataError>()) {
        errorsMap[name] = 0;
    }
    TVector<TWebPhoneCall> newData;
    for (auto& call : *calls) {
        TMessagesCollector errors;
        auto errorCode = supportManagerPtr->GetWebPhoneCallManager().GetCallData(*server, call.GetId(), call, errors);
        errorsMap[ToString(errorCode)] += 1;
        if (errorCode != TWebPhoneCallManager::ECallDataError::Ok) {
            WARNING_LOG << "Fail to get call data " << errors.GetStringReport() << Endl;
            continue;
        }
        newData.emplace_back(std::move(call));
    }
    TVector<TDBTag> dbTags;
    TMap<ui64, TDBTag> tags;
    auto& tagManager = server->GetDriveAPI()->GetTagsManager();
    {
        const TSet<TString> tagNames = tagManager.GetTagsMeta().GetRegisteredTagNames({TSupportOutgoingCallTag::TypeName});
        auto session = tagManager.GetUserTags().BuildTx<NSQL::Writable>();
        if (!tagManager.GetUserTags().RestoreTags({}, MakeVector(tagNames), dbTags, session)) {
            return MakeUnexpected<TString>(GetRobotId() + ": Fail to get call tags " + session.GetStringReport());
        }
        for (auto&& dbTag : dbTags) {
            auto tag = dbTag.MutableTagAs<TSupportOutgoingCallTag>();
            if (!tag) {
                return MakeUnexpected<TString>(GetRobotId() + ": Tag has wrong type " + dbTag->GetName());
            }
            ui64 originId = 0;
            if (!supportManagerPtr->GetWebPhoneCallManager().TryParseOriginCallId(tag->GetCallId(), originId)) {
                return MakeUnexpected<TString>(GetRobotId() + ": Tag has wrong call id " + tag->GetCallId());
            }
            tags.emplace(originId, dbTag);
        }
    }
    {
        auto supportSession = supportManagerPtr->BuildSession(/* readOnly = */ false);
        auto session = tagManager.GetUserTags().BuildTx<NSQL::Writable>();
        for (auto& call : newData) {
            if (!supportManagerPtr->GetWebPhoneCallManager().UpsertObject(call, GetRobotUserId(), supportSession)) {
                return MakeUnexpected<TString>(GetRobotId() + ": Cannot update call " + supportSession.GetStringReport());
            }
            auto it = tags.find(call.GetId());
            if (it != tags.end()) {
                if (!tagManager.GetUserTags().RemoveTag(it->second, GetRobotUserId(), server, session, /* force = */ true)) {
                    return MakeUnexpected<TString>(GetRobotId() + ": Fail to remove tag " + session.GetStringReport());
                }
                tags.erase(it);
            }
        }
        if (!session.Commit()) {
            return MakeUnexpected<TString>(GetRobotId() + ": Cannot commit call update " + session.GetStringReport());
        } else if (!supportSession.Commit()) {
            return MakeUnexpected<TString>(GetRobotId() + ": Cannot commit call update " + supportSession.GetStringReport());
        }
    }
    for (auto&& [code, value] : errorsMap) {
        if (value) {
            TUnistatSignalsCache::SignalAdd("webphone_data_sync", code, value);
        }
    }
    return new IRTBackgroundProcessState();
}
