#include "process.h"

#include <drive/backend/cars/car_model.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/actions/abstract.h>

#include <drive/library/cpp/taxi/fleet/client.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <rtline/library/json/parse.h>
#include <rtline/util/algorithm/ptr.h>

namespace {
    const TString StatusWorking = "working";
    const TString StatusNotWorking = "not_working";
}

TString TTaxiFleetSyncProcess::GetBrand(const TString& manufacturer) const {
    auto p = BrandRemap.find(manufacturer);
    if (p != BrandRemap.end()) {
        return p->second;
    } else {
        return manufacturer;
    }
}

TString TTaxiFleetSyncProcess::GetModel(const TString& shortName) const {
    auto p = ModelRemap.find(shortName);
    if (p != ModelRemap.end()) {
        return p->second;
    } else {
        return shortName;
    }
}

ui64 TTaxiFleetSyncProcess::GetYear(const TString& code) const {
    auto p = YearRemap.find(code);
    if (p != YearRemap.end()) {
        return p->second;
    } else {
        return 2019;
    }
}

namespace {
    TMaybe<ui64> ParseProductionYear(const TString& rawYear) {
        if (rawYear.empty()) {
            return {};
        }
        ui64 parsedYear;
        if (!TryFromString(rawYear, parsedYear)) {
            return {};
        }
        // Special case from DRIVEBACK-4074.
        if (parsedYear < 2018) {
            return {};
        }
        // Check that year has 4 digits (ignoring leading zeroes).
        if (parsedYear >= 10000) {
            return {};
        }
        return parsedYear;
    }

    TMaybe<ui64> GetProductionYear(const std::multimap<TString, TCarGenericAttachment>& documentAttachments, const TString& objectId) {
        if (auto it = documentAttachments.find(objectId); it != documentAttachments.end()) {
            TAtomicSharedPtr<IJsonBlobSerializableCarAttachment> documentAttachment = it->second.GetImpl();
            auto registryDocument = dynamic_cast<const TCarRegistryDocument*>(documentAttachment.Get());
            if (registryDocument) {
                return ParseProductionYear(registryDocument->GetProductionYear());
            }
        }
        return {};
    }
}

TExpectedState TTaxiFleetSyncProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const auto driveApi = server.GetDriveAPI();
    const auto database = driveApi;
    const auto& userTagsManager = Yensured(database)->GetTagsManager().GetUserTags();

    NDrive::TTaxiFleetClient::TOptions clientOptions;
    clientOptions.Endpoint = Endpoint;
    NDrive::TTaxiFleetClient client(clientOptions);

    TVector<TDBTag> tags;
    {
        auto session = userTagsManager.BuildSession(/*readOnly=*/true);
        if (!userTagsManager.RestoreTags({}, { AuthInfoTagName }, tags, session)) {
            return MakeUnexpected<TString>("cannot restore " + AuthInfoTagName + " tags: " + session.GetStringReport());
        }
    }
    for (auto&& tag: tags) try {
        const auto& userId = tag.GetObjectId();
        auto authInfo = tag.GetTagAs<TUserDictionaryTag>();
        Y_ENSURE(authInfo, "cannot cast " << tag.GetTagId() << " as DictionaryTag");
        auto apikey = authInfo->GetField("apikey");
        auto clientId = authInfo->GetField("client_id");
        auto parkId = authInfo->GetField("park_id");
        auto profileId = authInfo->GetField("profile_id");
        Y_ENSURE(apikey);
        Y_ENSURE(clientId);
        Y_ENSURE(parkId);
        Y_ENSURE(profileId);
        NDrive::TTaxiFleetClient::TRequest request;
        request.ClientId = *clientId;
        request.ParkId = *parkId;
        request.Token = *apikey;
        Y_ENSURE(request.ClientId);
        Y_ENSURE(request.ParkId);
        Y_ENSURE(request.Token);
        Y_ENSURE(*profileId);

        TVector<ISession::TConstPtr> currentSessions;
        Y_ENSURE(Yensured(driveApi)->GetCurrentUserSessions(userId, currentSessions, TInstant::Zero()));

        TSet<TString> currentObjectIds;
        for (auto&& currentSession : currentSessions) {
            if (!AllowedOfferConstructors.empty()) {
                auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(currentSession);
                auto currentOffer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
                auto currentOfferConstructorId = currentOffer ? currentOffer->GetBehaviourConstructorId() : TString();
                if (!AllowedOfferConstructors.contains(currentOfferConstructorId)) {
                    INFO_LOG << GetRobotId() << "skip " << userId << " due to offer " << currentOfferConstructorId << Endl;
                    continue;
                }
            }
            currentObjectIds.insert(Yensured(currentSession)->GetObjectId());
        }

        auto session = userTagsManager.BuildSession();
        auto stateTags = userTagsManager.RestoreEntityTags(userId, { StateTagName }, session);

        TSet<TString> registeredObjectIds;
        TString previousTaxiCarId;
        for (auto&& stateTag : *stateTags) {
            auto state = stateTag.GetTagAs<TUserDictionaryTag>();
            auto objectId = Yensured(state)->GetField("object_id");
            if (objectId) {
                registeredObjectIds.insert(*objectId);
            }
            auto previousCarId = Yensured(state)->GetField("previous_car_id");
            if (previousCarId) {
                previousTaxiCarId = *previousCarId;
            }
        }

        TSet<TString> addedObjectIds;
        TSet<TString> removedObjectIds;
        for (auto&& objectId : currentObjectIds) {
            if (!registeredObjectIds.contains(objectId)) {
                addedObjectIds.insert(objectId);
            }
        }
        for (auto&& objectId : registeredObjectIds) {
            if (!currentObjectIds.contains(objectId)) {
                removedObjectIds.insert(objectId);
            }
        }
        std::multimap<TString, TCarGenericAttachment> documentAttachments;
        if (UseProductionYear && !addedObjectIds.empty()) {
            documentAttachments = driveApi->GetCarAttachmentAssignments().GetAssignmentsOfType(addedObjectIds, EDocumentAttachmentType::CarRegistryDocument);
        }

        auto getOrCreateCar = [&](const TString& currentObjectId) -> NDrive::TTaxiFleetClient::TCar {
            auto currentObject = database->GetCarsData()->GetObject(currentObjectId);
            Y_ENSURE(currentObject);
            auto modelInfos = driveApi->GetModelsData()->GetCached(currentObject->GetModel());
            auto model = modelInfos.GetResultPtr(currentObject->GetModel());
            Y_ENSURE(model);

            auto setConfiguration = [&](NDrive::TTaxiFleetClient::TCar& car) {
                car.Amenities = MakeVector(Amenities);
                car.Brand = GetBrand(model->GetManufacturer());
                car.Categories = MakeVector(Categories);
                car.Color = Color;
                car.Model = GetModel(model->GetShortName());
                car.Number = currentObject->GetNumber();
                car.Year = GetYear(model->GetCode());
                if (UseProductionYear) {
                    car.Year = GetProductionYear(documentAttachments, currentObjectId).GetOrElse(car.Year);
                }
                if (currentObject->GetVin()) {
                    car.VIN = currentObject->GetVin();
                }
                if (currentObject->GetRegistrationID() != 0) {
                    car.STS = ToString(currentObject->GetRegistrationID());
                }
            };

            NDrive::TTaxiFleetClient::TOptionalCar car = client.GetCarByNumber(currentObject->GetNumber(), request).ExtractValueSync();
            if (!car) {
                NDrive::TTaxiFleetClient::TCar newCar;
                newCar.Callsign = currentObject->GetId();
                newCar.Status = StatusNotWorking;
                setConfiguration(newCar);

                if (currentObject->GetVin()) {
                    newCar.VIN = currentObject->GetVin();
                }
                if (currentObject->GetRegistrationID() != 0) {
                    newCar.STS = ToString(currentObject->GetRegistrationID());
                }
                auto asyncCar = client.CreateCar(newCar, currentObject->GetId(), request);
                asyncCar.Wait();
                NDrive::TEventLog::Log("TaxiFleetCarCreatingResult", NJson::TMapBuilder
                    ("car", NJson::ToJson(newCar))
                    ("result", NJson::ToJson(asyncCar))
                    ("profile_id", *profileId)
                    ("client_id", request.ClientId)
                    ("park_id", request.ParkId)
                    ("user_id", userId)
                );
                car = asyncCar.GetValueSync();
            }
            Y_ENSURE(car);
            Y_ENSURE(car->Id);
            if (car->Status != StatusWorking) {
                car->Status = StatusWorking;
                setConfiguration(*car);
                car->Number = currentObject->GetNumber();
                if (currentObject->GetVin()) {
                    car->VIN = currentObject->GetVin();
                }
                if (currentObject->GetRegistrationID() != 0) {
                    car->STS = ToString(currentObject->GetRegistrationID());
                }
                auto updateCarResult = client.UpdateCar(*car, request);
                updateCarResult.Wait();
                NDrive::TEventLog::Log("TaxiFleetUpdateCarResult", NJson::TMapBuilder
                    ("car", NJson::ToJson(car))
                    ("result", NJson::ToJson(updateCarResult))
                    ("profile_id", *profileId)
                    ("client_id", request.ClientId)
                    ("park_id", request.ParkId)
                    ("user_id", userId)
                );
                car = updateCarResult.GetValue();
            }
            return *car;
        };

        for (auto&& currentObjectId : addedObjectIds) {
            auto car = getOrCreateCar(currentObjectId);
            Y_ENSURE(car.Id);
            auto asyncCurrentCar = client.GetActiveCar(*profileId, request);
            asyncCurrentCar.Wait();
            NDrive::TEventLog::Log("TaxiFleetGetActiveCarResult", NJson::TMapBuilder
                ("result", NJson::ToJson(asyncCurrentCar))
                ("profile_id", *profileId)
                ("client_id", request.ClientId)
                ("park_id", request.ParkId)
                ("user_id", userId)
            );
            auto currentCar = asyncCurrentCar.GetValueSync();
            auto previousCarId = previousTaxiCarId
                ? previousTaxiCarId
                : (currentCar ? *currentCar->Id : TString{});

            Y_ENSURE(
                AddObjectIdToState(userId, currentObjectId, previousCarId, server, session),
                session.GetStringReport()
            );

            auto link = client.Link(*car.Id, *profileId, request);
            link.Wait();
            NDrive::TEventLog::Log("TaxiFleetLinkingResult", NJson::TMapBuilder
                ("car_id", *car.Id)
                ("profile_id", *profileId)
                ("client_id", request.ClientId)
                ("park_id", request.ParkId)
                ("user_id", userId)
                ("result", NJson::ToJson(link))
            );
            link.GetValue();
        }
        for (auto&& objectId : removedObjectIds) {
            TString previousObjectId;
            TString previousCarId;
            Y_ENSURE(
                RemoveObjectIdFromState(userId, objectId, previousObjectId, previousCarId, server, session),
                session.GetStringReport()
            );

            auto object = database->GetCarsData()->GetObject(objectId);
            Y_ENSURE(object);
            auto car = client.GetCarByNumber(object->GetNumber(), request).ExtractValueSync();
            if (car && car->Status == StatusWorking) {
                car->Status = StatusNotWorking;
                car->Categories = MakeVector(Categories);
                auto updateCarResult = client.UpdateCar(*car, request);
                updateCarResult.Wait();
                NDrive::TEventLog::Log("TaxiFleetUpdateCarResult", NJson::TMapBuilder
                    ("car", NJson::ToJson(car))
                    ("result", NJson::ToJson(updateCarResult))
                    ("profile_id", *profileId)
                    ("client_id", request.ClientId)
                    ("park_id", request.ParkId)
                    ("user_id", userId)
                );
                car = updateCarResult.GetValue();
            }

            if (RestoreLinking && previousObjectId) {
                auto previousCar = getOrCreateCar(previousObjectId);
                Y_ENSURE(previousCar.Id);
                previousCarId = *previousCar.Id;
            }
            if (RestoreLinking && previousCarId) {
                auto link = client.Link(previousCarId, *profileId, request);
                link.Wait();
                NDrive::TEventLog::Log("TaxiFleetRestoreLinkingResult", NJson::TMapBuilder
                    ("car_id", previousCarId)
                    ("profile_id", *profileId)
                    ("client_id", request.ClientId)
                    ("park_id", request.ParkId)
                    ("user_id", userId)
                    ("result", NJson::ToJson(link))
                );
                link.GetValue();
            }
        }
        Y_ENSURE(session.Commit(), "cannot commit transaction: " << session.GetStringReport());
    } catch (...) {
        NJson::TJsonValue info = CurrentExceptionInfo(/*forceBacktrace=*/true);
        ERROR_LOG << GetRobotId() << ": cannot process " << tag.GetObjectId() << ": " << info.GetStringRobust() << Endl;
        NDrive::TEventLog::Log("TaxiFleetError", NJson::TMapBuilder
            ("exception", std::move(info))
            ("tag_id", tag.GetTagId())
            ("user_id", tag.GetObjectId())
        );
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

bool TTaxiFleetSyncProcess::AddObjectIdToState(
    const TString& userId,
    const TString& objectId,
    const TString& previousCarId,
    const NDrive::IServer& server,
    NDrive::TEntitySession& session
) const {
    const auto api = server.GetDriveAPI();
    const auto& tagsManager = Yensured(api)->GetTagsManager();
    auto stateTag = tagsManager.GetTagsMeta().CreateTag(StateTagName);
    auto state = std::dynamic_pointer_cast<TUserDictionaryTag>(stateTag);
    Yensured(state)->SetField("object_id", objectId);
    Yensured(state)->SetField("previous_car_id", previousCarId);
    auto added = tagsManager.GetUserTags().AddTag(state, GetRobotUserId(), userId, &server, session, EUniquePolicy::NoUnique);
    return added.Defined();
}

bool TTaxiFleetSyncProcess::RemoveObjectIdFromState(
    const TString& userId,
    const TString& objectId,
    TString& previousObjectId,
    TString& previousCarId,
    const NDrive::IServer& server,
    NDrive::TEntitySession& session
) const {
    const auto api = server.GetDriveAPI();
    const auto& tagsManager = Yensured(api)->GetTagsManager();
    auto restoredTags = tagsManager.GetUserTags().RestoreEntityTags(userId, { StateTagName }, session);
    if (!restoredTags) {
        return false;
    }
    bool removed = false;
    for (auto&& tag : *restoredTags) {
        auto previousState = tag.GetTagAs<TUserDictionaryTag>();
        auto optionalObjectId = Yensured(previousState)->GetField("object_id");
        if (!optionalObjectId) {
            continue;
        }
        if (*optionalObjectId != objectId) {
            previousObjectId = *optionalObjectId;
            continue;
        }
        if (!tagsManager.GetUserTags().RemoveTag(tag, GetRobotUserId(), &server, session)) {
            return false;
        }
        auto optionalPreviousCarId = Yensured(previousState)->GetField("previous_car_id");
        previousCarId = optionalPreviousCarId.GetOrElse(previousCarId);
        removed = true;
    }
    if (!removed) {
        session.SetErrorInfo("TaxiFleetSyncProcess::RemoveObjectIdFromState", "cannot find tag for " + objectId);
        return false;
    }
    return true;
}

NDrive::TScheme TTaxiFleetSyncProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSVariants>("allowed_offer_constructors").SetVariants(IOfferBuilderAction::GetNames(impl)).SetMultiSelect(true);
    result.Add<TFSVariants>("amenities").SetVariants({
        "lightbox",
        "sticker",
    }).SetMultiSelect(true);
    result.Add<TFSVariants>("auth_info_tag").SetReference("user_tags");
    result.Add<TFSVariants>("categories").SetVariants({
        "business",
        "comfort",
        "econom",
        "personal_driver",
    }).SetMultiSelect(true);
    result.Add<TFSVariants>("color").SetVariants({
        TaxiColorBlack,
        TaxiColorYellow,
    });
    result.Add<TFSVariants>("endpoint").SetVariants({
        "https://fleet-api.taxi.tst.yandex.net",
        "https://fleet-api.taxi.yandex.net",
    });
    result.Add<TFSBoolean>("restore_linking").SetDefault(true);
    result.Add<TFSBoolean>("use_production_year").SetDefault(true);
    return result;
}

bool TTaxiFleetSyncProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["allowed_offer_constructors"], AllowedOfferConstructors) &&
        NJson::ParseField(value["amenities"], Amenities) &&
        NJson::ParseField(value["auth_info_tag"], AuthInfoTagName) &&
        NJson::ParseField(value["categories"], Categories) &&
        NJson::ParseField(value["color"], Color) &&
        NJson::ParseField(value["restore_linking"], RestoreLinking) &&
        NJson::ParseField(value["use_production_year"], UseProductionYear) &&
        NJson::ParseField(value["endpoint"], Endpoint, true);
}

NJson::TJsonValue TTaxiFleetSyncProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["allowed_offer_constructors"] = NJson::ToJson(AllowedOfferConstructors);
    result["amenities"] = NJson::ToJson(Amenities);
    result["auth_info_tag"] = AuthInfoTagName;
    result["categories"] = NJson::ToJson(Categories);
    result["color"] = Color;
    result["endpoint"] = Endpoint;
    result["restore_linking"] = RestoreLinking;
    result["use_production_year"] = UseProductionYear;
    return result;
}

TTaxiFleetSyncProcess::TFactory::TRegistrator<TTaxiFleetSyncProcess> TTaxiFleetSyncProcess::Registrator(TTaxiFleetSyncProcess::GetTypeName());
