#include "config.h"

#include <drive/backend/data/tech_dispatch.h>


TRTTechDispatchWatcherProcess::TFactory::TRegistrator<TRTTechDispatchWatcherProcess> TRTTechDispatchWatcherProcess::Registrator(TRTTechDispatchWatcherProcess::GetTypeName());

NDrive::TScheme TRTTechDispatchWatcherProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("limit_tag_name", "Тег при неуспешном назначении");
    return scheme;
}

bool TRTTechDispatchWatcherProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo, "limit_tag_name", LimitTagName)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTTechDispatchWatcherProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "limit_tag_name", LimitTagName);
    return result;
}

TExpectedState TRTTechDispatchWatcherProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    const auto& manager = server.GetDriveAPI()->GetTagsManager();

    TVector<TDBTag> taggedCars;
    auto orderTagDescriptions = manager.GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car, {TOrderTechDispatchTag::TypeName});
    {
        auto tx = manager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        if (!manager.GetDeviceTags().RestoreTagsRobust({}, MakeVector(NContainer::Keys(orderTagDescriptions)), taggedCars, tx)) {
            return MakeUnexpected<TString>("failed get car tags " + tx.GetStringReport());
        }

        size_t pos = 0;
        for (size_t i = 0; i < taggedCars.size(); ++i) {
            auto&& tag = taggedCars[i];
            if (context.GetFilteredCarIds().contains(tag.GetObjectId())) {
                if (pos < i) {
                    taggedCars[pos] = std::move(tag);
                }
                ++pos;
            }
        }
        taggedCars.resize(pos);
    }

    TVector<TAssignmentTechDispatchTag::TStationInfo> stations;
    {
        auto tx = manager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        auto optionalStations = TAssignmentTechDispatchTag::GetStationsInfo(tx, &server);
        if (!optionalStations) {
            return MakeUnexpected<TString>("failed get stations " + tx.GetStringReport());
        }
        stations = std::move(*optionalStations);
    }

    TMap<TString, std::pair<TString, TString>> completedTags;
    for (auto& station : stations) {
        for (const auto& tag : taggedCars) {
            if (completedTags.contains(tag.GetTagId())) {
                continue;
            }
            auto result = station.AddRate(tag, &server);
            switch (result) {
                case TAssignmentTechDispatchTag::TStationInfo::ERateResult::Success:
                    completedTags.emplace(tag.GetTagId(), std::make_pair(station.GetDescription().GetName(), TString()));
                    break;
                case TAssignmentTechDispatchTag::TStationInfo::ERateResult::IncorrectType:
                case TAssignmentTechDispatchTag::TStationInfo::ERateResult::IncorrectRate:
                    completedTags.emplace(tag.GetTagId(), std::make_pair(LimitTagName, ToString(result)));
                    break;
                default:
                    break;
            }
        }
        if (completedTags.size() >= taggedCars.size()) {
            break;
        }
    }

    auto robotUserPermissions = server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), {});
    if (!robotUserPermissions) {
        return MakeUnexpected<TString>("cannot create permissions");
    }

    auto tx = manager.GetDeviceTags().BuildTx<NSQL::Writable>();
    for (const auto& tag : taggedCars) {
        auto completedTag = completedTags.FindPtr(tag.GetTagId());
        TString evolveTagName = completedTag ? completedTag->first : LimitTagName;
        TString evolveTagComment = completedTag ? completedTag->second : "";
        if (evolveTagName && tag->GetName() != evolveTagName) {
            auto evolveTag = manager.GetTagsMeta().CreateTag(evolveTagName, evolveTagComment);
            if (!evolveTag) {
                ERROR_LOG << GetRobotId() << evolveTagName << " cannot create tag" << Endl;
                continue;
            }
            auto dbTag = manager.GetDeviceTags().EvolveTag(tag, evolveTag, *robotUserPermissions, &server, tx);
            if (!dbTag) {
                return MakeUnexpected<TString>("cannot evolve to " + evolveTagName + " " + tx.GetStringReport());
            }
        }
    }
    if (!tx.Commit()) {
        return MakeUnexpected<TString>("cannot commit " + tx.GetStringReport());
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();;
}
