#include "process.h"

#include <drive/backend/areas/areas.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/data/telematics.h>
#include <drive/backend/database/config.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>

#include <drive/library/cpp/threading/future.h>
#include <drive/telematics/api/sensor/interface.h>
#include <drive/telematics/api/server/client.h>
#include <drive/telematics/client/library/proxy.h>

#include <rtline/library/json/parse.h>
#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/cast.h>

NJson::TJsonValue TTelematicsCanRequesterProcess::TState::GetReport() const {
    NJson::TJsonValue result;
    result["values"] = NJson::ToJson(Values);
    return result;
}

TBlob TTelematicsCanRequesterProcess::TState::SerializeToBlob() const {
    return TBlob::FromString(GetReport().GetStringRobust());
}

bool TTelematicsCanRequesterProcess::TState::DeserializeFromBlob(const TBlob& data) {
    TStringBuf s(data.AsCharPtr(), data.Size());
    NJson::TJsonValue value;
    if (!NJson::ReadJsonFastTree(s, &value)) {
        return false;
    }
    return
        NJson::ParseField(value["values"], Values);
}

TExpectedState TTelematicsCanRequesterProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    auto currentState = std::dynamic_pointer_cast<TState>(state);
    if (!currentState) {
        WARNING_LOG << GetRobotId() << ": resetting state" << Endl;
        currentState = MakeAtomicShared<TState>();
    } else {
        INFO_LOG << GetRobotId() << ": " << currentState->GetReport().GetStringRobust() << Endl;
    }

    ui32 requestId = 0x715;
    ui8 requestIndex = 1;
    ui32 responseId = 0x77F;

    auto api = Yensured(server.GetDriveAPI());
    const auto& telematics = server.GetTelematicsClient();

    const auto& tagsMeta = api->GetTagsManager().GetTagsMeta();
    const auto& deviceTagManager = api->GetTagsManager().GetDeviceTags();

    TMap<TString, TDBTag> tags;
    if (TagName) {
        auto session = deviceTagManager.BuildSession(true);
        auto optionalTags = deviceTagManager.RestoreTags(TVector<TString>{}, { TagName }, session);
        if (!optionalTags) {
            return MakeUnexpected("cannot RestoreTags: " + session.GetStringReport());
        }
        for (auto&& tag : *optionalTags) {
            tags.emplace(tag.GetObjectId(), tag);
        }
    }

    TMap<TString, NDrive::TTelematicsClient::THandler> handlers;
    for (auto&& id : context.GetFilteredCarIds()) {
        auto imei = api->GetIMEI(id);
        if (!imei) {
            WARNING_LOG << GetRobotId() << ": no IMEI for " << id << Endl;
            continue;
        }

        NDrive::NVega::TCommandRequest::TObdRequest request;
        request.Id = requestId;
        request.Index = requestIndex;
        request.Data.assign({
            0x03, 0x22, 0x23, 0x64, 0x00, 0x00, 0x00, 0x00
        });
        request.ResponseId = responseId;
        NDrive::NVega::TCommand command(NDrive::NVega::ECommandCode::SCENARIO_OBD_REQUEST);
        command.Argument.Set(request);

        auto handler = telematics.Command(imei, command);
        handlers.emplace(id, std::move(handler));
    }

    auto timeout = TDuration::Seconds(10);
    for (auto&& [id, handler] : handlers) try {
        auto now = Now();
        auto deadline = now + timeout;
        auto response = handler.WaitAndEnsureSuccess(deadline).GetResponse<NDrive::TTelematicsClient::TCanResponse>();
        for (auto&& frame : Yensured(response)->Frames) {
            if (frame.Data.id == responseId) {
                auto value = 0.1 * frame.Data.data[4];
                INFO_LOG << GetRobotId() << ": value for " << id << ": " << value << Endl;
                NDrive::TSensor sensor;
                sensor.Since = now;
                sensor.Timestamp = now;
                sensor.Value = value;

                auto& values = currentState->Values[id];
                values.push_back(sensor);
                while (values.size() > ValuesCount) {
                    values.pop_front();
                }
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": exception while processing " << id << ": " << FormatExc(e) << Endl;
    }

    for (auto&& [id, sensors] : currentState->Values) try {
        size_t badValuesCount = 0;
        for (auto&& sensor : sensors) {
            auto value = sensor.ConvertTo<double>();
            if (value > 10) {
                continue;
            }
            if (value > 3) {
                badValuesCount += 1;
            }
        }
        if (badValuesCount > 1) {
            WARNING_LOG << GetRobotId() << ": values for " << id << " are suspicious" << Endl;
            if (TagName) {
                auto tag = tagsMeta.CreateTag(TagName);
                if (!tag) {
                    return MakeUnexpected("cannot build tag " + TagName);
                }
                Y_ENSURE(context.AddTag(id, tag));
            }
        } else {
            auto p = tags.find(id);
            if (p != tags.end()) {
                Y_ENSURE(context.RemoveTag(id, p->second));
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": exception while post-processing " << id << ": " << FormatExc(e) << Endl;
    }

    Y_ENSURE(context.ApplyModification(DryRun));
    return currentState;
}

NDrive::TScheme TTelematicsCanRequesterProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("dry_run", "Dry run");
    scheme.Add<TFSString>("tag_name", "Tag name");
    return scheme;
}

bool TTelematicsCanRequesterProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["dry_run"], DryRun) &&
        NJson::ParseField(value["tag_name"], TagName);
}

NJson::TJsonValue TTelematicsCanRequesterProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["dry_run"] = DryRun;
    result["tag_name"] = TagName;
    return result;
}

namespace {
    bool FinishDeferredCommandTag(const TDBTag& tag, const TSet<TString>& addedTags, const TSet<TString>& removedTags, const TString& robotUserId, const NDrive::IServer& server, NDrive::TEntitySession& tx) {
        const auto& tagManager = server.GetDriveDatabase().GetTagsManager();
        const auto& deviceTagManager = tagManager.GetDeviceTags();
        if (!deviceTagManager.RemoveTag(tag, robotUserId, &server, tx)) {
            return false;
        }
        for (auto&& tagName : addedTags) {
            auto addedTag = tagManager.GetTagsMeta().CreateTag(tagName);
            if (!addedTag) {
                tx.SetErrorInfo("FinishDeferredCommandTag", "cannot create tag " + tagName);
                return false;
            }
            auto added = deviceTagManager.AddTag(addedTag, robotUserId, tag.GetObjectId(), &server, tx);
            if (!added) {
                return false;
            }
        }
        {
            auto optionalRestoredTags = deviceTagManager.RestoreTagsRobust({ tag.GetObjectId() }, removedTags, tx);
            if (!optionalRestoredTags) {
                return false;
            }
            bool removed = deviceTagManager.RemoveTags(*optionalRestoredTags, robotUserId, &server, tx);
            if (!removed) {
                return false;
            }
        }
        return true;
    }
}

TExpectedState TTelematicsDeferredCommandExecutorProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const TDriveAPI* api = server.GetDriveAPI();
    if (!api) {
        ERROR_LOG << GetRobotId() << ": DriveApi is missing" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto d = api->GetTagsManager().GetTagsMeta().GetDescriptionByName(TagName);
    if (!d) {
        ERROR_LOG << GetRobotId() << ": description for " << TagName << " is missing" << Endl;
        return MakeUnexpected<TString>({});
    }
    auto description = dynamic_cast<const TTelematicsDeferredCommandTag::TDescription*>(d.Get());
    if (!description) {
        ERROR_LOG << GetRobotId() << ": description for " << TagName << " cannot be cast to TelematicsDeferredCommandTag" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto permissions = api->GetUserPermissions(GetRobotUserId(), {});
    if (!permissions) {
        ERROR_LOG << GetRobotId() << ": cannot get user permissions for " << GetRobotUserId() << Endl;
        return MakeUnexpected<TString>({});
    }

    const TDeviceTagsManager& deviceTagsManager = api->GetTagsManager().GetDeviceTags();
    TVector<TTaggedDevice> devices;
    if (!deviceTagsManager.GetObjectsFromCache({ TagName }, devices, TInstant::Zero())) {
        ERROR_LOG << GetRobotId() << ": cannot get " << TagName << " tagged objects from cache" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto addedTags = AddedTags;
    auto removedTags = RemovedTags;
    auto maybeTagNameOnSuccess = description->OptionalTagNameOnSuccess();
    if (maybeTagNameOnSuccess) {
        addedTags.insert(*maybeTagNameOnSuccess);
    }
    auto statuses = api->GetStateFiltersDB()->GetObjectStates();

    ui32 executed = 0;
    for (auto&& device : devices) {
        auto tag = device.GetTag(TagName);
        if (!tag) {
            ERROR_LOG << GetRobotId() << ": cannot get " << TagName << " for " << device.GetId() << Endl;
            continue;
        }

        auto deferredCommand = tag->GetTagAs<TTelematicsDeferredCommandTag>();
        if (!deferredCommand) {
            ERROR_LOG << GetRobotId() << ": cannot cast " << tag->GetTagId() << " to TelematicsDeferredCommandTag" << Endl;
            continue;
        }
        auto now = Now();
        if (deferredCommand->GetSince() > now) {
            INFO_LOG << GetRobotId() << ": not just yet for " << tag->GetTagId() << Endl;
            continue;
        }

        auto duration = deferredCommand->GetDuration();
        if (!duration) {
            duration = description->GetDefaultDuration();
        }
        if (deferredCommand->GetPerformer().empty()) {
            if (Limit && executed >= Limit) {
                DEBUG_LOG << GetRobotId() << ": skip " << device.GetId() << " since maximum number of executions has been reached" << Endl;
                continue;
            }

            auto status = statuses[device.GetId()];
            if (!AcceptedStatuses.contains(status)) {
                INFO_LOG << GetRobotId() << ": " << device.GetId() << " has unaccepted status " << status << Endl;
                continue;
            }
            if (!ActivityTimetable.Empty() && !ActivityTimetable.IsActualNow(now)) {
                DEBUG_LOG << GetRobotId() << ": " << device.GetId() << " timetable is not satisfied" << Endl;
                continue;
            }

            auto session = api->template BuildTx<NSQL::Writable>();
            if (duration) {
                if (!deviceTagsManager.SetTagPerformer(*tag, GetRobotUserId(), false, session, &server)) {
                    ERROR_LOG << GetRobotId() << ": cannot set tag performer for " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
            } else if (!CheckResult) {
                if (!FinishDeferredCommandTag(*tag, addedTags, removedTags, GetRobotUserId(), server, session)) {
                    ERROR_LOG << GetRobotId() << ": cannot FinishDeferredCommandTag " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
            }
            TTelematicsCommandTag::TCommand command;
            command.Code = description->GetCode();
            switch (command.Code) {
            case NDrive::NVega::ECommandCode::YADRIVE_WARMING:
            {
                NDrive::NVega::TCommandRequest::TWarming warming;
                warming.Time = duration.Minutes();
                command.Argument.Set(warming);
                break;
            }
            default:
            {
                if (command.Code != NDrive::NVega::ECommandCode::UNKNOWN) {
                    auto parameterSetting = description->GetParameterSetting();
                    parameterSetting["command"] = ToString(command.Code);
                    try {
                        command = *NDrive::ParseCommand(parameterSetting);
                    } catch (const std::exception& e) {
                        ERROR_LOG << GetRobotId() << ": cannot parse command for " << tag->GetTagId() << ": " << ToString(parameterSetting) << ": " << FormatExc(e) << Endl;
                        continue;
                    }
                }
                break;
            }
            }
            auto callback = std::function<void(const NThreading::TFuture<NDrive::TCommonCommandResponse>&)>();
            if (CheckResult) {
                callback = [
                    &server,
                    addedTags,
                    removedTags,
                    tag = *tag,
                    robotId = GetRobotId(),
                    robotUserId = GetRobotUserId()
                ] (const NThreading::TFuture<NDrive::TCommonCommandResponse>& r) {
                    const auto& response = r.GetValue();
                    if (response.Status == NDrive::TTelematicsClient::EStatus::Success) {
                        const auto& api = *Yensured(server.GetDriveAPI());
                        auto session = api.BuildTx<NSQL::Writable>();
                        if (!FinishDeferredCommandTag(tag, addedTags, removedTags, robotUserId, server, session)) {
                            ERROR_LOG << robotId << ": cannot FinishDeferredCommandTag " << tag.GetTagId() << ": " << session.GetStringReport() << Endl;
                            return;
                        }
                        if (!session.Commit()) {
                            ERROR_LOG << robotId << ": cannot commit transaction related to " << tag.GetTagId() << ": " << session.GetStringReport() << Endl;
                            return;
                        }
                        INFO_LOG << robotId << ": successfully processed tag " << tag.GetTagId() << " for " << tag.GetObjectId() << Endl;
                    } else {
                        ERROR_LOG << robotId << ": command " << response.Id << " for " << tag.GetObjectId() << " failed with status " << response.Status << Endl;
                    }
                };
            }
            auto response = TTelematicsCommandTag::Command(device.GetId(), command, TDuration::Seconds(30), *permissions, &server, session);
            if (!response.Initialized()) {
                ERROR_LOG << GetRobotId() << ": cannot execute telematics command for " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                continue;
            }
            if (!session.Commit()) {
                ERROR_LOG << GetRobotId() << ": cannot commit transaction related to " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                continue;
            }
            if (callback) {
                response.Subscribe(std::move(callback));
            }
            INFO_LOG << GetRobotId() << ": processed tag " << tag->GetTagId() << " of " << device.GetId() << Endl;
            executed += 1;
        } else {
            TCarTagsHistoryManager::TOptionalEvents events;
            {
                auto session = deviceTagsManager.BuildSession(true);
                events = deviceTagsManager.GetEventsByTag(tag->GetTagId(), session, 0, now - std::max(2 * duration, TDuration::Days(1)));
                if (!events) {
                    ERROR_LOG << GetRobotId() << ": cannot get history for " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
            }
            TInstant start;
            for (auto&& ev : *events) {
                if (!ev) {
                    continue;
                }
                if (ev.GetHistoryAction() == EObjectHistoryAction::SetTagPerformer) {
                    start = std::max(start, ev.GetHistoryInstant());
                }
            }
            if (now > start + duration) {
                auto session = api->template BuildTx<NSQL::Writable>();
                if (!deviceTagsManager.DropTagPerformer(*tag, GetRobotUserId(), session)) {
                    ERROR_LOG << GetRobotId() << ": cannot drop performer for tag " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
                if (!FinishDeferredCommandTag(*tag, addedTags, removedTags, GetRobotUserId(), server, session)) {
                    ERROR_LOG << GetRobotId() << ": cannot FinishDeferredCommandTag " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
                if (!session.Commit()) {
                    ERROR_LOG << GetRobotId() << ": cannot commit transaction related to " << tag->GetTagId() << ": " << session.GetStringReport() << Endl;
                    continue;
                }
            }
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TTelematicsDeferredCommandExecutorProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto descriptions = api ? api->GetTagsManager().GetTagsMeta().GetTagsByType(TTelematicsDeferredCommandTag::Type()) : ITagsMeta::TTagDescriptions();

    TVector<TString> tagNames;
    for (auto&& description : descriptions) {
        if (!description) {
            continue;
        }
        tagNames.push_back(description->GetName());
    }

    auto stateFiltersDb = api ? api->GetStateFiltersDB() : nullptr;
    TSet<TString> statuses = (stateFiltersDb) ? stateFiltersDb->GetAvailableStates() : TSet<TString>();

    auto registeredTags = api->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car);

    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("accepted_statuses", "Statuses that permit execution").SetVariants(statuses).SetMultiSelect(true);
    scheme.Add<TFSJson>("activity_timetable", "Tag execution timetable");
    scheme.Add<TFSVariants>("added_tags", "Tags to add after (successful) command execution").SetVariants(NContainer::Keys(registeredTags)).SetMultiSelect(true);
    scheme.Add<TFSVariants>("removed_tags", "Tags to remove after (successful) command execution").SetVariants(NContainer::Keys(registeredTags)).SetMultiSelect(true);
    scheme.Add<TFSBoolean>("check_result", "Remove tag only after successful execution");
    scheme.Add<TFSNumeric>("limit", "Maximum number of executions per run");
    scheme.Add<TFSVariants>("tag_name", "Name of the tag to execute").SetVariants(tagNames).SetRequired(true);
    return scheme;
}

bool TTelematicsDeferredCommandExecutorProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    const NJson::TJsonValue& activityTimetable = value["activity_timetable"];
    if (activityTimetable.IsDefined()) {
        if (!ActivityTimetable.DeserializeFromJson(activityTimetable)) {
            ERROR_LOG << GetRobotId() << ": cannot deserialize activity_timetable from " << activityTimetable.GetStringRobust() << Endl;
            return false;
        }
    }
    return
        NJson::ParseField(value["accepted_statuses"], AcceptedStatuses) &&
        NJson::ParseField(value["added_tags"], AddedTags) &&
        NJson::ParseField(value["removed_tags"], RemovedTags) &&
        NJson::ParseField(value["check_result"], CheckResult) &&
        NJson::ParseField(value["limit"], Limit) &&
        NJson::ParseField(value["tag_name"], TagName, true);
}

NJson::TJsonValue TTelematicsDeferredCommandExecutorProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    if (!ActivityTimetable.Empty()) {
        result["activity_timetable"] = ActivityTimetable.SerializeToJson();
    }
    if (Limit) {
        result["limit"] = Limit;
    }
    result["accepted_statuses"] = NJson::ToJson(AcceptedStatuses);
    result["added_tags"] = NJson::ToJson(AddedTags);
    result["removed_tags"] = NJson::ToJson(RemovedTags);
    result["check_result"] = CheckResult;
    result["tag_name"] = TagName;
    return result;
}

TExpectedState TTelematicsSensorRefreshProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    auto sensorsClient = server.GetSensorApi();
    auto pusher = sensorsClient ? sensorsClient->GetPusher() : nullptr;
    if (!pusher) {
        ERROR_LOG << GetRobotId() << ": pusher is not configured" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto now = Now();
    auto deadline = now + DeadlineExtension;
    auto threshold = now - AgeThreshold;
    auto snapshots = server.GetSnapshotsManager().GetSnapshots();
    for (auto&& [imei, snapshot] : snapshots.ByIMEI()) {
        for (auto&& sensor : snapshot.GetSensors()) {
            if (sensor.Timestamp < threshold) {
                NOTICE_LOG << GetRobotId() << ": pushing " << sensor.ToJson().GetStringRobust() << " for " << imei << Endl;
                pusher->Push(imei, sensor, deadline);
            }
        }
        auto location = snapshot.GetLocation();
        if (location) {
            if (location->Timestamp < threshold) {
                NOTICE_LOG << GetRobotId() << ": pushing " << location->ToJson().GetStringRobust() << " for " << imei << Endl;
                pusher->Push(imei, *location, deadline);
            }
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TExpectedState TTelematicsSensorRemapProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    Y_UNUSED(state);
    auto api = server.GetDriveAPI();
    if (!api) {
        ERROR_LOG << GetRobotId() << ": DriveApi is missing" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto client = server.GetSensorApi();
    auto pusher = client ? client->GetPusher() : nullptr;
    if (!pusher) {
        ERROR_LOG << GetRobotId() << ": pusher is missing" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto objects = MakeVector(context.GetFilteredCarIds());
    auto imeis = api->GetIMEIs(objects);

    auto sensors = client->GetSensor(imeis, Source, {}, FetchReplication);
    if (!sensors.Wait(FetchTimeout)) {
        ERROR_LOG << GetRobotId() << ": fetch timeout" << Endl;
        return MakeUnexpected<TString>({});
    }
    if (!sensors.HasValue()) {
        ERROR_LOG << GetRobotId() << ": fetch error " << NThreading::GetExceptionMessage(sensors) << Endl;
        return MakeUnexpected<TString>({});
    }

    for (auto&&[imei, multisensors] : sensors.GetValue()) {
        for (auto&& sensor : multisensors) {
            auto remapped = sensor;
            remapped.Id = Destination.Id;
            remapped.SubId = Destination.SubId;
            if (Multiplier) {
                if (std::holds_alternative<ui64>(remapped.Value)) {
                    remapped.Value = static_cast<ui64>(*Multiplier * std::get<ui64>(remapped.Value));
                } else if (std::holds_alternative<double>(remapped.Value)) {
                    remapped.Value = static_cast<double>(*Multiplier * std::get<double>(remapped.Value));
                } else {
                    ERROR_LOG << GetRobotId() << ": " << imei << " cannot multiply sensor " << Source << Endl;
                    continue;
                }
            }
            switch (remapped.Id) {
            case CAN_FUEL_LEVEL_P:
                remapped.Value = static_cast<ui64>(remapped.ConvertTo<double>());
                break;
            default:
                break;
            }
            pusher->Push(imei, remapped).Subscribe([](const NThreading::TFuture<NDrive::IPusher::TPushResult>& result) {
                if (result.HasException()) {
                    ERROR_LOG << "cannot push remapped sensor: " << NThreading::GetExceptionMessage(result) << Endl;
                } else {
                    auto pushResult = result.GetValue();
                    if (!pushResult.Written) {
                        ERROR_LOG << "cannot push remapped sensor: " << pushResult.Message << Endl;
                    }
                }
            });
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TTelematicsSensorRemapProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("source_id", "Source sensor Id").SetRequired(true);
    scheme.Add<TFSNumeric>("source_subid", "Source sensor SubId");
    scheme.Add<TFSNumeric>("destination_id", "Destination sensor Id").SetRequired(true);
    scheme.Add<TFSNumeric>("destination_subid", "Destination sensor SubId");
    scheme.Add<TFSNumeric>("multiplier", "Sensor multiplier");
    return scheme;
}

bool TTelematicsSensorRemapProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["source_id"], Source.Id, true) &&
        NJson::ParseField(value["source_subid"], Source.SubId) &&
        NJson::ParseField(value["destination_id"], Destination.Id, true) &&
        NJson::ParseField(value["destination_subid"], Destination.SubId) &&
        NJson::ParseField(value["multiplier"], Multiplier);
}

NJson::TJsonValue TTelematicsSensorRemapProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["source_id"] = Source.Id;
    result["source_subid"] = Source.SubId;
    result["destination_id"] = Destination.Id;
    result["destination_subid"] = Destination.SubId;
    result["multiplier"] = NJson::ToJson(Multiplier);
    return result;
}

NJson::TJsonValue TTelematicsSensorStatProcess::TState::GetReport() const {
    NJson::TJsonValue result;
    result["devices"] = NJson::ToJson(DeviceSensors);
    result["locations"] = NJson::ToJson(LocationSensors);
    result["timestamp"] = NJson::ToJson(Timestamp);
    return result;
}

TBlob TTelematicsSensorStatProcess::TState::SerializeToBlob() const {
    return TBlob::FromString(GetReport().GetStringRobust());
}

bool TTelematicsSensorStatProcess::TState::DeserializeFromBlob(const TBlob& data) {
    TStringBuf s(data.AsCharPtr(), data.Size());
    NJson::TJsonValue value;
    if (!NJson::ReadJsonFastTree(s, &value)) {
        return false;
    }
    return
        NJson::ParseField(value["devices"], DeviceSensors) &&
        NJson::ParseField(value["locations"], LocationSensors) &&
        NJson::ParseField(value["timestamp"], Timestamp);
}

TExpectedState TTelematicsSensorStatProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    auto previous = std::dynamic_pointer_cast<TState>(state);
    bool previousIsNull = false;
    if (!previous) {
        previous = MakeAtomicShared<TState>();
        previousIsNull = true;
    }

    auto snapshots = server.GetSnapshotsManager().GetSnapshots();
    auto timestamp = Now();
    auto previousDeviceSensors = previous->DeviceSensors;
    for (auto&& [id, snapshot] : snapshots) {
        if (context.GetFilteredCarIds().contains(id)) {
            previousDeviceSensors[id];
        }
    }

    auto next = MakeAtomicShared<TState>();
    next->DeviceSensors = previousDeviceSensors;
    next->LocationSensors = previous->LocationSensors;
    next->Timestamp = timestamp;

    auto nextSensors = next->DeviceSensors.begin();
    auto previousSensors = previousDeviceSensors.begin();
    for (auto&& [id, snapshot] : snapshots) {
        if (!context.GetFilteredCarIds().contains(id)) {
            DEBUG_LOG << GetRobotId() << ": skip " << id << Endl;
            continue;
        }
        if (!Advance(nextSensors, next->DeviceSensors.end(), id)) {
            ERROR_LOG << GetRobotId() << ": cannot find next sensors for " << id << Endl;
            return state;
        }
        if (!Advance(previousSensors, previousDeviceSensors.end(), id)) {
            ERROR_LOG << GetRobotId() << ": cannot find previous sensors for " << id << Endl;
            return state;
        }
        const auto& locationTags = snapshot.GetLocationTags();
        const auto& sensors = snapshot.GetSensors();
        for (auto&& sensor : sensors) {
            if (!Sensors.contains(sensor)) {
                continue;
            }
            auto value = sensor.TryConvertTo<double>();
            if (!value) {
                ERROR_LOG << GetRobotId() << ": cannot convert sensor " << sensor.ToJson().GetStringRobust() << " for " << id << Endl;
                continue;
            }
            nextSensors->second[sensor] = *value;
            if (!previousIsNull) {
                auto delta = *value - previousSensors->second[sensor];
                if (delta > SkipDelta) {
                    continue;
                }
                for (auto&& locationTag : locationTags) {
                    if (!LocationTags.contains(locationTag)) {
                        continue;
                    }
                    next->LocationSensors[locationTag][sensor] += delta;
                }
            }
        }
    }

    for (auto&& [locationTag, sensors] : next->LocationSensors) {
        for (auto&& [sensor, value] : sensors) {
            auto name = TStringBuilder() << locationTag << '-' << sensor.GetName();
            auto delta = value - previous->LocationSensors[locationTag][sensor];
            TUnistatSignalsCache::SignalAdd("telematics-delta", name, delta);
            INFO_LOG << GetRobotId() << ": " << name << " delta is " << delta << Endl;
        }
    }

    return next;
}

NDrive::TScheme TTelematicsSensorStatProcess::DoGetScheme(const IServerBase& server) const {
    auto impl = server.GetAs<NDrive::IServer>();
    auto api = impl ? impl->GetDriveAPI() : nullptr;
    auto tags = api ? api->GetAreasDB()->GetAreaTags() : NDrive::TLocationTags();
    auto sensorNames = TVector<TString>();
    for (auto&& sensor : NDevicesSnapshotManager::AllSensors) {
        sensorNames.push_back(ToString(sensor));
    }
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSVariants>("location_tags", "Location tags to keep track of").SetVariants(tags).SetMultiSelect(true).SetRequired(true);
    result.Add<TFSVariants>("sensors", "Sensors to keep track of").SetVariants(sensorNames).SetMultiSelect(true).SetRequired(true);
    result.Add<TFSNumeric>("skip_delta", "Max delta for offset");
    return result;
}

bool TTelematicsSensorStatProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    if (!NJson::ParseField(value["location_tags"], LocationTags, true) ||
        !NJson::ParseField(value["sensors"], SensorNames, true) ||
        !NJson::ParseField(value["skip_delta"], SkipDelta, false) || SkipDelta < 0.0)
    {
        return false;
    }
    for (auto&& name : SensorNames) {
        NDrive::TSensorId sensor;
        if (!TryFromString(name, sensor)) {
            return false;
        }
        Sensors.insert(sensor);
    }
    return true;
}

NJson::TJsonValue TTelematicsSensorStatProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["location_tags"] = NJson::ToJson(LocationTags);
    result["sensors"] = NJson::ToJson(SensorNames);
    if (SkipDelta < Max<double>()) {  // +-Max<double>() can be incorrectly deserialized back
        result["skip_delta"] = NJson::ToJson(SkipDelta);
    }
    return result;
}

bool TTelematicsSensorWatcherProcess::AddTag(const NDrive::IServer& server, const TString& objectId, TString&& comment) const {
    const IDriveTagsManager& tagsManager = server.GetDriveAPI()->GetTagsManager();
    auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(TagName, comment);
    auto session = tagsManager.GetDeviceTags().BuildTx<NSQL::Writable>();
    if (tagsManager.GetDeviceTags().AddTag(tag, GetRobotUserId(), objectId, &server, session, EUniquePolicy::Rewrite) && session.Commit()) {
        INFO_LOG << GetRobotId() << ": added tag for " << objectId << Endl;
        return true;
    } else {
        ERROR_LOG << GetRobotId() << ": cannot add tag for " << objectId << ": " << session.GetStringReport() << Endl;
        return false;
    }
}

bool TTelematicsSensorWatcherProcess::RemoveTag(const NDrive::IServer& server, const TConstDBTag& tag) const {
    const IDriveTagsManager& tagsManager = server.GetDriveAPI()->GetTagsManager();
    auto session = tagsManager.GetDeviceTags().BuildTx<NSQL::Writable>();
    auto restoredTag = tagsManager.GetDeviceTags().RestoreTag(tag.GetTagId(), session);
    if (!restoredTag) {
        ERROR_LOG << "cannot restore tag " << tag.GetTagId() << ": " << session.GetStringReport() << Endl;
        return false;
    }
    if (tagsManager.GetDeviceTags().RemoveTag(*restoredTag, GetRobotUserId(), &server, session) && session.Commit()) {
        INFO_LOG << GetRobotId() << ": removed tag " << tag.GetTagId() << " for " << tag.GetObjectId() << Endl;
        return true;
    } else {
        ERROR_LOG << GetRobotId() << ": cannot remove tag " << tag.GetTagId() << " for " << tag.GetObjectId() << ": " << session.GetStringReport() << Endl;
        return false;
    }
}

TExpectedState TTelematicsSensorWatcherProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    Y_UNUSED(context);
    const TDriveAPI* api = server.GetDriveAPI();
    if (!api) {
        ERROR_LOG << GetRobotId() << ": DriveApi is missing" << Endl;
        return MakeUnexpected<TString>({});
    }
    const IDriveTagsManager& tagsManager = api->GetTagsManager();

    const NDrive::ISensorApi* sensors = server.GetSensorApi();
    if (!sensors) {
        ERROR_LOG << GetRobotId() << ": SensorApi is missing" << Endl;
        return MakeUnexpected<TString>({});
    }

    auto asyncValues = sensors->GetSensor(Sensor, {}, FetchReplication);
    if (!asyncValues.Wait(TDuration::Seconds(10))) {
        ERROR_LOG << GetRobotId() << ": sensor " << Sensor << " request timeouted" << Endl;
        return MakeUnexpected<TString>({});
    }
    if (!asyncValues.HasValue()) {
        ERROR_LOG << GetRobotId() << ": sensor " << Sensor << " request exception: " << NThreading::GetExceptionMessage(asyncValues) << Endl;
        return MakeUnexpected<TString>({});
    }

    bool reverse = (AddValueThreshold && RemoveValueThreshold) ? (*AddValueThreshold < *RemoveValueThreshold) : true;
    auto values = asyncValues.ExtractValue();
    for (auto&&[imei, multisensors] : values) try {
        auto objectId = api->GetCarIdByIMEI(imei);
        if (!objectId) {
            DEBUG_LOG << GetRobotId() << ": no object for " << imei << Endl;
            continue;
        }
        if (!context.GetFilteredCarIds().contains(objectId)) {
            DEBUG_LOG << GetRobotId() << ": " << objectId << " is skipped" << Endl;
            continue;
        }
        auto expectedTag = tagsManager.GetDeviceTags().GetTagFromCache(objectId, TagName, TInstant::Zero());
        if (!expectedTag && expectedTag.GetError() != IEntityTagsManager::ETagError::Absent) {
            ERROR_LOG << GetRobotId() << ": cannot restore tag " << TagName << " for " << objectId << ": " << expectedTag.GetError() << Endl;
            continue;
        }

        if (multisensors.size() != 1) {
            ERROR_LOG << GetRobotId() << ": " << imei << " incorrect multisensors size: " << multisensors.size() << Endl;
            continue;
        }
        const auto& sensor = multisensors[0];
        const auto value = sensor.ConvertTo<double>();
        const auto age = TInstant::Now() - sensor.Since;
        const auto lag = TInstant::Now() - sensor.Timestamp;
        const bool actual = DeprecatedThreshold ? lag < DeprecatedThreshold.Get() : true;
        const bool stable = StableThreshold ? age >= StableThreshold.Get() : true;
        if (!expectedTag && AddValueThreshold && (reverse ^ (value >= *AddValueThreshold)) && actual && stable) {
            if (AddTag(server, objectId, TStringBuilder() << GetRobotId() << '\t' << Sensor << '\t' << value << '\t' << sensor.Timestamp.Get())) {
                continue;
            }
        }
        if (expectedTag && RemoveValueThreshold && (reverse ^ (value <= *RemoveValueThreshold))) {
            if (RemoveTag(server, *expectedTag)) {
                continue;
            }
        }
        if (LagThreshold) {
            if (expectedTag && (InverseLagThreshold ^ (lag < LagThreshold.Get()))) {
                if (RemoveTag(server, *expectedTag)) {
                    continue;
                }
            }
            if (!expectedTag && (InverseLagThreshold ^ (lag >= LagThreshold.Get()))) {
                if (AddTag(server, objectId, TStringBuilder() << GetRobotId() << '\t' << Sensor << '\t' << lag.Seconds())) {
                    continue;
                }
            }
        }
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": " << imei << " has triggered and exception: " << FormatExc(e) << Endl;
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TTelematicsSensorWatcherProcess::DoGetScheme(const IServerBase& server) const {
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    const auto registeredDeviceTags = api ? api->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::Car) : ITagsMeta::TTagDescriptionsByName();
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("id", "Sensor Id").SetRequired(true);
    scheme.Add<TFSNumeric>("subid", "Sensor SubId");
    scheme.Add<TFSNumeric>("fetch_replication", "Fetch replication");
    scheme.Add<TFSNumeric>("add_value_threshold", "Value threshold to add the tag");
    scheme.Add<TFSNumeric>("remove_value_threshold", "Value threshold to remove the tag");
    scheme.Add<TFSNumeric>("deprecated_threshold", "Deprecated threshold");
    scheme.Add<TFSNumeric>("stable_threshold", "Stable threshold");
    scheme.Add<TFSNumeric>("lag_threshold", "Lag threshold to add or remove the tag").SetDefault(7);
    scheme.Add<TFSBoolean>("inverse_lag_threshold", "Invert lag threshold logic");
    scheme.Add<TFSVariants>("tag_name", "Name of the tag to add").SetVariants(registeredDeviceTags).SetRequired(true);
    return scheme;
}

bool TTelematicsSensorWatcherProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["id"], Sensor.Id, true) &&
        NJson::ParseField(value["subid"], Sensor.SubId) &&
        NJson::ParseField(value["fetch_replication"], FetchReplication) &&
        NJson::ParseField(value["add_value_threshold"], AddValueThreshold) &&
        NJson::ParseField(value["remove_value_threshold"], RemoveValueThreshold) &&
        NJson::ParseField(value["deprecated_threshold"], DeprecatedThreshold) &&
        NJson::ParseField(value["stable_threshold"], StableThreshold) &&
        NJson::ParseField(value["lag_threshold"], LagThreshold) &&
        NJson::ParseField(value["inverse_lag_threshold"], InverseLagThreshold) &&
        NJson::ParseField(value["tag_name"], TagName, true);
}

NJson::TJsonValue TTelematicsSensorWatcherProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["id"] = Sensor.Id;
    result["subid"] = Sensor.SubId;
    result["fetch_replication"] = FetchReplication;
    result["add_value_threshold"] = NJson::ToJson(AddValueThreshold);
    result["remove_value_threshold"] = NJson::ToJson(RemoveValueThreshold);
    result["deprecated_threshold"] = NJson::ToJson(DeprecatedThreshold);
    result["stable_threshold"] = NJson::ToJson(StableThreshold);
    result["lag_threshold"] = NJson::ToJson(LagThreshold);
    result["inverse_lag_threshold"] = InverseLagThreshold;
    result["tag_name"] = TagName;
    return result;
}

TExpectedState TTelematicsStateUpdaterProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const TDriveAPI* api = server.GetDriveAPI();
    if (!api) {
        return MakeUnexpected<TString>("DriveApi is missing");
    }
    const NDrive::ISensorApi* sensorApi = server.GetSensorApi();
    if (!sensorApi) {
        return MakeUnexpected<TString>("SensorApi is missing");
    }

    const TString& dbname = api->GetConfig().GetDBName();
    const auto database = server.GetDatabase(dbname);
    if (!database) {
        return MakeUnexpected<TString>("database " + dbname + " is missing");
    }
    const auto table = database->GetTable(TableName);
    if (!table) {
        return MakeUnexpected<TString>("table " + TableName + " is missing");
    }

    TMap<TString, TMap<NDrive::TSensorId, TMaybe<NDrive::TSensor>>> snapshot;
    {
        TRecordsSet records;
        NSQL::TTableRecord condition;
        auto transaction = database->CreateTransaction(/*readonly=*/true);
        auto queryResult = table->GetRows(condition, records, transaction);
        if (!queryResult || !queryResult->IsSucceed()) {
            return MakeUnexpected<TString>("cannot acquire snapshot: " + transaction->GetErrors().GetStringReport());
        }
        for (auto&& record : records) {
            const auto& objectId = record.Get("car_id");
            if (!objectId) {
                continue;
            }
            for (auto&& sensorId : Sensors) {
                auto sensor = GetSensor(record, sensorId);
                snapshot[objectId][sensorId] = std::move(sensor);
            }
        }
    }

    TMap<TString, NDrive::TMultiSensor> delta;
    for (auto&& sensorId : Sensors) {
        auto sensors = sensorApi->GetSensor(sensorId);
        sensors.Wait();
        if (!sensors.HasValue()) {
            ERROR_LOG << GetRobotId() << ": cannot fetch " << sensorId << ": " << NThreading::GetExceptionMessage(sensors) << Endl;
            continue;
        }
        for (auto&&[imei, multisensors] : sensors.GetValue()) {
            auto objectId = api->GetCarIdByIMEI(imei);
            if (!objectId) {
                continue;
            }
            for (auto&& sensor : multisensors) {
                if (std::holds_alternative<TNull>(sensor.Value)) {
                    WARNING_LOG << GetRobotId() << ": skip sensor " << sensor << " for " << objectId << Endl;
                    continue;
                }
                auto previous = snapshot[objectId][sensor];
                if (previous) {
                    auto currentValue = sensor.ConvertTo<TString>(TString());
                    auto previousValue = previous->ConvertTo<TString>(TString());
                    if (currentValue == previousValue) {
                        continue;
                    }
                    if (sensor.Since < previous->Since) {
                        continue;
                    }
                }
                delta[objectId].push_back(sensor);
            }
        }
    }

    ui32 count = 0;
    ui32 limit = 100;
    auto transaction = database->CreateTransaction(/*readonly=*/false);
    for (auto&&[objectId, multisensors] : delta) try {
        if (multisensors.empty()) {
            continue;
        }

        NSQL::TTableRecord condition = NSQL::TRecordBuilder("car_id", objectId);
        NSQL::TTableRecord record = condition;
        for (auto&& sensor : multisensors) {
            auto name = sensor.GetName();
            record.Set(name, sensor.ConvertTo<TString>());
            record.Set(name + "_updated_at", ToString(sensor.Since.Get()));
        }
        auto queryResult = table->Upsert(record, transaction, condition);
        if (!queryResult || !queryResult->IsSucceed()) {
            return MakeUnexpected<TString>("cannot upsert record: " + transaction->GetErrors().GetStringReport());
        }
        if (count >= limit) {
            count = 0;
            if (transaction->Commit()) {
                transaction = database->CreateTransaction(/*readonly=*/false);
            } else {
                return MakeUnexpected<TString>("cannot commit transaction: " + transaction->GetErrors().GetStringReport());
            }
        }
        INFO_LOG << GetRobotId() << ": upserted record for " << objectId << Endl;
        count += 1;
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": an exception occurred with " << objectId << " : " << FormatExc(e) << Endl;
    }
    if (!transaction->Commit()) {
        return MakeUnexpected<TString>("cannot commit transaction: " + transaction->GetErrors().GetStringReport());
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TMaybe<NDrive::TSensor> TTelematicsStateUpdaterProcess::GetSensor(const NSQL::TTableRecord& record, NDrive::TSensorId sensorId) const {
    auto name = ToString(NDrive::NVega::GetSensorName(sensorId.Id));
    const TString& updated = record.Get(name + "_updated_at");
    const TString& value = record.Get(name);
    if (!value) {
        return {};
    }

    NDrive::TSensor result(sensorId);
    result.Value = NDrive::SensorValueFromString(value, sensorId.Id);
    if (updated) {
        result.Since = TInstant::ParseIso8601(updated);
    }
    return result;
}

TExpectedState TTelematicsConfiguratorSyncProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);

    NDrive::TTelematicsProxy proxy(Host, Port);
    TInstant timestamp;
    TVector<TString> imeis;
    try {
        timestamp = Now();
        imeis = proxy.List();
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": cannot fetch devices list: " << FormatExc(e) << Endl;
        return MakeUnexpected<TString>({});
    }

    const auto& server = context.GetServerAs<NDrive::IServer>();
    auto api = server.GetDriveAPI();
    auto sensors = server.GetSensorApi();
    auto pusher = sensors ? sensors->GetPusher() : nullptr;
    for (auto&& imei : imeis) {
        TString objectId;
        try {
            if (pusher) {
                NDrive::THeartbeat heartbeat(timestamp, Host);
                heartbeat.Name = NDrive::ConfiguratorHeartbeatName;
                pusher->Push(imei, heartbeat);
            }
            objectId = api->GetCarIdByIMEI(imei);
            if (objectId && Synchronize) {
                auto snapshot = server.GetSnapshotsManager().GetSnapshot(objectId);
                auto heartbeat = snapshot.GetHeartbeat();
                if (heartbeat && heartbeat->Timestamp + Threshold < timestamp) {
                    INFO_LOG << GetRobotId() << ": trying to reboot " << imei << ' ' << objectId << Endl;
                    auto pinInfo = TTelematicsConfigurationTag::GetPin(objectId, &server, ServerId, Now() + TDuration::Seconds(10));
                    proxy.Connect(imei, NContainer::Scalar(pinInfo->Value));
                    proxy.Command(NDrive::NVega::ECommandCode::RESTART);
                    proxy.Disconnect();
                    INFO_LOG << GetRobotId() << ": successfully scheduled to reboot " << imei << ' ' << objectId << Endl;
                }
            }
        } catch (const std::exception& e) {
            ERROR_LOG << GetRobotId() << ": an exception occurred during processing of " << imei << ' ' << objectId << ": " << FormatExc(e) << Endl;
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TTelematicsConfiguratorSyncProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSBoolean>("synchronize", "Try to synchronize partially offline devices");
    return scheme;
}

bool TTelematicsConfiguratorSyncProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["synchronize"], Synchronize);
}

NJson::TJsonValue TTelematicsConfiguratorSyncProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["synchronize"] = Synchronize;
    return result;
}

TExpectedState TTelematicsSettingsSyncProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    auto storage = context.GetServer().GetVersionedStorage(Storage);
    if (!storage) {
        ERROR_LOG << GetRobotId() << ": storage " << Storage << " is not found" << Endl;
        return MakeUnexpected<TString>({});
    }

    TVector<TSetting> all;
    context.GetServer().GetSettings().GetAllSettings(all, TInstant::Zero());
    for (auto&& setting : all) {
        TStringBuf key = setting.GetKey();
        if (!key.StartsWith(Prefix)) {
            continue;
        }
        TStringBuf name = key.Skip(Prefix.size());
        if (storage->SetValue(TString(name), setting.GetValue(), /*storeHistory=*/false)) {
            INFO_LOG << GetRobotId() << ": synchronized " << name << "=" << setting.GetValue() << Endl;
        } else {
            ERROR_LOG << GetRobotId() << ": cannot write key " << name << " with value " << setting.GetValue() << " to storage" << Endl;
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

IRTBackgroundProcessState::TFactory::TRegistrator<TTelematicsCanRequesterProcess::TState> TTelematicsCanRequesterProcess::TState::Registrator(TTelematicsCanRequesterProcess::GetTypeName());
IRTBackgroundProcessState::TFactory::TRegistrator<TTelematicsSensorStatProcess::TState> TTelematicsSensorStatProcess::TState::Registrator(TTelematicsSensorStatProcess::GetTypeName());

TTelematicsCanRequesterProcess::TFactory::TRegistrator<TTelematicsCanRequesterProcess> TTelematicsCanRequesterProcess::Registrator(TTelematicsCanRequesterProcess::GetTypeName());
TTelematicsSensorRefreshProcess::TFactory::TRegistrator<TTelematicsSensorRefreshProcess> TTelematicsSensorRefreshProcess::Registrator(TTelematicsSensorRefreshProcess::GetTypeName());
TTelematicsSensorRemapProcess::TFactory::TRegistrator<TTelematicsSensorRemapProcess> TTelematicsSensorRemapProcess::Registrator(TTelematicsSensorRemapProcess::GetTypeName());
TTelematicsSensorStatProcess::TFactory::TRegistrator<TTelematicsSensorStatProcess> TTelematicsSensorStatProcess::Registrator(TTelematicsSensorStatProcess::GetTypeName());
TTelematicsSensorWatcherProcess::TFactory::TRegistrator<TTelematicsSensorWatcherProcess> TTelematicsSensorWatcherProcess::Registrator(TTelematicsSensorWatcherProcess::GetTypeName());
TTelematicsStateUpdaterProcess::TFactory::TRegistrator<TTelematicsStateUpdaterProcess> TTelematicsStateUpdaterProcess::Registrator(TTelematicsStateUpdaterProcess::GetTypeName());
TTelematicsDeferredCommandExecutorProcess::TFactory::TRegistrator<TTelematicsDeferredCommandExecutorProcess> TTelematicsDeferredCommandExecutorProcess::Registrator(TTelematicsDeferredCommandExecutorProcess::GetTypeName());
TTelematicsConfiguratorSyncProcess::TFactory::TRegistrator<TTelematicsConfiguratorSyncProcess> TTelematicsConfiguratorSyncProcess::Registrator(TTelematicsConfiguratorSyncProcess::GetTypeName());
TTelematicsSettingsSyncProcess::TFactory::TRegistrator<TTelematicsSettingsSyncProcess> TTelematicsSettingsSyncProcess::Registrator(TTelematicsSettingsSyncProcess::GetTypeName());
