#pragma once

#include <drive/backend/rt_background/common/config.h>

#include <drive/telematics/protocol/vega.h>

#include <rtline/util/types/timestamp.h>

class TTelematicsCanRequesterProcess : public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;

    class TState: public IRTBackgroundProcessState {
    public:
        using TValues = TMap<TString, std::deque<NDrive::TSensor>>;

    public:
        TValues Values;

    public:
        using IRTBackgroundProcessState::IRTBackgroundProcessState;

        virtual TString GetType() const override {
            return TTelematicsCanRequesterProcess::GetTypeName();
        }

        virtual NJson::TJsonValue GetReport() const override;
        virtual TBlob SerializeToBlob() const override;
        virtual bool DeserializeFromBlob(const TBlob& /*data*/) override;

    private:
        static TFactory::TRegistrator<TState> Registrator;
    };

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_can_requester";
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TString TagName;
    ui32 ValuesCount = 2;
    bool DryRun = true;

private:
    static TFactory::TRegistrator<TTelematicsCanRequesterProcess> Registrator;
};

class TTelematicsDeferredCommandExecutorProcess : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_deferred_command_executor";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TSet<TString> AddedTags;
    TSet<TString> RemovedTags;
    TSet<TString> AcceptedStatuses = { "available" };
    TTimeRestrictionsPool<TTimeRestriction> ActivityTimetable;
    TString TagName;
    ui64 Limit = 0;
    bool CheckResult = false;

private:
    static TFactory::TRegistrator<TTelematicsDeferredCommandExecutorProcess> Registrator;
};

class TTelematicsSensorRefreshProcess : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    using TBase::TBase;

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_sensor_refresh";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

private:
    TDuration AgeThreshold = TDuration::Days(1);
    TDuration DeadlineExtension = TDuration::Days(7);

private:
    static TFactory::TRegistrator<TTelematicsSensorRefreshProcess> Registrator;
};

class TTelematicsSensorRemapProcess : public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;

public:
    using TBase::TBase;

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_sensor_remap";
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    NDrive::TSensorId Source;
    NDrive::TSensorId Destination;
    TMaybe<float> Multiplier;

    ui32 FetchReplication = 7;
    TDuration FetchTimeout = TDuration::Seconds(10);

private:
    static TFactory::TRegistrator<TTelematicsSensorRemapProcess> Registrator;
};

class TTelematicsSensorStatProcess: public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;

    class TState: public IRTBackgroundProcessState {
    public:
        using TSensorValues = TMap<NDrive::TSensorId, double>;
        using TDeviceSensors = TMap<TString, TSensorValues>;
        using TLocationSensors = TMap<TString, TSensorValues>;

    public:
        TDeviceSensors DeviceSensors;
        TLocationSensors LocationSensors;
        TInstant Timestamp;

    public:
        using IRTBackgroundProcessState::IRTBackgroundProcessState;

        virtual TString GetType() const override {
            return TTelematicsSensorStatProcess::GetTypeName();
        }

        virtual NJson::TJsonValue GetReport() const override;
        virtual TBlob SerializeToBlob() const override;
        virtual bool DeserializeFromBlob(const TBlob& /*data*/) override;

    private:
        static TFactory::TRegistrator<TState> Registrator;
    };

public:
    using TBase::TBase;

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_sensor_stat";
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    NDrive::TLocationTags LocationTags;
    TSet<NDrive::TSensorId> Sensors;
    TSet<TString> SensorNames;
    double SkipDelta = Max<double>();

private:
    static TFactory::TRegistrator<TTelematicsSensorStatProcess> Registrator;
};

class TTelematicsSensorWatcherProcess: public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;

public:
    using TBase::TBase;

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_sensor_watcher";
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    bool AddTag(const NDrive::IServer& server, const TString& objectId, TString&& comment) const;
    bool RemoveTag(const NDrive::IServer& server, const TConstDBTag& tag) const;

private:
    NDrive::TSensorId Sensor;
    ui32 FetchReplication = 7;
    TMaybe<double> AddValueThreshold;
    TMaybe<double> RemoveValueThreshold;
    TSeconds DeprecatedThreshold;
    TSeconds StableThreshold;
    TSeconds LagThreshold;
    bool InverseLagThreshold = false;
    TString TagName;

private:
    static TFactory::TRegistrator<TTelematicsSensorWatcherProcess> Registrator;
};

class TTelematicsStateUpdaterProcess : public IRTRegularBackgroundProcess {
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_state_updater";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

private:
    TMaybe<NDrive::TSensor> GetSensor(const NSQL::TTableRecord& record, NDrive::TSensorId sensorId) const;

private:
    TVector<NDrive::TSensorId> Sensors = {
        VEGA_MCU_FIRMWARE_VERSION,
        VEGA_GSM_FIRMWARE_VERSION,
        VEGA_GPS_FIRMWARE_VERSION,
        CAN_ODOMETER_KM,
        CAN_FUEL_LEVEL_P,
    };
    TString TableName = "car_telematics_state";

private:
    static TFactory::TRegistrator<TTelematicsStateUpdaterProcess> Registrator;
};

class TTelematicsConfiguratorSyncProcess : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_configurator_sync";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    TString Host = "mt.carsharing.yandex.net";
    ui16 Port = 24690;
    ui16 ServerId = 1;
    TDuration Threshold = TDuration::Minutes(10);
    bool Synchronize = false;

private:
    static TFactory::TRegistrator<TTelematicsConfiguratorSyncProcess> Registrator;
};

class TTelematicsSettingsSyncProcess : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "telematics_settings_sync";
    }

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

private:
    TString Storage = "telematics_settings";
    TString Prefix = "telematics.settings.";

private:
    static TFactory::TRegistrator<TTelematicsSettingsSyncProcess> Registrator;
};
