#include "process.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/cars/hardware.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/dictionary_tags.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/roles/manager.h>

#include <drive/library/cpp/yt/common/init.h>

#include <mapreduce/yt/interface/client.h>
#include <mapreduce/yt/interface/io-inl.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTRoadChargesWatcher> TRTRoadChargesWatcher::Registrator(TRTRoadChargesWatcher::GetTypeName());
TRTInstantWatcherState::TFactory::TRegistrator<TRTRoadChargesWatcherState> TRTRoadChargesWatcherState::Registrator(TRTRoadChargesWatcher::GetTypeName());

TString TRTRoadChargesWatcherState::GetType() const {
    return TRTRoadChargesWatcher::GetTypeName();
}

TExpectedState TRTRoadChargesWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    const TRTRoadChargesWatcherState* rState = dynamic_cast<const TRTRoadChargesWatcherState*>(state.Get());
    TInstant from = rState ? rState->GetLastInstant() : FirstInstant;
    TInstant until = ModelingNow() - Delay;
    if (MaxPeriod) {
        until = Min(from + MaxPeriod, until);
    }
    if (LastInstant && until > LastInstant) {
        until = LastInstant;
    }
    if (from >= until) {
        return state;
    }
    TInstant lastEvent = from;
    auto filteredCars = context.GetFilteredCarIds();
    TVector<TTransponderEvent> events;
    for (ui32 i = 0; i < RoadsInfo.size(); ++i) {
        if (RoadsInfo[i].GetRoadType()) {
            auto retrievedEvents = RoadsInfo[i].GetRoadType()->GetTransponderEvents(from, until, server, filteredCars);
            if (retrievedEvents) {
                for (auto&& ev : retrievedEvents.GetRef()) {
                    lastEvent = Max(lastEvent, ev.GetDateTime());
                    ev.SetRoadType(i);
                    events.push_back(std::move(ev));
                }
            } else {
                TTollRoadLogger::CantRetrieveEvents(GetRTProcessName());
                TString error = retrievedEvents.GetError();
                return MakeUnexpected<TString>(std::move(error));
            }
        } else {
            TTollRoadLogger::CantRetrieveEvents(GetRTProcessName());
            return nullptr;
        }
    }

    TVector<TTransponderEvent> filtredEvents;
    TVector<TFailedEvent> failedEvents;
    for (auto&& ev : events) {
        if (ev.GetDateTime() < lastEvent) {
            if (ev.GetCarId()) {
                filtredEvents.push_back(std::move(ev));
            } else {
                TTollRoadLogger::CarNotFound(ev.GetTransponderId(), GetRTProcessName());
                failedEvents.push_back(std::make_pair(std::move(ev), "car_not_found"));
            }
        }
    }
    auto compiledEvents = FillSessionData(filtredEvents, failedEvents, server);
    if (!DumpFailedEvents(failedEvents, server)) {
        return MakeUnexpected<TString>("can't dump failed events");
    }
    if (!AddBillingTags(compiledEvents, server)) {
        return MakeUnexpected<TString>("can't add billing tags");
    }
    TAtomicSharedPtr<TRTRoadChargesWatcherState> result = new TRTRoadChargesWatcherState();
    result->SetLastInstant(lastEvent);
    return result;
}

TVector<TTransponderEvent> TRTRoadChargesWatcher::FillSessionData(TVector<TTransponderEvent>& events, TVector<TFailedEvent>& failedEvents, const NDrive::IServer& server) const {
    TVector<TTransponderEvent> compiledEvents;
    for (auto&& event : events) {
        auto rides = GetHistoryRides(event, server);
        if (rides && !rides->empty()) {
            if (!FillRidingData(event, rides.GetRef(), server)) {
                TTollRoadLogger::SessionNotFound(event.GetTransponderId(), event.GetCarId(), GetRTProcessName());
                failedEvents.push_back(std::make_pair(std::move(event), "session_not_found"));
                continue;
            }
            if (!CheckOfferAttributes(event, server)) {
                continue;
            }
            compiledEvents.push_back(std::move(event));
        } else {
            failedEvents.push_back(std::make_pair(std::move(event), "history_rides_not_found"));
        }
    }
    return compiledEvents;
}

TMaybe<TVector<THistoryRideObject>> TRTRoadChargesWatcher::GetHistoryRides(const TTransponderEvent& event, const NDrive::IServer& server) const {
    TVector<THistoryRideObject> rides;
    THistoryRidesContext ridesContext(server, event.GetDateTime());
    auto tx = Yensured(server.GetDriveAPI())->GetMinimalCompiledRides().BuildSession(true);
    auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("rtroad_charges_watcher", &server);
    if (!ridesContext.InitializeCars({ event.GetCarId() }, tx, ydbTx)) {
        TTollRoadLogger::CantInitSession(event.GetTransponderId(), event.GetCarId(), GetRTProcessName());
        return Nothing();
    }
    bool hasMore = true;
    rides = ridesContext.GetSessions(event.GetDateTime(), MaxSessionsToProcess, &hasMore);
    if (hasMore) {
        TTollRoadLogger::TooManySessions(event.GetTransponderId(), event.GetCarId(), GetRTProcessName());
        return Nothing();
    }
    return rides;
}

bool TRTRoadChargesWatcher::FillRidingData(TTransponderEvent& event, TVector<THistoryRideObject>& rides, const NDrive::IServer& server) const {
    THistoryRideObject::FetchFullRiding(&server, rides);
    for (auto&& ride : rides) {
        auto fullCompiledRiding = ride.GetFullCompiledRiding();
        if (!fullCompiledRiding) {
            TTollRoadLogger::CantInitSession(event.GetTransponderId(), event.GetCarId(), GetRTProcessName());
            continue;
        }
        const TFullCompiledRiding& fullRiding = *fullCompiledRiding;
        TInstant ridingStart = TInstant::Zero();
        if (fullRiding.HasSnapshotsDiff() && fullRiding.GetSnapshotsDiffUnsafe().HasMileage() && fullRiding.GetSnapshotsDiffUnsafe().GetMileageUnsafe() < MinRideDistance) {
            continue;
        }
        for (const auto& localEvent : fullRiding.GetLocalEvents()) {
            if (ridingStart != TInstant::Zero() && localEvent.GetInstant() > event.GetDateTime()) {
                event.SetSessionId(fullRiding.GetSessionId()).SetOffer(fullRiding.GetOffer()).SetUserId(ride.GetUserId());
                return true;
            } else if (localEvent.GetTagName() == "old_state_riding" && localEvent.GetInstant() <= event.GetDateTime()) {
                ridingStart = localEvent.GetInstant();
            } else {
                ridingStart = TInstant::Zero();
            }
        }
        if (ridingStart != TInstant::Zero()) {
            event.SetSessionId(fullRiding.GetSessionId()).SetOffer(fullRiding.GetOffer()).SetUserId(ride.GetUserId());
            return true;
        }
    }
    return false;
}

TMaybe<bool> TRTRoadChargesWatcher::CheckIfShouldBill(const TTransponderEvent& event, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    if (event.GetRoadType() < 0 || event.GetRoadType() >= RoadsInfo.size()) {
        return Nothing();
    }
    TDBTag settingsTag = server.GetDriveAPI()->GetUserSettings(event.GetUserId(), session);
    if (!settingsTag.HasData()) {
        return Nothing();
    }
    TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
    if (!settingsData) {
        return Nothing();
    }

    auto road = RoadsInfo[event.GetRoadType()];
    if (road.GetBillingAllowedSettingName()) {
        auto billingSetting = settingsData->GetField(road.GetBillingAllowedSettingName());
        bool billingAllowed;
        if (billingSetting && TryFromString<bool>(billingSetting.GetRef(), billingAllowed) && billingAllowed) {
            return true;
        }
    }

    TMaybe<bool> result = {false};
    if (road.GetFreeRidesNumberSettingName()) {
        auto numberSetting = settingsData->GetField(road.GetFreeRidesNumberSettingName());
        ui32 currentFreeRides;
        if (!numberSetting || !TryFromString<ui32>(numberSetting.GetRef(), currentFreeRides)) {
            currentFreeRides = 0;
        }
        if (road.GetBillingAllowedSettingName() && currentFreeRides >= road.GetFreeRidesNumber()) {
            settingsData->SetField(road.GetBillingAllowedSettingName(), ToString(true));
            result = {true};
        } else {
            ++currentFreeRides;
            settingsData->SetField(road.GetFreeRidesNumberSettingName(), ToString(currentFreeRides));
        }
        if (settingsTag.GetTagId().empty()) {
            if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(settingsTag.GetData(), GetRobotUserId(), event.GetUserId(), &server, session)) {
                result = Nothing();
            }
        } else {
            if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(settingsTag, event.GetUserId(), session)) {
                result = Nothing();
            }
        }
    } else {
        result = {true};
    }
    return result;
}

void TrySetLandingContext(ITag::TPtr tag, const TTransponderEvent& event, TString dateTimeFormat, const NDrive::IServer& server) {
    TLandingUserTag* landingTag = dynamic_cast<TLandingUserTag*>(tag.Get());
    if (landingTag) {
        landingTag->SetContextParameter(ToString(TRTRoadChargesWatcher::EContextParameterKey::BillingAmount), server.GetLocalization()->FormatPrice(ELocalization::Rus, event.GetAmount()));
        landingTag->SetContextParameter(ToString(TRTRoadChargesWatcher::EContextParameterKey::EventTimestamp), event.GetDateTime().FormatLocalTime(dateTimeFormat.data()));
        if (event.GetEntryLocation()) {
            landingTag->SetContextParameter(ToString(TRTRoadChargesWatcher::EContextParameterKey::TollRoadEntry), event.GetEntryLocation());
        }
        if (event.GetExitLocation()) {
            landingTag->SetContextParameter(ToString(TRTRoadChargesWatcher::EContextParameterKey::TollRoadExit), event.GetExitLocation());
        }
    }
}

void TrySetBillingContext(ITag::TPtr tag, const TTransponderEvent& event, const NDrive::IServer& server) {
    TOperationTag* operationTag = dynamic_cast<TOperationTag*>(tag.Get());
    if (operationTag) {
        operationTag->SetAmount(event.GetAmount());
        operationTag->SetSessionId(event.GetSessionId());
        ICommonOffer::TPtr offerPtr = event.GetOffer();
        if (offerPtr) {
            if (offerPtr->GetSelectedCharge() && server.GetDriveAPI()->HasBillingManager()) {
                auto userAccounts = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetUserAccounts(event.GetUserId());
                for (auto&& account : userAccounts) {
                    if (account->GetUniqueName() == offerPtr->GetSelectedCharge() && account->EnableTollRoadsPay()) {
                        const auto& customTollRoads = MakeSet(account->GetTollRoadsToPayFor());
                        if (!customTollRoads || customTollRoads.contains(::ToString(event.GetRoadType()))) {
                            operationTag->SetAdditionalAccounts({offerPtr->GetSelectedCharge()});
                            break;
                        }
                    }
                }
            }
            operationTag->SetPreferredCard(offerPtr->GetSelectedCreditCard());
        }
    }
}

bool TRTRoadChargesWatcher::AddBillingTags(const TVector<TTransponderEvent>& events, const NDrive::IServer& server) const {
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    TVector<TString> freeRideNotifications;
    TVector<TString> fullRideNotifications;
    for (auto&& event : events) {
        TVector<ITag::TPtr> tags;
        auto shouldBill = CheckIfShouldBill(event, server, session);
        if (shouldBill.Empty()) {
            TTollRoadLogger::BillingCheckFailed(event.GetUserId(), GetRTProcessName());
            return false;
        } else if (!shouldBill.GetRef()) {
            for (auto&& tagName : RoadsInfo[event.GetRoadType()].GetBillingSkippedTagNames()) {
                TString comment = "Skipped charge: " + event.ToString();
                auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName, comment);
                TrySetLandingContext(tag, event, RoadsInfo[event.GetRoadType()].GetLandingDateTimeFormat(), server);
                tags.push_back(tag);
            }
            freeRideNotifications.emplace_back(event.ToString());
        } else {
            for (auto&& tagName : RoadsInfo[event.GetRoadType()].GetBillingTagNames()) {
                TString comment = "Toll road charge charge: " + event.ToString();
                auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(tagName, comment);
                TrySetLandingContext(tag, event, RoadsInfo[event.GetRoadType()].GetLandingDateTimeFormat(), server);
                TrySetBillingContext(tag, event, server);
                tags.push_back(tag);
            }
            fullRideNotifications.emplace_back(event.ToString());
        }
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTags(tags, GetRobotUserId(), event.GetUserId(), &server, session)) {
            TTollRoadLogger::FailedToAddTags(session.GetMessages().GetStringReport(), GetRTProcessName());
            return false;
        }
    }
    if (!session.Commit()) {
        TTollRoadLogger::FailedToAddTags(session.GetMessages().GetStringReport(), GetRTProcessName());
        return false;
    }
    if (freeRideNotifications.size()) {
        NDrive::INotifier::MultiLinesNotify(server.GetNotifier(NotifierName), "Free rides" + GetRTProcessName(), freeRideNotifications);
    }
    if (fullRideNotifications.size()) {
        NDrive::INotifier::MultiLinesNotify(server.GetNotifier(NotifierName), "Paid rides" + GetRTProcessName(), fullRideNotifications);
    }
    TTollRoadLogger::BillingTagAssigned(events.size(), GetRTProcessName());
    return true;
}

bool TRTRoadChargesWatcher::CheckOfferAttributes(const TTransponderEvent& sessionInfo, const NDrive::IServer& server) const {
    if (OfferAttributesFilter.IsEmpty()) {
        return true;
    }
    ICommonOffer::TPtr offerPtr = sessionInfo.GetOffer();
    if (!offerPtr) {
        TTollRoadLogger::NoOffer(sessionInfo.GetSessionId(), GetRTProcessName());
        return false;
    }
    auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(offerPtr->GetBehaviourConstructorId());
    auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
    if (!offerBuilder) {
        TTollRoadLogger::NoOfferBuilderAction(sessionInfo.GetSessionId(), GetRTProcessName());
        return false;
    }
    if (!(OfferAttributesFilter.IsMatching(offerBuilder->GetGrouppingTags()))) {
        TTollRoadLogger::FilteredOut(GetRTProcessName());
        return false;
    }
    return true;
}

bool TRTRoadChargesWatcher::DumpFailedEvents(const TVector<TFailedEvent>& failedEvents, const NDrive::IServer& server) const {
    if (failedEvents.empty()) {
        return true;
    }
    if (FailedEventsNotifierName) {
        for (auto&& failedEvent : failedEvents) {
            TString carData;
            if (failedEvent.first.GetCarId()) {
                TMaybe<TDriveCarInfo> carObject = server.GetDriveAPI()->GetCarsData()->GetObject(failedEvent.first.GetCarId());
                if (carObject) {
                    carData = carObject->GetHRReport();
                }
            }
            NDrive::INotifier::Notify(server.GetNotifier(FailedEventsNotifierName), failedEvent.first.ToString() + "\n" + carData + " Fail reason: " + failedEvent.second);
        }
    }
    if (!YTTablePath || !YTClusterName) {
        TTollRoadLogger::YTDumpSkipped(YTTablePath, YTClusterName, GetRTProcessName());
        return true;
    }
    NYT::JoblessInitializeOnce();
    auto ytClient = NYT::CreateClient(YTClusterName, NYT::TCreateClientOptions());
    try {
        auto transaction = ytClient->StartTransaction();
        auto writer = transaction->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(YTTablePath).Append(true));
        for (auto&& failedEvent : failedEvents) {
            NYT::TNode recordNode;
            recordNode["transponder_id"] = failedEvent.first.GetTransponderId();
            recordNode["car_id"] = failedEvent.first.GetCarId();
            recordNode["datetime"] = ToString(failedEvent.first.GetDateTime());
            recordNode["session_id"] = ToString(failedEvent.first.GetSessionId());
            recordNode["entry"] = ToString(failedEvent.first.GetEntryLocation());
            recordNode["exit"] = ToString(failedEvent.first.GetExitLocation());
            recordNode["amount"] = failedEvent.first.GetAmount();
            recordNode["fail_reason"] = failedEvent.second;
            writer->AddRow(recordNode);
        }
        writer->Finish();
        transaction->Commit();
    } catch (const std::exception& e) {
        TTollRoadLogger::YTDumpFailed(YTTablePath, YTClusterName, FormatExc(e), GetRTProcessName());
        return false;
    }
    return true;
}

NDrive::TScheme TRTRoadChargesWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("first_instant", "Искать проезды с даты").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSDuration>("delay", "Искать проезды с отставанием (мин)").SetDefault(TDuration::Zero());
    scheme.Add<TFSDuration>("max_period", "Максимальный обрабатываемый период (0 - без ограничения)").SetDefault(TDuration::Zero());
    scheme.Add<TFSNumeric>("last_instant", "Искать проезды до даты").SetVisual(TFSNumeric::EVisualType::DateTime).SetDefault(0);
    scheme.Add<TFSNumeric>("min_ride_distance", "Минимальная длина поездки (м)").SetDefault(0);
    scheme.Add<TFSNumeric>("max_sessions", "Максимальное количество сессий в поиске").SetDefault(100000);
    scheme.Add<TFSString>("yt_table_path", "Путь к таблице для сохранения ошибок");
    scheme.Add<TFSString>("yt_cluster_name", "Кластер YT");
    scheme.Add<TFSString>("offer_attribute_filter", "Фильтр для атрибутов офферов");
    scheme.Add<TFSVariants>("notifier", "Нотифаер").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSVariants>("failed_notifier", "Нотифаер для нераспознанных транспондеров").SetVariants(server.GetNotifierNames());

    {
        NDrive::TScheme& roadTypes = scheme.Add<TFSArray>("road_types", "Дороги", 100000).SetElement<NDrive::TScheme>();
        roadTypes.Add<TFSVariants>("road_type", "Дорога").InitVariantsClass<ITollRoadConfig>().SetRequired(true);
        roadTypes.Add<TFSText>("config", "Конфигурация клиента").SetRequired(true);
        roadTypes.Add<TFSString>("billing_allowed_setting", "Поле тега настроек с разрешением снимать деньги").SetRequired(false);
        roadTypes.Add<TFSString>("free_rides_number_setting", "Поле тега настроек с счетчиком бесплатных проездов").SetRequired(false);
        roadTypes.Add<TFSNumeric>("free_rides_number", "Количество бесплатных проездов").SetRequired(false);
        roadTypes.Add<TFSArray>("billing_tags", "Теги, которые навесятся при оплате проезда").SetElement<TFSString>().SetRequired(false);
        roadTypes.Add<TFSArray>("billing_skipped_tags", "Теги, которые навесятся при бесплатном проезде").SetElement<TFSString>().SetRequired(false);
        roadTypes.Add<TFSString>("landing_datetime_format", "Формат вывода даты в интроскрин").SetRequired(false);
    }
    return scheme;
}

bool TRTRoadChargesWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    bool success = true;
    success &= NJson::ParseField(jsonInfo["first_instant"], FirstInstant, true);
    success &= NJson::ParseField(jsonInfo["last_instant"], LastInstant);
    success &= NJson::ParseField(jsonInfo["yt_table_path"], YTTablePath);
    success &= NJson::ParseField(jsonInfo["yt_cluster_name"], YTClusterName);
    success &= NJson::ParseField(jsonInfo["min_ride_distance"], MinRideDistance);
    success &= NJson::ParseField(jsonInfo["max_sessions"], MaxSessionsToProcess);
    success &= NJson::ParseField(jsonInfo["delay"], Delay);
    success &= NJson::ParseField(jsonInfo["max_period"], MaxPeriod);
    success &= NJson::ParseField(jsonInfo["notifier"], NotifierName);
    success &= NJson::ParseField(jsonInfo["failed_notifier"], FailedEventsNotifierName);
    if (!success || !jsonInfo["road_types"].IsArray()) {
        return false;
    }
    TString offerFilter;
    if (NJson::ParseField(jsonInfo["offer_attribute_filter"], offerFilter)) {
        OfferAttributesFilter = TTagsFilter::BuildFromString(offerFilter);
    } else {
        return false;
    }
    for (auto&& typeJson : jsonInfo["road_types"].GetArraySafe()) {
        TRoadInfo roadInfo;
        if (roadInfo.DeserializeFromJson(typeJson)) {
            RoadsInfo.push_back(std::move(roadInfo));
        } else {
            return false;
        }
    }
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTRoadChargesWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "first_instant", FirstInstant);
    TJsonProcessor::Write(result, "last_instant", LastInstant);
    TJsonProcessor::WriteDurationString(result, "delay", Delay);
    TJsonProcessor::WriteDurationString(result, "max_period", MaxPeriod);
    TJsonProcessor::Write(result, "yt_table_path", YTTablePath);
    TJsonProcessor::Write(result, "yt_cluster_name", YTClusterName);
    TJsonProcessor::Write(result, "min_ride_distance", MinRideDistance);
    TJsonProcessor::Write(result, "max_sessions", MaxSessionsToProcess);
    TJsonProcessor::Write(result, "offer_attribute_filter", OfferAttributesFilter.ToString());
    TJsonProcessor::Write(result, "notifier", NotifierName);
    TJsonProcessor::Write(result, "failed_notifier", FailedEventsNotifierName);

    if (RoadsInfo.size()) {
        NJson::TJsonValue& roadTypesJson = result.InsertValue("road_types", NJson::JSON_ARRAY);
        for (auto&& road : RoadsInfo) {
            roadTypesJson.AppendValue(road.SerializeToJson());
        }
    }
    return result;
}

bool TRTRoadChargesWatcher::TRoadInfo::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    TString type;
    if (!NJson::ParseField(jsonInfo["road_type"], type)) {
        return false;
    }
    RoadConfig = ITollRoadConfig::TFactory::Construct(type);
    if (!RoadConfig || !RoadConfig->DeserializeFromJson(jsonInfo["config"])) {
        return false;
    }
    RoadType = RoadConfig->Construct();
    if (!RoadType) {
        return false;
    }
    if (!NJson::ParseField(jsonInfo["billing_allowed_setting"], BillingAllowedSettingName)
        || !NJson::ParseField(jsonInfo["free_rides_number_setting"], FreeRidesNumberSettingName)
        || !NJson::ParseField(jsonInfo["free_rides_number"], FreeRidesNumber)
        || !NJson::ParseField(jsonInfo["landing_datetime_format"], LandingDateTimeFormat)) {
            return false;
    }
    if (!TJsonProcessor::ReadContainer(jsonInfo, "billing_tags", BillingTagNames)
        || !TJsonProcessor::ReadContainer(jsonInfo, "billing_skipped_tags", BillingSkippedTagNames)) {
        return false;
    }
    return true;
}

NJson::TJsonValue TRTRoadChargesWatcher::TRoadInfo::SerializeToJson() const {
    NJson::TJsonValue roadInfoJson = NJson::JSON_MAP;
    if (RoadType) {
        roadInfoJson.InsertValue("road_type", RoadType->GetType());
    }
    if (RoadConfig) {
        roadInfoJson.InsertValue("config", RoadConfig->SerializeToJson());
    }
    roadInfoJson.InsertValue("billing_allowed_setting", BillingAllowedSettingName);
    roadInfoJson.InsertValue("free_rides_number_setting", FreeRidesNumberSettingName);
    roadInfoJson.InsertValue("free_rides_number", FreeRidesNumber);
    roadInfoJson.InsertValue("landing_datetime_format", LandingDateTimeFormat);
    TJsonProcessor::WriteContainerArray(roadInfoJson, "billing_tags", BillingTagNames);
    TJsonProcessor::WriteContainerArray(roadInfoJson, "billing_skipped_tags", BillingSkippedTagNames);
    return roadInfoJson;
}
