#include "road_type.h"

#include <drive/backend/cars/hardware.h>

TExpected<TVector<TTransponderEvent>, TString> TWhsdRoad::GetTransponderEvents(TInstant from, TInstant to, const NDrive::IServer& server, const TSet<TString>& filter) const {
    const TCarAttachmentAssignmentDB& carAttachments = server.GetDriveAPI()->GetCarAttachmentAssignments();
    auto future = ApiClient->GetTransponderEvents(from, to);
    TVector<TTransponderEvent> result;
    while (future.Initialized()) {
        future.Wait();
        if (future.HasException()) {
            return MakeUnexpected<TString>(NThreading::GetExceptionMessage(future));
        } else if (!future.HasValue()) {
            return MakeUnexpected<TString>("future has no value");
        }
        auto results = future.GetValue();
        future = results.GetNextPageRequest();
        for (auto&& event : results.GetTransponderEvents()) {
            TString carId = carAttachments.GetCarByTransponderSpbCode(event.GetTransponderId(), TInstant::Zero());
            if ((!IncludeWhsd && event.GetId().size() > 15) || (!IncludeIntercity && event.GetId().size() <= 15)) {
                TTollRoadLogger::FilteredOutRoadType(event.GetId(), event.GetTransponderId(), IncludeWhsd, IncludeIntercity);
                continue;
            }
            if (carId && !filter.contains(carId)) {
                TTollRoadLogger::FilteredOutCarId(carId);
                continue;
            }
            if (event.GetAmount() >= 0) {
                TTollRoadLogger::PositiveOrZeroMoney();
                continue;
            }
            TTransponderEvent transpEvent(event, carId);
            result.push_back(std::move(transpEvent));
        }
    }
    std::reverse(result.begin(), result.end());
    return result;
}

TString TTransponderEvent::ToString() const {
    return TStringBuilder()
        << "Session: " << SessionId << ", "
        << "amount: " << Amount << ", "
        << "date: " << DateTime.FormatLocalTime("%Y-%m-%d %H:%M:%S") << ", "
        << "from: " << EntryLocation << ", "
        << "to: " << ExitLocation << ", "
        << "car: " << CarId << ", "
        << "transponder: " << TransponderId << ", "
        << "api event: " << EventId;
}
