#pragma once

#include "logger.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/offers/abstract.h>

#include <drive/library/cpp/whsd/client.h>

class TTransponderEvent {
    R_FIELD(TString, CarId);
    R_FIELD(TString, EventId);
    R_FIELD(TString, TransponderId);
    R_FIELD(TString, EntryLocation);
    R_FIELD(TString, ExitLocation);
    R_FIELD(TString, UserId);
    R_FIELD(TString, SessionId);
    R_FIELD(TInstant, DateTime);
    R_FIELD(ui32, Amount, 0);
    R_FIELD(ui32, RoadType, -1);
    R_FIELD(IOffer::TPtr, Offer);

public:
    TTransponderEvent() = default;

    TTransponderEvent(const TString& carId, const TString& transponderId, ui32 amount, TInstant dateTime, TString entry = "", TString exit = "", const TString eventId = "")
        : CarId(carId)
        , EventId(eventId)
        , TransponderId(transponderId)
        , EntryLocation(entry)
        , ExitLocation(exit)
        , DateTime(dateTime)
        , Amount(amount)
    {
    }

    TTransponderEvent(TWhsdTransponderEvent event, const TString& carId = "")
        : CarId(carId)
    {
        TransponderId = event.GetTransponderId();
        EventId = event.GetId();
        EntryLocation = event.GetEntryLocation();
        ExitLocation = event.GetExitLocation();
        DateTime = event.GetDate();
        Amount = std::abs(event.GetAmount());
    }

    TString ToString() const;
};

class ITollRoad {
public:
    virtual ~ITollRoad() = default;
    virtual TExpected<TVector<TTransponderEvent>, TString> GetTransponderEvents(TInstant from, TInstant to, const NDrive::IServer& server, const TSet<TString>& filter) const = 0;
    virtual TString GetType() const = 0;
};

class TWhsdRoad : public ITollRoad {
public:
    TWhsdRoad(const TWhsdClientConfig& config, bool includeWhsd = true, bool includeIntercity = false)
        : IncludeWhsd(includeWhsd)
        , IncludeIntercity(includeIntercity)
    {
        ApiClient = MakeHolder<TWhsdClient>(config);
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "whsd";
    }

    virtual TExpected<TVector<TTransponderEvent>, TString> GetTransponderEvents(TInstant from, TInstant to, const NDrive::IServer& server, const TSet<TString>& filter) const override;

private:
    THolder<IWhsdClient> ApiClient;
    bool IncludeWhsd;
    bool IncludeIntercity;
};
