#include "process.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/device_snapshot/manager.h>

#include <drive/telematics/api/server/client.h>

TExpectedState TTransformationConsistencyProcess::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const {
    Y_UNUSED(state);
    const auto& database = server.GetDriveDatabase();
    const auto& telematics = server.GetTelematicsClient();
    const auto sessionBuilder = database.GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing");
    const auto sessions = sessionBuilder->GetSessionsActual();
    for (auto&& session : sessions) {
        if (!session) {
            continue;
        }
        if (session->GetClosed()) {
            continue;
        }
        if (!context.GetFilteredCarIds().contains(session->GetObjectId())) {
            continue;
        }
        auto billingCompilation = session->GetCompilationAs<TBillingCompilation>();
        if (!billingCompilation) {
            ERROR_LOG << GetRobotId() << ": cannot get BillingCompilation from " << session->GetSessionId() << Endl;
            continue;
        }
        auto lastEvent = session->GetLastEvent();
        auto lastTag = lastEvent ? lastEvent->GetData() : nullptr;
        auto lastTagName = lastTag ? lastTag->GetName() : "unknown";
        if (lastTagName != TChargableTag::Acceptance && lastTagName != TChargableTag::Parking) {
            continue;
        }
        auto snapshot = server.GetSnapshotsManager().GetSnapshot(session->GetObjectId());
        auto speed = snapshot.GetSensor(VEGA_SPEED, MaxSensorAge);
        auto canSpeed = snapshot.GetSensor(CAN_SPEED, MaxSensorAge);
        auto engineOn = snapshot.GetSensor(CAN_ENGINE_IS_ON);

        NJson::TJsonValue ev;
        {
            ev = NJson::TMapBuilder
                ("object_id", session->GetObjectId())
                ("session_id", session->GetSessionId())
                ("accepted", billingCompilation->IsAccepted())
                ("engine_on", NJson::ToJson(engineOn))
                ("can_speed", NJson::ToJson(canSpeed))
                ("speed", NJson::ToJson(speed))
            ;
        }

        if (!speed || !engineOn) {
            NDrive::TEventLog::Log("TransformationConsistencyBadSnapshot", ev);
            continue;
        }
        auto speedValue = speed->ConvertTo<double>(0);
        auto canSpeedValue = canSpeed ? canSpeed->ConvertTo<double>(0) : (1 + SpeedThreshold);
        auto engineOnValue = engineOn->ConvertTo<bool>(false);
        if (speedValue < SpeedThreshold || canSpeedValue < SpeedThreshold || !engineOnValue) {
            continue;
        }
        if (!billingCompilation->IsAccepted()) {
            NDrive::TEventLog::Log("TransformationConsistencyNotAccepted", ev);
            continue;
        }
        auto imei = server.GetDriveAPI()->GetIMEI(session->GetObjectId());
        if (!imei) {
            ERROR_LOG << GetRobotId() << ": no imei for " << session->GetObjectId() << Endl;
            continue;
        }

        auto deadline = Now() + TelematicsTimeout;
        auto getSpeedHandler = telematics.Command(imei, NDrive::NVega::TCommand::GetParameter(*speed));
        auto getCanSpeedHandler = telematics.Command(imei, NDrive::NVega::TCommand::GetParameter(*canSpeed));
        auto getEngineOnHandler = telematics.Command(imei, NDrive::NVega::TCommand::GetParameter(*engineOn));
        auto secondStageError = TString();
        try {
            auto getSpeedResponse = getSpeedHandler.WaitAndEnsureSuccess(deadline).GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
            auto getCanSpeedResponse = getCanSpeedHandler.WaitAndEnsureSuccess(deadline).GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
            auto getEngineOnResponse = getEngineOnHandler.WaitAndEnsureSuccess(deadline).GetResponse<NDrive::TTelematicsClient::TGetParameterResponse>();
            speedValue = Yensured(getSpeedResponse)->Sensor.ConvertTo<double>(0);
            canSpeedValue = Yensured(getCanSpeedResponse)->Sensor.ConvertTo<double>(0);
            engineOnValue = Yensured(getEngineOnResponse)->Sensor.ConvertTo<bool>(false);
        } catch (const std::exception& e) {
            secondStageError = FormatExc(e);
        }
        ev.InsertValue("get_speed_handler", NJson::ToJson(getSpeedHandler));
        ev.InsertValue("get_can_speed_handler", NJson::ToJson(getCanSpeedHandler));
        ev.InsertValue("get_engine_on_handler", NJson::ToJson(getEngineOnHandler));
        if (secondStageError) {
            ev.InsertValue("error", secondStageError);
            NDrive::TEventLog::Log("TransformationConsistencySecondStageError", ev);
            continue;
        }
        if (speedValue < SpeedThreshold || canSpeedValue < SpeedThreshold || !engineOnValue) {
            NDrive::TEventLog::Log("TransformationConsistencySecondStageSkip", ev);
            continue;
        }

        auto objectHash = FnvHash<ui32>(session->GetObjectId()) % 100;
        auto threshold = static_cast<ui32>(RolloutFraction * 100);
        if (threshold <= objectHash) {
            NDrive::TEventLog::Log("TransformationConsistencyDryRun", ev);
        } else {
            auto tx = database.GetTagsManager().GetDeviceTags().BuildTx<NSQL::Writable>();
            tx.SetOriginatorId(GetRobotUserId());
            auto expectedTag = database.GetTagsManager().GetDeviceTags().RestoreTag(lastEvent->GetTagId(), tx);
            if (!expectedTag) {
                ERROR_LOG << GetRobotId() << ": cannot restore tag " << lastEvent->GetTagId() << " for " << session->GetSessionId() << ": " << tx.GetStringReport() << Endl;
                continue;
            }
            auto tag = std::move(*expectedTag);
            auto currentTagName = tag->GetName();
            if (currentTagName != lastTagName) {
                ERROR_LOG << GetRobotId() << ": " << session->GetSessionId() << " state changed from " << lastTagName << " to " << currentTagName << Endl;
                continue;
            }
            auto permissions = Yensured(server.GetDriveAPI())->GetUserPermissions(session->GetUserId(), {});
            if (!permissions) {
                ERROR_LOG << GetRobotId() << ": null permissions for " << session->GetSessionId() << Endl;
                continue;
            }
            if (!TChargableTag::DirectEvolve(tag, TChargableTag::Riding, *permissions, server, tx, /*context=*/nullptr)) {
                ERROR_LOG << GetRobotId() << ": cannot DirectEvolve session " << session->GetSessionId() << ": " << tx.GetStringReport() << Endl;
                continue;
            }
            if (!tx.Commit()) {
                ERROR_LOG << GetRobotId() << ": cannot commit tx for " << session->GetSessionId() << ": " << tx.GetStringReport() << Endl;
                continue;
            }
            NDrive::TEventLog::Log("TransformationConsistencyRestored", ev);
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TTransformationConsistencyProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSDuration>("max_sensor_age", "Max sensor age").SetDefault(MaxSensorAge);
    scheme.Add<TFSNumeric>("rollout_fraction", "Rollout franction").SetMin(0).SetMax(1).SetPrecision(2).SetDefault(RolloutFraction);
    scheme.Add<TFSNumeric>("speed_threshold", "Speed threshold").SetDefault(SpeedThreshold);
    return scheme;
}

bool TTransformationConsistencyProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["max_sensor_age"], MaxSensorAge) &&
        NJson::ParseField(value["rollout_fraction"], RolloutFraction) &&
        NJson::ParseField(value["speed_threshold"], SpeedThreshold);
}

NJson::TJsonValue TTransformationConsistencyProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["max_sensor_age"] = NJson::ToJson(MaxSensorAge);
    result["rollout_fraction"] = RolloutFraction;
    result["speed_threshold"] = SpeedThreshold;
    return result;
}

TTransformationConsistencyProcess::TFactory::TRegistrator<TTransformationConsistencyProcess> TTransformationConsistencyProcess::Registrator(TTransformationConsistencyProcess::GetTypeName());
