#include "config.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>

#include <rtline/util/algorithm/container.h>

#include <util/generic/adaptor.h>
#include <util/string/join.h>

TRTHistoryWatcherState::TFactory::TRegistrator<TRTCarTriggeredState> TRTCarTriggeredState::Registrator(TCarTriggerWathcer::GetTypeName());
TCarTriggerWathcer::TFactory::TRegistrator<TCarTriggerWathcer> TCarTriggerWathcer::Registrator(TCarTriggerWathcer::GetTypeName());

TString TRTCarTriggeredState::GetType() const {
    return TCarTriggerWathcer::GetTypeName();
}

const TSet<TString>& TCarTriggerWathcer::GetFilteredObjectIds(const TTagsModificationContext& context) const {
    return context.GetFilteredCarIds();
}

const IEntityTagsManager& TCarTriggerWathcer::GetEntityTagsManager(const NDrive::IServer& server) const {
    return server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
}


TRTHistoryWatcherState::TFactory::TRegistrator<TRTUserTriggeredState> TRTUserTriggeredState::Registrator(TUserTriggerWathcer::GetTypeName());
TUserTriggerWathcer::TFactory::TRegistrator<TUserTriggerWathcer> TUserTriggerWathcer::Registrator(TUserTriggerWathcer::GetTypeName());

TString TRTUserTriggeredState::GetType() const {
    return TUserTriggerWathcer::GetTypeName();
}

const IEntityTagsManager& TUserTriggerWathcer::GetEntityTagsManager(const NDrive::IServer& server) const {
    return server.GetDriveAPI()->GetTagsManager().GetUserTags();
}

const TSet<TString>& TUserTriggerWathcer::GetFilteredObjectIds(const TTagsModificationContext& /*context*/) const {
    return Default<TSet<TString>>();
}

TExpectedState TRTTriggeredModificationWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const NDrive::IServer& server, TTagsModificationContext& context) const {
    GetModification(context).Clear();
    const TRTHistoryWatcherState* state = dynamic_cast<const TRTHistoryWatcherState*>(stateExt.Get());
    THolder<TRTHistoryWatcherState> result(BuildState());

    const auto& tagManager = GetEntityTagsManager(server);
    ui64 currentEventId = 0;
    {
        auto session = tagManager.BuildSession();
        auto maxLockedEventId = tagManager.GetLockedMaxEventId(session);
        if (!maxLockedEventId) {
            return MakeUnexpected("cannot GetLockedMaxEventId: " + session.GetStringReport());
        }
        currentEventId = *maxLockedEventId;
    }
    ui64 historyIdCursor = state ? state->GetLastEventId() : currentEventId;

    auto session = tagManager.BuildSession(/*readOnly=*/true);
    auto limit = 100000;
    auto events = tagManager.GetEventsSince(historyIdCursor + 1, session, limit);
    if (!events) {
        return MakeUnexpected("cannot GetEventsSince: " + session.GetStringReport());
    }

    TVector<TDBTag> tags;
    TMap<TString, TVector<TDBTag>> tagsByObjects;
    if (!tagManager.RestoreTags(GetFilteredObjectIds(context), MakeVector(NContainer::Keys(TagOperations)), tags, session)) {
        return MakeUnexpected("cannot RestoreTags: " + session.GetStringReport());
    }

    for (auto&& i : tags) {
        tagsByObjects[i.GetObjectId()].emplace_back(i);
    }

    ui64 maxEventId = historyIdCursor;
    TSet<TString> ignoredObjects;
    for (auto&& i : Reversed(*events)) {
        if (i.GetHistoryEventId() > currentEventId) {
            continue;
        }
        maxEventId = std::max(maxEventId, i.GetHistoryEventId());
        const TString& objectId = i.TConstDBTag::GetObjectId();
        const TString& tagName = i->GetName();
        if (TagOperations.contains(tagName)) {
            ignoredObjects.emplace(objectId + "--" + tagName);
        }
        bool triggerUsage =
            (RemovedTags.contains(tagName) && i.GetHistoryAction() == EObjectHistoryAction::Remove) ||
            (NewTags.contains(tagName) && i.GetHistoryAction() == EObjectHistoryAction::Add);
        if (triggerUsage) {
            if (!AreaTagsFilter.Empty() && !AreaTagsFilter.Filter(i->GetObjectSnapshotAs<THistoryDeviceSnapshot>())) {
                continue;
            }
            for (auto&& tagOperation : TagOperations) {
                if (ignoredObjects.contains(objectId + "--" + tagOperation.second.GetTagName())) {
                    continue;
                }
                tagOperation.second.Apply(objectId, tagsByObjects, context, *this);
                ignoredObjects.emplace(objectId + "--" + tagOperation.second.GetTagName());
            }
        }
    }
    const auto doReport = [this, &server, &context]() {
        GetModification(context).NotifyModifications(GetDryRunMode(), GetEmptyReport(), server.GetNotifier(GetNotifierName()));
    };
    if (!GetModification(context).ApplyModification(doReport, GetEntityTagsManager(server), GetDryRunMode(), GetRTProcessName())) {
        return MakeUnexpected<TString>({});
    }
    result->SetLastEventId(maxEventId);
    return result;
}

NDrive::TScheme TRTTriggeredModificationWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("tag_operations", "через запятую [+/-]название тега для использования при срабатывании условий");
    scheme.Add<TFSString>("removed_tags", "срабатывать на удалении tag1, tag2, tag3, ...");
    scheme.Add<TFSString>("new_tags", "срабатывать на добавлении tag1, tag2, tag3, ...");

    const NDrive::IServer* frServer = server.GetAsPtrSafe<NDrive::IServer>();
    if (frServer) {
        TAreaTagsFilter::AddScheme(scheme, *frServer);
    }

    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("dry_run_mode", "только нотификации").SetDefault(false);
    scheme.Add<TFSBoolean>("empty_report", "Нотификация о холостых прогонах").SetDefault(false);
    return scheme;
}

bool TRTTriggeredModificationWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_STRING_OPT(jsonInfo, "notifier", NotifierName);
    JREAD_BOOL_OPT(jsonInfo, "dry_run_mode", DryRunMode);

    if (!AreaTagsFilter.DeserializeFromJson(jsonInfo)) {
        return false;
    }

    TString tagOperations;
    JREAD_STRING(jsonInfo, "tag_operations", tagOperations);
    AssertCorrectConfig(!!tagOperations, "incorrect TagName field in config");
    TSet<TString> tagOperationsSet;
    StringSplitter(tagOperations).SplitBySet(", ").SkipEmpty().Collect(&tagOperationsSet);
    for (auto&& i : tagOperationsSet) {
        TTagOperation to(i);
        TagOperations.emplace(to.GetTagName(), std::move(to));
    }

    {
        TString tagNames;
        JREAD_STRING_OPT(jsonInfo, "removed_tags", tagNames);
        StringSplitter(tagNames).SplitBySet(", ").SkipEmpty().Collect(&RemovedTags);
    }

    {
        TString tagNames;
        JREAD_STRING_OPT(jsonInfo, "new_tags", tagNames);
        StringSplitter(tagNames).SplitBySet(", ").SkipEmpty().Collect(&NewTags);
    }
    JREAD_BOOL_OPT(jsonInfo, "empty_report", EmptyReport);
    return true;
}

NJson::TJsonValue TRTTriggeredModificationWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "dry_run_mode", DryRunMode);
    JWRITE(result, "notifier", NotifierName);

    AreaTagsFilter.SerializeToJson(result);

    TSet<TString> tagOperationDescription;
    for (auto&& i : TagOperations) {
        tagOperationDescription.emplace(i.second.ToString());
    }

    JWRITE(result, "tag_operations", JoinSeq(",", tagOperationDescription));
    JWRITE(result, "removed_tags", JoinSeq(",", RemovedTags));
    JWRITE(result, "new_tags", JoinSeq(",", NewTags));
    JWRITE(result, "empty_report", EmptyReport);
    return result;
}

bool TRTTriggeredModificationWatcher::TTagOperation::Apply(const TString& objectId, const TMap<TString, TVector<TDBTag>>& tagsByObject, TTagsModificationContext& context, const TRTTriggeredModificationWatcher& owner) const {
    auto itTags = tagsByObject.find(objectId);

    bool successRemove = true;
    if (itTags != tagsByObject.end()) {
        for (auto&& i : itTags->second) {
            if (i->GetName() == TagName) {
                if (RemoveTag) {
                    successRemove &= owner.GetModification(context).RemoveTag(objectId, i, true);
                } else {
                    return false;
                }
            }
        }
    }
    if (RemoveTag && successRemove) {
        return true;
    }
    auto tag = context.GetServer().GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(GetTagName(), owner.GetRobotUserId());
    if (!tag) {
        return false;
    }
    return owner.GetModification(context).AddTag(objectId, tag, true);
}
