#include "processor.h"

#include <drive/backend/billing/manager.h>

TUserFinalDeletionRobot::TFactory::TRegistrator<TUserFinalDeletionRobot> TUserFinalDeletionRobot::Registrator("user_final_deletion");

TExpectedState TUserFinalDeletionRobot::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    auto launchActuality = Now();

    if (!server->GetDriveAPI()->HasBillingManager()) {
        ERROR_LOG << "Billing manager is undefined" << Endl;
        return nullptr;
    }
    const auto& userTagManager = server->GetDriveAPI()->GetTagsManager().GetUserTags();
    TMap<TString, TString> userToTagId;
    {
        auto session = userTagManager.BuildSession(/* readonly = */ true);
        TVector<TDBTag> dbTags;
        if (!userTagManager.RestoreTags({}, {TagForDeletion}, dbTags, session)) {
            return MakeUnexpected("cannot RestoreTags: " + session.GetStringReport());
        }
        for (auto&& tag : dbTags) {
            userToTagId.emplace(tag.GetObjectId(), tag.GetTagId());
        }
    }

    TOptionalTagHistoryEvents optionalEvents;
    {
        auto queryOptions = TTagEventsManager::TQueryOptions();
        queryOptions.SetTags({ TagForDeletion });
        auto session = userTagManager.BuildSession(/* readonly = */ true);
        optionalEvents = userTagManager.GetEvents(0, session, queryOptions);
        if (!optionalEvents) {
            return MakeUnexpected("cannot GetEventsAll: " + session.GetStringReport());
        }
    }

    TMap<TString, TInstant> latestTagAddition;
    for (auto&& event : *optionalEvents) {
        const TConstDBTag& tag = event;
        auto userId = tag.GetObjectId();
        auto addedAt = event.GetHistoryInstant();

        if (event.GetHistoryAction() != EObjectHistoryAction::Add || tag->GetName() != TagForDeletion || !userToTagId.contains(userId)) {
            continue;
        }

        auto it = latestTagAddition.find(userId);
        if (it != latestTagAddition.end()) {
            it->second = Max(it->second, addedAt);
        } else {
            latestTagAddition.emplace(userId, addedAt);
        }
    }

    auto usersDB = server->GetDriveAPI()->GetUsersData();
    for (auto&& userIt : latestTagAddition) {
        auto& userId = userIt.first;
        auto timePassed = launchActuality - userIt.second;
        if (timePassed >= WaitingPeriod) {
            auto session = userTagManager.BuildSession();
            {
                auto billingTasks = server->GetDriveAPI()->GetBillingManager().GetActiveTasksManager().GetUsersTasks(NContainer::Scalar(userId), session);
                if (!billingTasks) {
                    NDrive::TEventLog::Log("UserFinalDeletion", NJson::TMapBuilder
                        ("user_id", userId)
                        ("error", "Get billing tasks error: " + session.GetStringReport())
                    );
                    NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "При удалении пользователя " + userId + " не удалось проверить текущие списания");
                    continue;
                }
                if (!billingTasks->empty()) {
                    NDrive::TEventLog::Log("UserFinalDeletion", NJson::TMapBuilder
                        ("user_id", userId)
                        ("skip_reason", "Has billing tasks")
                        ("billing_tasks_size", billingTasks->size())
                    );
                    NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "При удалении пользователя " + userId + " найдены текущие списания");
                    continue;
                }
            }
            auto isAlreadyDeleted = usersDB->IsDeletedByTags(userId, session);
            if (!isAlreadyDeleted) {
                NDrive::TEventLog::Log("UserFinalDeletion", NJson::TMapBuilder
                    ("user_id", userId)
                    ("error", "Get user tags error: " + session.GetStringReport())
                );
                NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "При удалении пользователя " + userId + " не удалось проверить текущие теги");
                continue;
            }
            if (!userTagManager.RemoveTagsSimple({userToTagId[userId]}, GetRobotUserId(), session, true) && NotifierName) {
                NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "При удалении пользователя " + userId + "не удалось снять тег");
                continue;
            }
            if (!*isAlreadyDeleted) {
                if ((!usersDB->DeleteUser(userId, GetRobotUserId(), server, session) || !session.Commit()) && NotifierName) {
                    NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "При удалении пользователя " + userId + "не удалось закоммитить сессию с переводом статуса и удалением тега");
                }
            } else {
                if (!session.Commit() && NotifierName) {
                    NDrive::INotifier::Notify(server->GetNotifier(NotifierName), "При удалении пользователя " + userId + "не удалось закоммитить сессию с только удалением тега");
                }
            }
        }
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TUserFinalDeletionRobot::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("tag_for_deletion", "Имя тега для удаления пользователей").SetRequired(true);
    scheme.Add<TFSDuration>("waiting_period", "Сколько ждать перед тем, как удалить").SetRequired(true);
    scheme.Add<TFSVariants>("notifier_name", "Способ нотификации").SetVariants(server.GetNotifierNames());

    return scheme;
}

NJson::TJsonValue TUserFinalDeletionRobot::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "tag_for_deletion", TagForDeletion);
    TJsonProcessor::WriteDurationString(result, "waiting_period", WaitingPeriod);
    if (NotifierName) {
        TJsonProcessor::Write(result, "notifier_name", NotifierName);
    }

    return result;
}

bool TUserFinalDeletionRobot::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "tag_for_deletion", TagForDeletion);
    JREAD_DURATION(jsonInfo, "waiting_period", WaitingPeriod);
    JREAD_STRING_OPT(jsonInfo, "notifier_name", NotifierName);

    return true;
}
