#include "recheck_tags_handler.h"

TRecheckTagsHandler::TFactory::TRegistrator<TRecheckTagsHandler> TRecheckTagsHandler::Registrator("user_recheck_tags_handler");

TExpectedState TRecheckTagsHandler::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    TSet<TString> usersForReentranceAttempt;
    {
        TVector<TDBTag> dbTags;
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, {ReentranceTagName}, dbTags, session)) {
            return new IRTBackgroundProcessState();
        }
        for (auto&& tag : dbTags) {
            usersForReentranceAttempt.insert(tag.GetObjectId());
        }
    }

    if (usersForReentranceAttempt.size() > 0) {
        auto actuality = Now();
        const auto& registrationManager = server->GetUserRegistrationManager();
        if (ReentrancePolicy == ERecheckPolicy::Full) {
            if (!server->GetDriveAPI()->HasDocumentPhotosManager()) {
                return MakeUnexpected<TString>("user photo manager undefined");
            }
            auto reentranceAssignments = server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().GetLastAssignmentsForUsers(usersForReentranceAttempt);
            for (auto&& assignment : reentranceAssignments) {
                registrationManager->HandleVerifiedAssignment(assignment, GetRobotUserId(), actuality);
            }
        } else if (ReentrancePolicy == ERecheckPolicy::OnlyBackgroundCheck) {
            auto usersFR = server->GetDriveAPI()->GetUsersData()->FetchInfo(usersForReentranceAttempt);
            for (auto&& userIt : usersFR) {
                TDriveUserData mutableUser = std::move(userIt.second);
                registrationManager->DoBackgroundCheckEntrance(mutableUser, GetRobotUserId());
            }
        } else {
            Y_UNREACHABLE();
        }
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRecheckTagsHandler::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("reentrance_tag_name", "Имя тега для повторной попытки впуска в сервис").SetRequired(true);
    scheme.Add<TFSVariants>("reentrance_policy", "Политика впуска").InitVariants<ERecheckPolicy>();

    return scheme;
}

NJson::TJsonValue TRecheckTagsHandler::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "reentrance_tag_name", ReentranceTagName);
    TJsonProcessor::Write(result, "reentrance_policy", ToString(ReentrancePolicy));

    return result;
}

bool TRecheckTagsHandler::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "reentrance_tag_name", ReentranceTagName);

    TString reentrancePolicyStr = ToString(ERecheckPolicy::Full);
    JREAD_STRING_OPT(jsonInfo, "reentrance_policy", reentrancePolicyStr);
    if (!TryFromString(reentrancePolicyStr, ReentrancePolicy)) {
        return false;
    }

    return true;
}
