#include "yang_history_aggregator.h"

#include <drive/library/cpp/yt/node/cast.h>

TYangHistoryAggregator::TFactory::TRegistrator<TYangHistoryAggregator> TYangHistoryAggregator::Registrator(TYangHistoryAggregator::GetTypeName());

TExpectedState TYangHistoryAggregator::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();

    if (!server->GetDriveAPI()->HasDocumentPhotosManager()) {
        MaybeNotify(server, "Отсутствует DocumentPhotosManager");
        return nullptr;
    }

    NYT::TNode::TListType list;
    try {
        list = YtClient->List(ResultsPath);
    } catch (const yexception& e) {
        MaybeNotify(server, "Не удалось получить список таблиц в директории: " + ResultsPath);
        return MakeUnexpected<TString>("Не удалось получить список таблиц в директории: " + ResultsPath + " " + e.what());
    }

    TVector<TString> tablesForDeletion;
    TMap<TString, TString> historyBySecret;
    for (auto&& i : list) {
        TVector<std::pair<TString, TString>> newHistoricalEntries;
        if (!i.IsMap()) {
            TString path = ResultsPath;
            if (!path.EndsWith("/")) {
                path += "/";
            }
            path += i.AsString();
            try {
                auto reader = YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
                for (; reader->IsValid(); reader->Next()) {
                    const NYT::TNode& row = reader->GetRow();

                    TString secretId;
                    if (row.HasKey("secretId") && row["secretId"].IsString()) {
                        secretId = row["secretId"].AsString();
                    } else {
                        continue;
                    }

                    TString history;
                    if (row.HasKey("history")) {
                        NJson::TJsonValue historyJson = NYT::FromNode<NJson::TJsonValue>(row["history"]);
                        history = historyJson.GetStringRobust();
                    } else {
                        continue;
                    }

                    newHistoricalEntries.emplace_back(std::make_pair(std::move(secretId), std::move(history)));
                }
                tablesForDeletion.emplace_back(path);
            } catch (...) {
                MaybeNotify(server, "Не удалось подклеить данные о проверке из таблицы: " + i.AsString());
                continue;
            }

            for (auto&& newHistoricalEntry : newHistoricalEntries) {
                historyBySecret.emplace(std::move(newHistoricalEntry.first), std::move(newHistoricalEntry.second));
            }
            if (historyBySecret.size() >= 100) {
                break;
            }
        }
    }

    TVector<TString> secretIds;
    for (auto&& it : historyBySecret) {
        secretIds.emplace_back(it.first);
    }

    auto assignmentFetchResult = server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().FetchInfo(secretIds);
    for (auto&& assignment : assignmentFetchResult) {
        auto assignmentData = assignment.second;
        auto newHistory = historyBySecret[assignmentData.GetId()];
        if (assignmentData.GetHistory() != newHistory) {
            assignmentData.SetHistory(newHistory);
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetDocumentPhotosManager().GetDocumentVerificationAssignments().Upsert(assignmentData, session) || !session.Commit()) {
                MaybeNotify(server, "Не удалось подклеить данные о проверке к заданию: " + assignmentData.GetId());
            }
        }
    }

    NYT::TRemoveOptions options;
    for (auto&& path : tablesForDeletion) {
        try {
            YtClient->Remove(NYT::TYPath(path), options);
        } catch (...) {
            MaybeNotify(server, "Не удалось удалить отработанные данные о проверке: " + path);
        }
    }

    return new IRTBackgroundProcessState();
}

void TYangHistoryAggregator::MaybeNotify(const NDrive::IServer* server, const TString& message) const {
    DEBUG_LOG << "Notification: " << message << Endl;
    if (NotifierName && server->GetNotifier(NotifierName)) {
        NDrive::INotifier::Notify(server->GetNotifier(NotifierName), message);
    }
}

NDrive::TScheme TYangHistoryAggregator::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("notifier_name", "Способ нотификации о проблемах").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSString>("yt_cluster", "Кластер YT").SetRequired(true);
    scheme.Add<TFSString>("results_path", "Директория с результатам").SetRequired(true);

    return scheme;
}

NJson::TJsonValue TYangHistoryAggregator::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "notifier_name", NotifierName);
    TJsonProcessor::Write(result, "yt_cluster", YtCluster);
    TJsonProcessor::Write(result, "results_path", ResultsPath);

    return result;
}

bool TYangHistoryAggregator::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "notifier_name", NotifierName);
    JREAD_STRING(jsonInfo, "yt_cluster", YtCluster);
    JREAD_STRING(jsonInfo, "results_path", ResultsPath);

    YtClient = NYT::CreateClient(YtCluster, NYT::TCreateClientOptions());

    return true;
}
